/*
 * TraklaEditor with Java
 * Copyright (C) 1996 Ari archie Korhonen
 *
 * @author: Ari Korhonen <Ari.Korhonen@hut.fi>
 * @version: 2.0b
 */

package tred;

import java.applet.Applet;
import java.util.*;
import java.awt.*; 
import java.io.*;
import java.net.*;

/** 
  This class provides the initialization of the TraklaEditor. 
  It parses the parameters provided by the HTML-document and creates
  the GraphPanel. It also sets up buttons needed for editor. 
  */
public final class TraklaEditor extends Applet {
  private String postURL;
  private String courseID;
  private String userID;
  private String roundID;
  private String exerciseID;
  private Solution solution;
  public GraphPanel panel;
  public Panel buttons, play_buttons;
  public Button ok_b, begin_b, end_b, back_b, forw_b; 
  public Choice size, font;
  public String realName;
  public Dialog options;

  /** Returns the URL to cgi-script which is used to deliver export. */
  public String getPostURL() { return postURL; }
  /** Returns the current course id. */
  public String getCourseID() { return courseID; }
  /** Returns the current user id. */
  public String getUserID() { return userID; }
  /** Returns the current round of exercises. */
  public String getRoundID() { return roundID; }
  /** Returns the current exercise number. */
  public String getExerciseId() { return exerciseID; }

  /*
   * returns substring between fr_chr and to_chrs (endpoints not included).
   * null = begin/end.
   */
  private static String parseStr(String str, String fr_chr, String to_chr) {
    if (str == null)
      return null;
    int f = (fr_chr == null) ? 0 : str.indexOf(fr_chr);
    int t;
    if ((to_chr == null) || ((t = str.indexOf(to_chr)) == -1))
      t = str.length();
    if ((f == -1) || (f >= t))
      return null;
    return str.substring(f == 0 ? 0 : f + 1, t);
  }

  /*
   * Squeeze out all occurrences of a character from a string */
  private static String squeezeOut(String from, char toss) {
    if (from == null)
      return null;
    char[] chars = from.toCharArray();
    int len = chars.length;
    for (int i = 0; i < len; i++) {
      if (chars[i] == toss) {
	--len;
	System.arraycopy(chars, i + 1, chars, i , len - i);
	--i; // reexamine this spot
      }
    }
    return new String(chars, 0, len);
  }

  /*
   * Parse macro */
  private static String macro(String str) {
    if ((str == null) || (str.length() < 3))
      return str;
    // %E() == expression elements
    if (str.substring(0,3).equals("%E(")) {
      return "+-*/^!l";
    }
    // %S(n) == n spaces
    // %N(n) == n numbers (0123456789012...)
    if ((str.substring(0,3).equals("%S(")) ||
	(str.substring(0,3).equals("%N("))) {
      int i = str.indexOf(")");
      if (i>0) {
	String dummy = "        ";
	if (str.substring(0,3).equals("%N("))
	  dummy = "0123456789";
	try {
	  int n = Integer.parseInt(str.substring(3, i));
	  while (n > dummy.length())
	    dummy = dummy + dummy;
	  return dummy.substring(0, n);
	}
	catch (NumberFormatException e) {
	  System.err.println("Warning: NumberFormatException: " + e.getMessage());
	}
      }
    }
    // %U(n) == unique letters (ABCD...)
    if ((str.substring(0,3).equals("%U("))) {
      int i = str.indexOf(")");
      if (i>0) {
	try {
	  int n = Integer.parseInt(str.substring(3, i));
	  byte ascii[] = new byte[256];
	  for (i = 0; i<256; i++)
	    ascii[i] = (byte)i;
	  return new String(ascii, 0, (int)'A', n);
	}
	catch (NumberFormatException e) {
	  System.err.println("Warning: NumberFormatException: " + e.getMessage());
	}
      }
    }

    /* Undefined macros are ignored */
    return str;
  }

  /** 
    Parses the parameters and initializes the GraphPanel and all buttons
    needed for editing. 
    */
  public void init() {
    setLayout(new BorderLayout());
    
    buttons = new Panel();
    add("North", buttons);
    play_buttons = new Panel();
    add("South", play_buttons);

    /* new browsers seems to hang in reset so this is sisabled
    buttons.add(new Button("Reset"));
    */

    Button export = new Button("Export");
    buttons.add(export);

    // definition of choice size
    size = new Choice();
    size.addItem("Size: 7");
    size.addItem("Size: 8");
    size.addItem("Size: 9");
    size.addItem("Size: 10");
    size.addItem("Size: 11");
    size.addItem("Size: 12");
    size.addItem("Size: 13");
    size.select(3);
    buttons.add(size);

    font = new Choice();
    String list[] = size.getToolkit().getFontList();
    for (int i=0; i<list.length - 1; i++) {
      font.addItem("Font: " + list[i]);
      if (list[i] == size.getFont().getName())
	font.select(i);
    }
    font.enable();
    buttons.add(font);


    play_buttons.add(begin_b = new Button("begin"));
    begin_b.disable();
    play_buttons.add(end_b = new Button("end"));
    end_b.disable();
    play_buttons.add(back_b = new Button("backward")); 
    back_b.disable();
    play_buttons.add(forw_b = new Button("forward"));
    forw_b.disable();

    panel = new GraphPanel(this);
    add("Center", panel);

    // parsing parameters

    // Why this is needed!? Why static field is not initialized to 0/null
    // after reload?
    EditorObject.animImpl = null;

    try {
      String obj;
      int s = 20;

      obj = getParameter("itemsize");
      if (obj != null)
	s = Integer.parseInt(obj);
      if ((s>=14) && (s<=26)) 
	size.select(s/2 - 7);
      panel.resize(s);
      
      URL docBase = getDocumentBase();
      obj = docBase.getFile();
      for (StringTokenizer t = new StringTokenizer(obj, "/"); 
	   t.hasMoreTokens() ; ) {
	courseID = userID;
	userID = roundID;
	roundID = t.nextToken();
      }
      exerciseID = parseStr(parseStr(roundID, ".", null), null, ".");
      roundID = parseStr(roundID, null, ".");

      /* 
       * This param is kludge: it's only used to deliver the testround flag 
       */
      obj = getParameter("realname");
      panel.setTestRound(false);
      if (obj != null) {
	realName = obj;
	if (realName.regionMatches(true, 0, "demo", 0, 4))
	  panel.setTestRound(true);
      }
	
      obj = getParameter("userid");
      if (obj != null)
	userID = obj;
      obj = getParameter("courseid");
      if (obj != null)
	courseID = obj;
      obj = getParameter("exercise");
      if (obj != null) {
	roundID = parseStr(obj, null, ".");
	exerciseID = parseStr(obj, ".", null);
      }
      if ((roundID == null) || (exerciseID == null))
	panel.add("stream", "Invalid exercise ID: ", 
		  roundID + "." + exerciseID,0); 
      
      if ((postURL = getParameter("posturl")) == null)
	export.disable();

      obj = getParameter("objects");
      if (obj != null)
	for (StringTokenizer t = new StringTokenizer(obj, ",") ; 
	     t.hasMoreTokens() ; ) {
	  String str = t.nextToken();
	  if (str != null) {
	    str = str.trim();
	    String label = parseStr(parseStr(str, " ", ";"), null, "+");
	    String items = macro(squeezeOut(parseStr(str, ";", "+"), (char)' '));
	    int space = 0;
	    int height = 0;
	    try {
	      String spaces = parseStr(str, "+", null);
	      String spacex = parseStr(spaces, null, "+");
	      String spacey = parseStr(spaces, "+", null);
	      space = spacex == null ? 0 : Integer.parseInt(spacex, 10);
	      height = spacey == null ? 0 : Integer.parseInt(spacey, 10);
	    }
	    catch (NumberFormatException e) {
	      space = 0;
	      height = 0;
	    }
	    panel.add(parseStr(parseStr(parseStr(str, null, " "), 
					null, ";"), 
			       null, "+"),
		      label, items, space, height);
	  }
	}
      else
	panel.add("stream", "Missing parameter: ", "objects", 0);
      
      // for method export() we have to know which objects to return
      obj = getParameter("output");
      if (obj != null)
	for (StringTokenizer t = new StringTokenizer(obj, ",") ; 
	     t.hasMoreTokens() ; ) {
	  String str = t.nextToken();
	  panel.setOutput(parseStr(parseStr(str, null, ";"), null, "+"), 
			  parseStr(str, ";", "+"),
			  parseStr(str, "+", null));
	}
      else export.disable();

      // Encoding the output might be usefull in some cases
      // The encode line should be in form s11/s12,s21/s22,...,sn1/sn2
      // where sk1 is substituted with sk2
      obj = getParameter("encode");
      if (obj != null)
	for (StringTokenizer t = new StringTokenizer(obj, ",") ; 
	     t.hasMoreTokens() ; ) {
	  String str = t.nextToken();
	  StringTokenizer tt = new StringTokenizer(str, "/");
	  String s1 = tt.nextToken();
	  String s2 = tt.nextToken();
	  panel.encode(s1, s2);
	}

      // append item to the given panel
      obj = getParameter("append");
      if (obj != null)
	for (StringTokenizer t = new StringTokenizer(obj, ",") ; 
	     t.hasMoreTokens() ; ) {
	  String str = t.nextToken();
	  panel.append(parseStr(str,null,";"), parseStr(str,";",null));
	}

      // set order of the output (for trees)
      obj = getParameter("order");
      if (obj != null)
        for (StringTokenizer t = new StringTokenizer(obj, ",") ; 
             t.hasMoreTokens() ; ) {
          String str = t.nextToken();
          panel.setOrder(parseStr(str,null,";"), parseStr(str,";",null));
        }

      // mark listarray type objects
      obj = getParameter("listarray");
      if (obj != null)
	for (StringTokenizer t = new StringTokenizer(obj, ",") ; 
	     t.hasMoreTokens() ; ) {
	  String str = t.nextToken();
	  panel.setListArray(parseStr(str,null,";"), parseStr(str,";",null));
	}

      // mark infinite type objects
      obj = getParameter("infinite");
      if (obj != null)
	for (StringTokenizer t = new StringTokenizer(obj, ",") ; 
	     t.hasMoreTokens() ; ) {
	  String str = t.nextToken();
	  panel.setInfinite(parseStr(str,null,";"), parseStr(str,";",null));
	}

      // mark shadowed panels
      obj = getParameter("shadowed");
      if (obj != null)
	for (StringTokenizer t = new StringTokenizer(obj, ",") ; 
	     t.hasMoreTokens() ; ) {
	  String str = t.nextToken();
	  panel.shadowed(str);
	}
      
      // mark not movable items
      obj = getParameter("disable");
      if (obj != null) 
	for (StringTokenizer t = new StringTokenizer(obj, ",") ;
	     t.hasMoreTokens() ; ) {
	  String str = t.nextToken();
	  panel.disableItem(str);
	}

      obj = getParameter("solution");
      if (obj != null) {
	Button solve = new Button("Solve");
	buttons.add(solve);
	solution = new Solution(obj);
      }

      // show slot names
      obj = getParameter("show");
      if (obj != null) 
	for (StringTokenizer t = new StringTokenizer(obj, ",") ;
	     t.hasMoreTokens() ; ) {
	  String str = t.nextToken();
	  panel.showSlot(str);
	}

      // rename object str1[k] with string Sk, k=1..n
      // usage: rename <str1>/<dstr><str2>
      // where chars.length() = n, <str2> = S1<dstr>S2<dstr>...<dstr>Sn
      // separator = <dstr>
      obj = getParameter("rename");
      if (obj != null) {
	StringTokenizer t = new StringTokenizer(obj, "/");
	String str1 = t.nextToken();
	String str2;
	int p = 0, q = 0;

	if ((str1 != null) && 
	    (t.hasMoreTokens()) && 
	    (str2 = t.nextToken()) != null) {
	  String dstr = str2.substring(0,1);
	  for (int i=0; (i<str1.length()) && (p >= 0); i++) {
	    while ((q = str2.indexOf(dstr, p++)) == p - 1);
	    panel.rename(str1.substring(i, i + 1), 
			 str2.substring(p - 1, q == -1 ? str2.length() : q));
	    p = q;
	  }
	}
	else System.err.println("Warning: don't know what to replace: " + obj);
      }
      
    } catch (NoSuchElementException e) {
      System.err.println("NoSuchElementException: " + e.getMessage());
    }
    validate();
  }

  private static ExportThread exp;
  /** 
   * This method is called when an action occurs inside this component. 
   * @param arg contains the arg field of the event e argument.
   * @param e event e
   * @return true to indicate that it has successfully handled 
   * the action or false if the event that triggered the action should be 
   * passed up to the component's parent. 
   */
  public synchronized boolean action(Event e, Object arg) {
    /* System.out.println("action " + e); */
    if (e.target instanceof Button) {
      if ("Ok".equals(arg)) { 
	exp.deliverEvent(e); 
      }
      else if ("Reset".equals(arg)) { notify(); init(); }
      else if ("Export".equals(arg)) { 
	exp = new ExportThread(this); 
	exp.start();
      }
      else if ("Solve".equals(arg)) { solution.solve(); }
      else if ("backward".equals(arg)) { EditorObject.moveBackward(); }
      else if ("forward".equals(arg)) { EditorObject.moveForward(); }
      else if ("begin".equals(arg)) { EditorObject.moveBegin(); }
      else if ("end".equals(arg)) { EditorObject.moveEnd(); }
      panel.repaint();
      return true;
    }
    else if (e.target instanceof Choice) {
      if (arg != null) {
	if (arg.toString().startsWith("Size")) {
	  panel.resize(size.getSelectedIndex()*2 + 14);
	  /* panel.repaint(); */
	  validate();
	  /* panel.validate(); */
	  return true;
	}
	if (arg.toString().startsWith("Font: ")) {
	  panel.setFont(font.getSelectedItem().substring(6));
	  /* panel.repaint(); */
	  validate();
	  /* panel.validate(); */
	  return true;
	}
      }
    }
    return false;
  }

  /*
  public void deliverEvent(Event evt) {
    super.deliverEvent(evt);
  }
  public boolean postEvent(Event evt) {
    return super.postEvent(evt);
  }
  public boolean handleEvent(Event evt) {
    return super.handleEvent(evt);
  }
  */

  /** Returns the version and author of this applet. */
  public String getAppletInfo() {
    return "TraklaEditor version 2.1a by Ari Korhonen <Ari.Korhonen@hut.fi> <http://www.hut.fi/~archie>";
  }
}
  
class ExportThread extends Thread {
  TraklaEditor applet;

  ExportThread(TraklaEditor applet) {
    this.applet = applet;
  }

  public void run() {
    export();
  }

  // Just a little parser routine to clean up the output the scgi-script
  // sends back while exporting...
  private String parseHTML(String str) {
    StringBuffer sb = new StringBuffer();
    boolean body = false;
    boolean br = false;
    for (int i=0; i<str.length(); ) {
      String s;
      // next character is "less than"
      if (str.substring(i, i+1).equals("<")) {
	int p = str.indexOf(">", i);
	if (p == -1)
	  p = str.length();
	s = str.substring(i, p + 1).trim(); 
	i = p + 1;
      }
      else { // some plain text follows
	int p = str.indexOf("<", i);
	if (p == -1) // just plain text follows
	  p = str.length();
	s = str.substring(i, p);
	i = p;
      }
      if (s.toUpperCase().equals("<BODY>"))
	body = true;
      else if (s.toUpperCase().equals("</BODY>"))
	body = false;
      else if ((s.toUpperCase().equals("</P>")) || 
	       (s.toUpperCase().equals("</H1>")) ||
	       (s.toUpperCase().equals("</H2>")) ||
	       (s.toUpperCase().equals("</H3>")) ||
	       (s.toUpperCase().equals("</H4>")) ||
	       (s.toUpperCase().equals("</H5>")))
	sb.append("\n");
      if ((body) && (!s.substring(0, 1).equals("<")))
	sb.append(s);
    }
    return sb.toString();
  }

  public synchronized void export() {
    GraphPanel gp = applet.panel;
    applet.removeAll();
    /* applet.validate(); */
    applet.panel = new GraphPanel(applet);
    applet.add("North", new Button("Ok"));
    applet.add("Center", applet.panel);
    TextArea t = (TextArea)
      (applet.panel).add("textarea", "Export:", "Trying to send. Wait...\n", 10);
    applet.validate();
    String inputLine;
    DataInputStream inStream = null;

    try {
      inStream = applet.panel.export(gp);
      if (inStream == null)
	t.appendText("Cannot get inputStream.\n");
      while ((inputLine = inStream.readLine()) != null)
	t.appendText(parseHTML(inputLine));
      inStream.close();
    } catch (MalformedURLException e) {
      t.appendText("Caught MalformedURLException: " + e.getMessage());
      t.appendText("\nPlease, report to author.\n");
    } catch (IOException e) {
      t.appendText("Caught IOException: " + e.getMessage());
      t.appendText("\nPlease, report to author.\n");
    } /* catch (NullPointerException e) {
      t.appendText("Internal nullPointerException in Class TraklaEdit.");
      t.appendText("Press Ok and try again.");
      t.appendText("\nIf the error don't go away, please, report to author.\n");
    } */ finally {
      try {
	t.validate();
	applet.validate();
	wait();
      } catch (InterruptedException e) {
	System.err.println("Caught InterruptedException " + e.getMessage());
      }
      applet.removeAll();
      applet.add("North", applet.buttons);
      applet.add("South", applet.play_buttons);
      applet.panel = gp;
      applet.add("Center", applet.panel);
      applet.validate();
    }      
  }

  public synchronized boolean deliverEvent(Event e) {
    /* System.out.println("deliverEvent " + e); */
    if (e.target instanceof Button) {
      notify();
      return true;
    }
    return false;
  }
}


