package tred;

import java.awt.*; 
import java.io.*;
import java.util.*;

/** 
 * This is the abstract class Slot for all kind of slots and is intended
 * to be subclassed. 
 */
public abstract class Slot extends EditorObject {
  private Slot myRoot = this;
  private boolean isLink = false;
  protected Item item;
  protected Item shadowItem;
  protected EditorPanel editorPanel;
  protected int id;
  protected static int nextCloneId = 0;
  public ListObject list;
  

  boolean SLOT_DEBUG = false;

  /** This is the generic constructor for all slots. */
  public Slot(EditorPanel panel, int id) {
    editorPanel = panel;
    this.id = id;
    this.lbl = String.valueOf(id);
    item = null;
  }

  /** @return myRoot */
  public Slot getMyRoot() {
    return myRoot;
  }
  public boolean isLinkSlot() {
    return isLink;
  }
  public void setLinkSlot() {
    isLink = true;
    list = null;
  }

  /** 
   * This is the generic paint method for all slots. This method is 
   * intended to be overridden. Every subclass should have it's own
   * paint method which invokes this super paint method as it's
   * first statement.
   * Note. If the slot has a ListObject, those slots in the list are 
   * not drawed! Anyway, because of implementation of stack, shadowed 
   * items of the list must be drawn on the screen.
   * 
   */
  protected boolean paint(Graphics g) {
    if (isVisible() || SLOT_DEBUG) {
      if (shadowItem != null) { // paint shadowItem
	FontMetrics fm = g.getFontMetrics();
	int h = fm.getHeight();
	int w = fm.stringWidth(shadowItem.lbl);
	g.drawString(shadowItem.lbl, x+1+(width-w)/2, y-2+(height+h)/2);
      }
      if ((editorPanel.showableSlots()) || SLOT_DEBUG) { // paint shdw slots
	FontMetrics fm = g.getFontMetrics();
	int h = fm.getHeight();
	int w = fm.stringWidth(lbl);
	g.drawString(lbl, x+1+(width-w)/2, y-itemsize+(height+h)/2);
      }
    }
    return true;
  }

  /** this method is rewritten to method setLocation in the future. */
  protected int paint(Graphics g, int x, int y) {
    width = itemsize;
    height = itemsize;

    if (list != null) {
      Enumeration enum = list.elements();
      int i = 1;
      while (enum.hasMoreElements()) {
	Slot slot = (Slot)enum.nextElement();
	if ((slot.item != null) || SLOT_DEBUG) {
	  if (editorPanel.isTreeOrientated()) {
	    slot.y = y;
	    slot.x = x + 2 * i++ * itemsize/2;
	  }
	  else {
	    slot.x = x;
	    slot.y = y + 3 * i++ * itemsize/2;
	  }
	}
	// slot.hide(hidden);
      }
    }
    this.x = x;
    this.y = y;
    return width;
  }


  protected int setLocation(int x, int y) {
    width = itemsize;
    height = itemsize;

    if (list != null) {
      Enumeration enum = list.elements();
      int i = 1;
      while (enum.hasMoreElements()) {
	Slot slot = (Slot)enum.nextElement();
	slot.getBounds = true;
	if ((slot.item != null) || SLOT_DEBUG) {
	  if (editorPanel.isTreeOrientated()) {
	    slot.y = y;
	    slot.x = x + 2 * i++ * itemsize/2;
	  }
	  else {
	    slot.x = x;
	    slot.y = y + 3 * i++ * itemsize/2;
	  }
	}
	// slot.hide(hidden);
      }
    }
    this.x = x;
    this.y = y;
    return width;
  }

  /** 
   * This method is invoked when new item is put to list of this slot. 
   * @param item which is put to list of this slot.
   * @return null if the argument item is null; new clone of this slot
   * otherwise.
   * this slot is returned.
   */
  protected Slot putList(Item item) {
    if (item == null) 
      this.item = null;
    else {
      Slot newSlot = (Slot)this.clone();
      if (editorPanel.getOrder())
	list.add(newSlot);
      else
	list.append(newSlot);
      newSlot.item = item;
      newSlot.myRoot = this;
      return newSlot;
    }
    return null;
  }

  /** 
   * This method is invoked when an item is going to move into this slot.
   * If generic movements are not allowed a subclass should override this
   * method and return null in case a movement is invalid.
   * Method returns rslot which is the slot where the item really is
   * attached to. Usually rslot is this slot. Anyway, if the panel that
   * this slot belongs to is a listArray, the rslot is the (cloned) slot 
   * where the item is moved to.
   * @param item which moves into this slot.
   * @return rslot if item == null or this slot is empty; null if this slot 
   * is occupied. 
   */
  protected Slot putItem(Item item) {
    editorPanel.touch();

    if ((list != null) && (this == this.getMyRoot()))
      return putList(item);
    else {
      if (item == null)
	this.item = null;
      else {
	if ((this.item == null) && 
	    ((shadowItem == null) || (shadowItem == item))) {
	  this.item = item;
	  //if (editorPanel.isShadowed())
	  //  shadowItem = item;
	}
	else 
	  /* one slot can hold only one item and shadowslot can't hold
	     any other item than original item. */
	  return null;
      }
    }
    return this;
  }

  /* */
  protected Slot insertItem(Item item, EditorObject after) {
    /* invoked("insertItem " + item + " after " + after); */
    editorPanel.touch();
    
    if (list == null)
      return null;
    Slot newSlot = (Slot)this.clone();
    list.insert(newSlot, after);
    newSlot.item = item;
    if (item == null)
      newSlot.setLinkSlot();
    /* newSlot.list = this.list */
    return newSlot;
  }
  /** This method is invoked when a new item is inserted into this tree. 
   * The item is inserted into this slot between some existing items.
   * @param item - item which is inserted
   */
  protected Slot insertItemBefore(Item item, EditorObject before) {
    editorPanel.touch();
    
    /* invoked("insertItemBefore: " + item + " " + before); */
    if (list == null)
      return null;
    Slot newSlot = (Slot)this.clone();
    list.insertBefore(newSlot, before);
    newSlot.item = item;
    if (item == null)
      newSlot.setLinkSlot();
    return newSlot;
  }

  /** @return the item attached to this slot. */
  public Item getItem() { return item; }
  
  /** 
   * This class doesn't implement cloneable but allow subclasses to support
   * clone. This method is implemented to provide a clone implementation
   * that clones all fields correctly if the default implementation isn't
   * correct.
   */
  public Object clone() {
    try {
      Slot slot = (Slot)super.clone();
      slot.id = nextCloneId++;
      slot.lbl = "c" + String.valueOf(slot.id);
      /* invoked("cloned: " + slot.lbl); */
      return slot;
    } catch (CloneNotSupportedException e) {
      throw new InternalError(e.toString());
    }
  }

  /** @return String getClass().toString(). */
  public String toString() {
    return getClass().toString() + " Value:" + lbl + " (" +
      ((item == null) ? (list != null ? list.toString() : "???") : (item.toString())) + ")";
  }
}



