package tred;

import java.util.*;
import java.awt.*; 
import java.io.*;
import java.net.*;

//-------- exceptions

class NoSuchPanelException extends Exception {
  String message;

  NoSuchPanelException(String str) {
    message = str;
  }
}

class NoSuchObjectException extends Exception {
  String message;

  NoSuchObjectException(String str) {
    message = str;
  }
}

/** GraphPanel class to control all the EditorObjects and buttons. */
public final class GraphPanel extends Panel {
  private boolean DEBUG = true;

  private static final int HASHSTREAM = 1;

  private final Color drawColor = Color.black;

  private TraklaEditor editor;
  private Graphics graphics, offGraphics;
  private Image offScreen;
  private Dimension offScreenSize;
  private int size, lastx = 0, lasty = 0;
  private boolean spacePressed = false, testRound = false;
  private EditorObject pick, click;
  private Hashtable encodeTable = new Hashtable(); 


  //-------- public fields

  public Vector objects = new Vector();
  public Button export, begin, end, forward, backward;
  public int nextPosition;

  //-------- constructor 

  /** 
   * This is the constructor for GraphPanel invoked by Applet.
   */
  public GraphPanel(TraklaEditor editor) {
    setLayout(null);
    Insets insets = insets();
    this.editor = editor;
    begin = editor.begin_b;
    end = editor.end_b;
    forward = editor.forw_b;
    backward = editor.back_b;
    nextPosition = 1;
  }


  //-------- private methods

  /** returns the object in place (x,y). Null if there is no object. */
  private EditorObject findEditorObject(int x, int y) {
    boolean firstElement = true;
    Enumeration enum = objects.elements();
    while (enum.hasMoreElements()) {
      EditorObject object = (EditorObject)enum.nextElement();
      if ((object != pick) && (!object.disabled()) && (object.inside(x, y)))
	return object.element(x,y);
    }
    return null;
  }

  private void markMovement() {
    EditorObject.markMovement();
  }

  // return the editorPanel labeled str
  public EditorPanel findPanel(String str) throws NoSuchPanelException {
    Enumeration enum = objects.elements();
    while (enum.hasMoreElements()) {
      EditorObject object = (EditorObject)enum.nextElement();
      if ((str.equals(object.lbl)) && (object instanceof EditorPanel))
	return (EditorPanel)object;
    }
    throw new NoSuchPanelException(str);
  }

  // return the editorPanel labeled str
  public EditorObject findObject(String str) throws NoSuchObjectException {
    Enumeration enum = objects.elements();
    while (enum.hasMoreElements()) {
      EditorObject object = (EditorObject)enum.nextElement();
      if (str.equals(object.lbl))
	return (EditorObject)object;
    }
    throw new NoSuchObjectException(str);
  }

  // protected methods

  /** 
   * This method is invoked when the TraklaEdit wants to determine
   * wheter this exercise is demo exercise or not;
   * @param flag is true if this is a demo exercise.
   */
  protected void setTestRound(boolean flag) {
    testRound = flag;
  }

  /** Removes the object from the list of editorObjects. */
  protected void removeObject(EditorObject object) {
    if (!objects.removeElement(object))
      System.err.println("Warning: GraphPanel removeElement failed.");
  }

  /** Adds an new object to the end of the list of editorObjects. */
  protected void addObject(EditorObject object) {
    objects.addElement(object);
    object.setGraphPanel(this);
  }

  /** Inserts an new object to the beginning of the list of editorObjects. */
  protected void insertObject(EditorObject object) {
    objects.insertElementAt(object, 0);
    object.setGraphPanel(this);
  }

  /** Set listArray properties for argument panel str. */
  protected void setListArray(String str, String items) {
    try {
      findPanel(str).setListArray(items);
    }
    catch (NoSuchPanelException e) {
      System.err.println("Param listarray: " + e.message + " not found.");
    }
  }

  /** Set infinite properties for argument panel str. */
  protected void setInfinite(String str, String items) {
    try {
      /* System.out.println("infinite: " + str + " with " + items); */
      findPanel(str).setInfinite();
    }
    catch (NoSuchPanelException e) {
      System.err.println("Param setInfinite: " + e.message + " not found.");
    }
  }


  protected void append(String panel, String items) {
    try {
      findPanel(panel).append(items);
    }
    catch (NoSuchPanelException e) {
      System.err.println("Param listarray: " + e.message + " not found.");
    }
  }


  /** Set order of the output (for trees) to panel str. */
  protected void setOrder(String str, String order) {
    try {
      findObject(str).setOrder(order);
    }
    catch (NoSuchObjectException e) {
      System.err.println("Param listarray: " + e.message + " not found.");
    }
  }

  /** Implement slots of argument panel str as shadowed. */
  protected void shadowed(String str) {
    Enumeration enum = objects.elements();
    while (enum.hasMoreElements()) {
      EditorObject object = (EditorObject)enum.nextElement();
      if ((str.equals(object.lbl)) && (object instanceof EditorPanel))
	((EditorPanel)object).shadowed();
    }
  }

  /** Disables movement of argument item str.  */
  protected void disableItem(String str) {
    Enumeration enum = objects.elements();
    while (enum.hasMoreElements()) {
      EditorObject object = (EditorObject)enum.nextElement();
      if (str.equals(object.lbl))
	object.disable();
    }
  }

  /** 
   * Marks object to be exportable. Object is searched from the GraphPanel's
   * list of objects and method object.setOutput(dstr, addStr, before) 
   * is invoked. Parameter before is optional and is determined from
   * the input parameter obj. 
   * @see TraklaEditor.output()
   * @param String obj object to be exported.
   * @param String dstr to mark null items.
   * @param String addStr to add before answer.
   * @see EditorObject.setOutput()
   */ 
  protected void setOutput(String obj, String dstr, String addStr) {
    EditorObject before = null;
    EditorObject after = null;
    String beforeStr = null;
    String afterStr = null;
    int i;

    if ((i = obj.indexOf("\\")) != -1) {
      beforeStr = obj.substring(i+1, obj.length());
      obj = obj.substring(0, i);

      // search before/after string

      if (beforeStr == null)
	System.err.println("Warning: Parameter missing (output)");
      else {
	Enumeration enum = objects.elements();
	while ((enum.hasMoreElements()) && (before == null)) {
	  EditorObject object = (EditorObject)enum.nextElement();
	  if (beforeStr.equals(object.lbl))
	    before = object;
	}
	if ((before == null) && (!beforeStr.toLowerCase().equals("end")))
	  System.err.println("Warning: Output Before Object " + 
			   obj + " not found.");
      }
    }
      
    if ((i = obj.indexOf("/")) != -1) {
      afterStr = obj.substring(i+1, obj.length());
      obj = obj.substring(0, i);
      
      if (afterStr == null)
	System.err.println("Warning: Parameter missing (output)");
      else {      
	Enumeration enum = objects.elements();
	while ((enum.hasMoreElements()) && (after == null)) {
	  EditorObject object = (EditorObject)enum.nextElement();
	  if (afterStr.equals(object.lbl))
	    after = object;
	}
	if ((after == null) && (!beforeStr.toLowerCase().equals("end")))
	  System.err.println("Warning: Output After Object " + 
			     obj + " not found.");
      }
    }

    // search object

    Enumeration enum = objects.elements();
    EditorObject found = null;
    while ((enum.hasMoreElements())) {
      EditorObject object = (EditorObject)enum.nextElement();
      if (obj.equals(object.lbl)) {
	if ((before == null) && (after == null) && ("end".equals(beforeStr)))
	  before = object;
	object.setOutput(dstr, addStr, before, after);
	found = object;
      }
    }
    if (found == null)
      System.err.println("Warning: Output Object " + obj + " not found!");
  }

  /** This method is invoked when a new encoding substitution is added to
   * database. The HashTable for the elements is created the first time 
   * this method is invoked.
   * @param s1 is the from-string (char) 
   * @param s2 is the to-string (string) 
   */
  protected void encode(String s1, String s2) {
    if ((s1 !=null) && (s2 != null))
      encodeTable.put(s1, s2);
  }

  /** Resets all movable objects to the initial states. */
  protected void reset() {
    EditorObject.resetBuffers();
    Enumeration enum = objects.elements();
    while (enum.hasMoreElements()) {
      EditorObject object = (EditorObject)enum.nextElement();
      if (object instanceof EditorPanel)
	((EditorPanel)object).reset();
    }
    begin.disable();
    end.disable();
    forward.disable();
    backward.disable();
    repaint();
  }

  /** 
    This method is invoked when user wants to export exercise. An URL
    connection is made to server defined in parameter getPostURL. There should
    be an cgi-script which emails the message to TRAKLA-server. This method
    sets all the headers needed for email. After that all output methods are
    invoked to combine the answer. 
   */
  protected DataInputStream export(GraphPanel gp) 
       throws MalformedURLException, IOException {
    DataInputStream inStream = null;
    try {
      URL url = new URL(editor.getPostURL());
      URLConnection urlc = url.openConnection();
      if (urlc == null)
	throw new IOException("URLConnection failed");
      // use pipedStreams to encode some characters
      PipedOutputStream out = new PipedOutputStream();
      PipedInputStream in = new PipedInputStream(out);
      urlc.setDoOutput(true);

      // get the outputStream to URL (cgi-script)
      MyPrintStream outputPrintStream = new MyPrintStream(urlc.getOutputStream());
      if (outputPrintStream == null)
	throw new IOException("Cannot get outputPrintStream");

      // duplicate: answers header goes without encoding!
      MyPrintStream outStream = outputPrintStream;

      // this output goes to cgi-script
      outStream.println("comments=Message generated by TraklaEdit");
      outStream.print("&CID=" + editor.getCourseID());
      outStream.print("&ID=" + editor.getUserID());
      //      outStream.print("&REALNAME=myrealname");
      //      outStream.print("&REPLY=myreplyaddress");
      outStream.println("&answer=; www exercise");
      if (gp.testRound)
	outStream.println("#TESTROUND " + editor.getRoundID());
      else
	outStream.println("#ROUND " + editor.getRoundID());
      outStream.println("#" + editor.getExerciseId());

      // rest of the output goes with encoding
      outStream = new MyPrintStream(out);
      Enumeration enum = gp.objects.elements();
      while (enum.hasMoreElements()) {
	EditorObject object = (EditorObject)enum.nextElement();
	// feature: only class EditorPanel is exportable
	if (object instanceof EditorPanel)
	  object.export(outStream);
      }
      // close the stream in order to somehow terminate the next while-loop!
      outStream.close();

      // encoding
      int ch;
      while ((ch = in.read()) != -1) {
	String str = (String)gp.encodeTable.get(String.valueOf((char)ch));
	if (str != null)
	  outputPrintStream.print(str);
	else
	  outputPrintStream.print((char)ch);
      }
      outputPrintStream.println();
      outputPrintStream.close();

      inStream = new DataInputStream(urlc.getInputStream());

    } catch (MalformedURLException e) {
      System.err.println("Caught MalformedURLException: " + e.getMessage());
      throw(e);
    } catch (IOException e) {
      System.err.println("Caught IOException: " + e.getMessage());
      throw(e);
    } finally {
      return inStream;
    }
  }

  protected Object add(String obj, String label, String items, 
		       int space) {
    return add(obj, label, items, space, 0);
  }

  /** 
   * This method is invoked when new object is added into the editor by 
   * class TraklaEdit. 
   */
  protected Object add(String obj, String label, String items, 
		       int space, int height) {
    
    if (obj == null) {
      System.err.println("Error: Invalid parameter value (objects)");
      return null;
    }

    if (obj.equals("button")) {
      EditorButton editorButton = new EditorButton(this, label, null, 0);
      addObject(editorButton);
    }
    else if (obj.equals("stream")) {
      StreamPanel streamPanel = null;
      switch (height) {
      case 0:
	streamPanel = new StreamPanel(this, label, items, space);
	break;
      default:
	Dimension dim = new Dimension(space, height);
	streamPanel = new StreamPanel(this, label, items, dim);
      }
      addObject(streamPanel);
    }
    else if (obj.equals("hashstream")) {
      StreamPanel 
	streamPanel = new StreamPanel(this, label, items, space, HASHSTREAM);
      addObject(streamPanel);
    }
    else if (obj.equals("nodepanel")) {
      NodePanel nodePanel = new NodePanel(this, label, items, space);
      addObject(nodePanel);
    }
    /*
      tree MatrixPanel for instance... tree is a parameter
     */
    else if (obj.equals("tree")) {
      Tree tree = new Tree(this, label, items);
      System.out.println("tesdf");
      addObject(tree);
    }
    else if (obj.equals("bplus")) {
      BPlus bplus = new BPlus(this, label, items);
      addObject(bplus);
    }
    else if (obj.equals("nodepair")) {
      NodeListPanel nodeLP = new NodeListPanel(this, label, items, space);
      addObject(nodeLP);
    }
    else if (obj.equals("stack")) {
      tred.Stack stack = new tred.Stack(this, label, items, space);
      addObject(stack);
    }
    else if (obj.equals("textfield")) {
      TextFieldPanel tfPanel = new TextFieldPanel(this, label, items, space);
      addObject(tfPanel);
    }
    else if (obj.equals("textarea")) {
      Object o;
      TextAreaPanel taPanel = new TextAreaPanel(this, label, items, space);
      addObject(taPanel);
      return taPanel.textArea();
    }
    else if (obj.equals("graphpanel")) {
      Graph graph = new Graph(this, label, items, space);
      addObject(graph);
    }
    else if (obj.equals("trash")) {
      Trash trash = new Trash(this, label);
      addObject(trash);
    }
    else if (obj.equals("matrix")) {
      MatrixPanel matrix = new MatrixPanel(this, label, items, space);
      addObject(matrix);
    }
    
    return null;
  }

  /** 
   * This method is invoked by TraklaEditor when an object is renamed.
   * The method looks for an object called str1 and renames it with str2.
   * Only first occurence is renamed.
   */
  protected void rename(String str1, String str2) {
    Enumeration enum = objects.elements();

    while (enum.hasMoreElements()) {
      EditorObject object = (EditorObject)enum.nextElement();
      if (((EditorObject)object).lbl.equals(str1)) {
	((EditorObject)object).lbl = str2;
	break;
      }
    }
  }

  /*
   * This method is invoked in order to display the names of the slots
   * of the given panel. 
   * The name of slot is given when the panel is initialized. The name is a
   * number 0,1,...,n. 
   * @param str is the name of panel which slots are wanted to have names.
   */
  protected void showSlot(String str) {
    Enumeration enum = objects.elements();

    while (enum.hasMoreElements()) {
      EditorObject object = (EditorObject)enum.nextElement();
      if ((object instanceof EditorPanel) &&
	  ((EditorObject)object).lbl.equals(str)) {
	((EditorPanel)object).show();
      }
    }
  }


  /** 
   * This method is called when the this component receives the input focus. 
   * This method is usually called by handleEvent (II-1.10.22), 
   * in which case the what argument contains the arg field of the event
   * argument. 
   *
   * The method returns true to indicate that it has successfully handled 
   * the action; or false if the event that triggered the action should be 
   * passed up to the component's parent. Most applications should return
   * either true or the value of super.handleEvent(evt). 
   *
   * The getFocus method of Component simply returns true. 
   *
   * The what argument is currently always null. 
   *
   * @param evt - the event that caused the action 
   * @param what - the action 
   * @return true if the event has been handled and no further action 
   * is necessary; false if the event is to be
   * given to the component's parent. 
   */
  /*
  public boolean gotFocus(Event evt, Object what)  {
    return true;
  }
  public boolean mouseEnter(Event evt, int x, int y) {
    requestFocus();
    return true;
  }
  public boolean mouseExit(Event evt, int x, int y) {
    return true;
  }
  public boolean mouseMove(Event evt, int x, int y) {
    return true;
  }

  public boolean lostFocus(Event evt, Object what)  {
    nextFocus();
    return true;
  }
  public void addNotify() {
    super.addNotify();
  }
  public void removeNotify() {
    super.removeNotify();
    }
  */

  /* Vector not needed but implemented for future */
  /* At this moment the font is set for the whole system. Anyway the
     font must be got from SOME Component (Java rules...)
     So, those components which requests allowSetFont are
     put in the vector for future plans (to possibly change
     the font for every Component separately).
     */
  /* Vector setFontObjects = new Vector(); */
  
  public void setFont(String fontName) {
    Font f = getFont();
    setFont(new Font(fontName, f.getStyle(), (int)(EditorObject.itemsize/2)));
    resize();
  }

  public void reshape(int x, int y, int width, int height) {
    super.reshape(x, y, width, height);
    resize(width, height);
  }
  
  public void resize(Dimension d) {
    Enumeration e = objects.elements();
    if ((d.width != 0) && (d.height != 0)) {
      // using double buffering
      if ( (offScreen == null)
	   || (d.width != offScreenSize.width)
	   || (d.height != offScreenSize.height) ) {
	offScreen = createImage(d.width, d.height);
	offScreenSize = d;
	offGraphics = offScreen.getGraphics();
      }
      Font f = getFont();
      offGraphics.setFont(new Font(f.getName(), f.getStyle(), 
				   (int)(EditorObject.itemsize/2)));
      while (e.hasMoreElements()) {
	EditorObject obj = (EditorObject)e.nextElement();
	obj.setFont(new Font(f.getName(), f.getStyle(), 
			     (int)(EditorObject.itemsize/2)));
      }
      EditorObject.setFontMetrics(offGraphics.getFontMetrics());
    }
    /* EditorObject.touchAll(); */
    repaint();
  }
  
  public void resize(int width, int height) {
    resize(new Dimension(width, height));
  }

  /** This method is invoked when the size of the screen is changed. */
  public void resize(int newsize) {
    EditorObject.setItemSize(newsize);
    resize(size());
  }
  public void resize() {
    resize(EditorObject.getItemSize());
  }


  /** 
   * This method returns the size of this graphPanel. 
   * @return Class Dimension d 
   */
  public Dimension getSize() {
    return size();
  }

  //-------- event handling

  /** Invokes method update(). */
  public void paint(Graphics g) { 
    update(g); 
  }
  
  /** 
    The AWT calls the update method in response to a call to repaint
    the GraphPanel. Double buffering is used to eliminate
    flashing. This method invokes paint() methods for all objects in
    the GraphPanel until all objects are ready (they don't slide across
    the screen anymore). 
    */
  Rectangle pickRec;
  public synchronized void update(Graphics g) {
    if (pick == null)
      pickRec = g.getClipRect();
    // clear the clipping area
    offGraphics.setColor(getBackground());
    offGraphics.fillRect(pickRec.x, pickRec.y, 
			 pickRec.width, pickRec.height);
    offGraphics.setColor(drawColor);

    /*
    // DEBUG
    if (DEBUG) {
      offGraphics.drawRect(pickRec.x, pickRec.y, 
			   pickRec.width, pickRec.height); 
    System.out.println("pickRec = " + pickRec);
    }
    */

    // set the fontMetrics for all objects
    EditorObject.setFontMetrics(offGraphics.getFontMetrics());
    for (int element = objects.size() - 1; element >= 0; element--) {
      EditorObject obj = (EditorObject)objects.elementAt(element);
      if ((obj != pick) && (obj.getBounds().intersects(pickRec)))
	obj.paint(offGraphics, pickRec);
    }
    if (pick != null) {
      pickRec = pick.getBounds();
      pick.paint(offGraphics, pickRec);
    } 
    g.drawImage(offScreen, 0, 0, null);
    validate();
    /* System.out.println("validated"); */
  }

  
  /** 
   * This method is called when the mouse button is pushed inside this 
   * component, in which case the x and y arguments contains the x and y 
   * field of the event argument. The coordinate is relative to the
   * top-left corner of this component. 
   * @param evt the event that caused the action 
   * @param x the x coordinate 
   * @param y the y coordinate 
   * @return true if the event has been handled and no further action is 
   * necessary; false if the event is to be given to the component's parent. 
   */
  public boolean mouseDown(Event evt, int x, int y) {
    /* System.out.println("MouseDown " + evt); */
    markMovement();
    if (((pick = findEditorObject(x, y)) != null) && (!pick.disabled())) {
      pick.fixed = true;
      pick.dx = pick.x - x;
      pick.dy = pick.y - y;
      repaint();
    } else pick = null;
    markMovement();
    return true;
  }

  /**
   * This method is called when the mouse button is moved inside this 
   * component with the button pushed, in which case the x and y arguments 
   * contains the x and y field of the event argument. The coordinate
   * is relative to the top- left corner of this component. 
   * Mouse drag events continue to get sent to this component even when the 
   * mouse has left the bounds of the component. The drag events continue 
   * until a mouse up event occurs. 
   *
   * @param evt the event that caused the action 
   * @param x the x coordinate 
   * @param y the y coordinate 
   * @return true if the event has been handled and no further action is 
   * necessary; false if the event is to be given to the component's parent. 
   */
  public boolean mouseDrag(Event evt, int x, int y) {
    /* System.out.println("MouseDrag " + evt); */
    lastx = 0;
    lasty = 0;
    if (pick != null) {
      pick.move(x, y);
      repaint();
    }
    return true;
  }
  
  /**
   * This method is called when the mouse button is released inside this 
   * component, in which case the x and y arguments contains the x and y field 
   * of the event argument. The coordinate is relative to the top-left corner 
   * of the component. 
   *
   * @param evt the event that caused the action 
   * @param x the x coordinate 
   * @param y the y coordinate 
   * @return true if the event has been handled and no further action is 
   * necessary; false if the event is to be given to the component's parent. 
   */
  public boolean mouseUp(Event evt, int x, int y) {
    /* System.out.println("MouseUp " + evt);  */
    // was there any picked object?
    if (pick != null) {
      pick.fixed = false;
      EditorObject obj = findEditorObject(x,y);
      if (obj != null) {
	markMovement();
	/* if ((x == lastx) && (y == lasty)) { // double click -> split
	  obj.split(x,y);
	  x = 0;
	  y = 0;
	} 
	else 
	*/
	if ((lastx == 0) && (lasty == 0)) // other operations
	  pick.move(obj, x, y);
	else // click
	  pick.click();
      }
      markMovement();
      lastx = x;
      lasty = y;
      pick.dx = x;
      pick.dy = y;
      pick = null;
      repaint();
    }
    return true;
  }

  public boolean action(Event e, Object arg) {
    if (e.target instanceof Button) {
      if ("Pop".equals(arg)) { 
	Enumeration enum = objects.elements();
	while (enum.hasMoreElements()) {
	  EditorObject obj = (EditorObject)enum.nextElement();
	  if ((obj instanceof tred.Stack) && 
	      (((tred.Stack)obj).pop == e.target)) {
	    ((tred.Stack)obj).pop();
	    return true;
	  }
	}
      }
      else {
	Enumeration enum = objects.elements();
	while (enum.hasMoreElements()) {
	  EditorObject obj = (EditorObject)enum.nextElement();
	  if ((obj instanceof EditorButton) && 
	      (obj.toString().equals(arg))) {
	    ((EditorButton)obj).push();
	    return true;
	  }
	}
      }
    }
    return false;
  }
}



