package tred;

import java.awt.*; 
import java.util.*;

/** 
 * An abstract class Element is a container for other elements. These
 * subelements may be elements, edges, slots or items. This class
 * is intended to be subclassed so that the subclass gives the
 * structure of the subelements.
 */ 
public abstract class Element extends EditorObject {
  private int space;  
  private int position;

  public ListObject elements = new ListObject();
  
  /**
   * This is the abstract contructor for all elements and every subclass
   * should invoke this constructor in order to set the fields graphPanel,
   * label and space. This constructor also sets the logical position of 
   * the panel on the screen (x,y,width,height), updates some fields in 
   * graphPanel and creates new AnimatedImpl object (if one is not allready 
   * created). Finally this object is added into graphPanel.
   * @param graphPanel which this panel belongs to.
   * @param label is the name of this panel.
   * @param space is the space between slots.
   */
  public Element(GraphPanel graphPanel, String label, int space) {
    Dimension d = graphPanel.getSize();

    this.graphPanel = graphPanel;
    this.lbl = label;
    this.space = space;
    this.position = graphPanel.nextPosition;
    this.x = 0;
    this.y = position * itemsize;
    this.width = d.width;
    this.height = 2*itemsize;

    graphPanel.nextPosition += 2;
    if (animImpl == null)
      animImpl = new AnimatedImpl(graphPanel);
    addPanel(this);
  }

  //-------- protected methods

  /** 
   * This method returns the absolute space between subelements of 
   * this element. Variable space is the relative space between subelements
   * given at the initialization of the element.
   * @return (int) (space * itemsize / 100)
   */
  public final int getSpace() {
    return (int)(space*itemsize/100);
  }

  /** 
   * This method is invoked when a new Element is created. 
   * The call is forwarded to AnimatedImpl.addPanel.
   * @see AnimatedImpl.addPanel()
   */
  public void addPanel(Element element) { 
    animImpl.addPanel(element);
  }

  public void move(int x, int y) {
    this.x = x;
    this.y = y;
  }



  /**
   * This method is invoked when this element is drawn into the screen. 
   */
  public boolean paint(Graphics g) {
    Enumeration elem = elements.elements();
    while (elem.hasMoreElements()) {
      EditorObject obj = (EditorObject)elem.nextElement();
      obj.paint(g);
    }
    return true;
  }
}
