package tred;

import java.awt.*; 
import java.util.*;

/** 
 * An abstract class EditorPanel to panels. This class is intended to be
 * subclassed. 
 */ 
public abstract class EditorPanel extends EditorObject {
  private Vector list = new Vector();
  private boolean showableSlots = false;
  /** order if listArray */
  private boolean stackOrder = true;
  private boolean orientation = false;
  private boolean isInfinite = false;
  protected boolean isListPanel = false;
  /** Table of slots. */
  protected Slot slots[] = new Slot[512];
  /** The length of string items (#nbr of items to be created by panel). */
  protected int len;
  /** Number of slots in this panel. */
  protected int nslots;
  /** Items to be created by panel. */
  protected String items;
  /** Space between slots. */
  protected int space;
  /** Logical position of the panel on the screen. */
  protected int position;

  //-------- constructor

  /**
   * This is the abstract contructor for all panels and every subclass
   * should invoke this constructor in order to set the graphPanel,
   * label, items, len (length of items string i.e. number of items),
   * and space fields. This constructor also sets the logical position of 
   * the panel on the screen, updates some fields in graphPanel
   * and creates new AnimatedImpl object (if one is not allready created).
   * @param graphPanel which this panel belongs to.
   * @param label is the name of this panel.
   * @param items are the items which belongs this panel.
   * @param space is the space between slots.
   */
  public EditorPanel(GraphPanel graphPanel, String label, String items, int space) {
    this.graphPanel = graphPanel;
    this.lbl = label;

    // kludge: String.replace do not work!
    if (items != null)
      for (int i=0; i<items.length(); i++)
	if (items.substring(i, i+1).equals("_"))
	  items = items.substring(0, i) + " " + items.substring(i+1);

    this.items = items;
    this.space = space;
    len = items == null ? 0 : items.length();
    position = graphPanel.nextPosition;
    x = 0;
    y = position * itemsize;
    Dimension d = graphPanel.getSize();
    width = d.width;
    height = 2*itemsize;
    graphPanel.nextPosition += 2;
    if (animImpl == null)
      animImpl = new AnimatedImpl(graphPanel);
    addPanel(this);

  }

  //-------- protected methods

  /** This method returns the space between slots of this panel. 
   * @return space between two slots. 
   */
  protected final int getSpace() {
    /* the following fails in lkhp-machines with netscape3 */
    /*    return (int)Math.rint(space*itemsize/100); */
    return (int)(space*itemsize/100);
  }

  /* @return boolean value of variable showableSlots */
  protected final boolean showableSlots() {
    return showableSlots;
  }

  /** 
   * This method is invoked in order to get the state of this panel for
   * export.
   * @param str is the substitute string for empty slot.
   * @return the current state (string of items in the slots) of this panel. 
   * @see animImpl.export
   */
  protected String getState(String str) {
    String s = "";
    Item item;
    for (int i=0; i<nslots; i++) {
      if ((item = slots[i].getItem()) != null) {
	if ((!item.disabled()) || (isListPanel))
	  s += item.lbl == null ? " " : item.lbl;
	if ((isListPanel) && (slots[i].list != null)) {
	  s += ": ";
	  Enumeration enum = slots[i].list.elements();
	  while (enum.hasMoreElements()) {
	    Slot slot = (Slot)enum.nextElement();
	    if ((item = slot.getItem()) != null)
	      s += item.lbl + " ";
	  }
	  s += "\n";
	}
      }
      else 
	if (str != null)
	  s += str;
    }
    return s;
  }

  //-------- public methods

  /** 
   * Subclasses which have something to reset should override this method. 
   * The reset method of EditorPanel simply do nothing.
   */
  public void reset() {}

  public void append(String items) {}

  /** 
   * This method is invoked when a new EditorPanel is created. 
   * The call is forwarded to AnimatedImpl.addPanel.
   * @see AnimatedImpl.addPanel()
   */
  public void addPanel(EditorPanel panel) { 
    animImpl.addPanel(panel);
  }

  public final boolean getOrder() {
    return stackOrder;
  }

  public void setOrder(String order) {
    if (order.equals("APPEND") || order.equals("append"))
      stackOrder = false;
    else if (order.equals("STACK") || order.equals("stack"))
      stackOrder = true;
  }

  public final boolean isTreeOrientated() {
    return orientation;
  }

  public final void setOrientation(boolean flag) {
    orientation = flag;
  }

   /** 
    * This method creates the ListObject for the items in this panel. 
    * @param items items to be listObjected.
    * @see ListObject
    */
  public void setListArray(String items) {
    isListPanel = true;
    StringTokenizer t = null;
    if (items != null)
      t = new StringTokenizer(items, ";");
    for (int i=0; i<nslots; i++) {
      slots[i].list = new ListObject();
      Item item = slots[i].getItem();
      if (item != null)
	item.disable();
      if ((items != null) && t.hasMoreTokens()) {
	String lbls = t.nextToken();
	for (int j=0; j<lbls.length(); j++) {
	  NodeItem nodeItem = new NodeItem(lbls.substring(j, j+1)); 
	  nodeItem.move(slots[i],0,0);
	  graphPanel.insertObject(nodeItem);
	}
      }
    }
  }

  public void setInfinite() {
    isInfinite = true;
    for (int i=0; i<nslots; i++) {
      Item item = slots[i].getItem();
      item.setInfinite();
    }
  }

  /** This method marks slots of this panel as shadowed. */
  public void shadowed() {
    for (int i=0; i<nslots; i++) {
      if (slots[i].shadowItem == null)
	slots[i].shadowItem = slots[i].item;
      if (slots[i].list != null) {
	Enumeration enum = slots[i].list.elements();
	while (enum.hasMoreElements()) {
	  Slot slot = (Slot)enum.nextElement();
	  if (slot.shadowItem == null)
	    slot.shadowItem = slot.item;
	}
      }
    }
  }

  /** This method marks names of slots of this panels to be displayed. */
  public void show() {
    showableSlots = true;
  }

  /** 
   * This method determines which slot of this panel is located in (x,y).
   * @return the slot in coordinates (x,y). If there is no slot in the (x,y)
   * return null.
   */
  public Slot getSlot(int x, int y) {
    for (int i=0; i<nslots; i++) {
      if ((x > slots[i].x) && 
	  (y > slots[i].y) && 
	  (x < slots[i].x + slots[i].width) && 
	  (y < slots[i].y + slots[i].height)) {

	// should I remove this code?
	/*
	  if (isListPanel) {
	  Slot newslot = (Slot)slots[i].clone();
	  newslot.id = -1;
	  int idx;
	  if ((idx = list.indexOf(slots[i])) == -1)
	    list.insertElementAt(slots[i],++idx);
	  list.insertElementAt(newslot, idx);
	  for (idx = list.indexOf(slots[i]) - 1;
	       ((Slot)list.elementAt(idx)).id == -1;
	       idx--)
	    ((Slot)list.elementAt(idx)).y += 2*itemsize;
	  return newslot;
	}
	*/

	return slots[i];
      }
    }
    return null;
  }

  public void move(int x, int y) {
    this.x = x;
    this.y = y;
  }

  /**
   * This method is invoked when this panel is draw into the screen. 
   */
  public boolean paint(Graphics g) {
    //   System.out.println("sfsd");
    if (lbl != null)
      g.drawString(lbl, x, y + itemsize/2 + fontMetrics.getAscent()/2);
    return true;
  }
}



