package tred;

import java.awt.*; 
import java.io.*;

/** Class Edge is the implementation for edges of the Class Graph. */
public class Edge extends EditorObject {
  private static int nextId = 0;
  protected Slot fslot, tslot;
  private int id;
  /* private Graph g; */
  private boolean traveled;

  /** 
   * This constructor initializes all the fields. It also sets the
   * label for this edge.
   * @param g is the Graph which this edge belongs to.
   * @param id is the id-number of this edge. 
   * @param fslot is the slot where this edge starts from.
   * @param tslot is the slot where this edge ends to.
   */
  public Edge(int id, Slot fslot, Slot tslot) {
    this.id = id;
    this.fslot = fslot;
    this.tslot = tslot;
    if ((fslot == null) || (tslot == null))
      System.err.println("Error: Cannot initialize edge between null slots.");
    else
      setLabel();
    /* invoked("new edge " + fslot + " to " + tslot); */
  }
  
  public Edge(Slot fslot, Slot tslot) {
    this(nextId++, fslot, tslot);
  }

  private void setLabel() {
    this.lbl = fslot + "->" + tslot;
  }

  // -------- private methods

  private int min(int a, int b) {
      return a < b ? a : b;
  }

  private int max(int a, int b) {
      return a > b ? a : b;
  }

  // -------- public methods

  private void setTo(Slot slot) {
    /* invoked("setTo " + fslot + " to " + slot); */
    tslot = slot;
    setLabel();
  }

  public Slot getTo() {
    return tslot;
  }

  private void setFrom(Slot slot) {
    fslot = slot;
    setLabel();
  }

  public Slot getFrom() {
    return fslot;
  }

  public Rectangle getBounds() {
    Slot s1 = getFrom().getMyRoot();
    Slot s2 = getTo().getMyRoot();
    if ((s1 != null) && (s2 != null)) {
      int dc = itemsize/2;
      int s1x = s1.x + dc;
      int s1y = s1.y + dc;
      int s2x = s2.x + dc;
      int s2y = s2.y + dc;
      x = (s1x < s2x) ? s1x : s2x;
      y = (s1y < s2y) ? s1y : s2y;
      width = Math.abs(s1x - s2x);
      height = Math.abs(s1y - s2y);
      Rectangle r = new Rectangle(x, y, width, height);
      return r;
    }
    else return null;
  }

  /**
   * Edge is not a movable object as items are but when edge is picked
   * (and this method invoked) we toggle the traveled flag of this edge.
   */
  public void move(EditorObject obj, int x, int y) {
    if ((x != 0) && (y != 0)) {
      if (obj instanceof Slot) {
    	moveObject(this, getTo(), obj);
	setTo((Slot)obj);
      }
      else {
	moveObject(this, this, this);
	traveled = !traveled;
      }
    }
    else if (obj instanceof Slot)
      setTo((Slot)obj);
    else if (obj instanceof Edge)
      traveled = !traveled;
  }

  public void click() {
    move(this, 1, 1);
  }


  /**
   * This method is invoked by the GraphPanel in order to determine if
   * this edge is "near enough" to position (x,y).
   * @return true if this edge is "near enought" (some two pixels)
   * to point (x,y). 
   */
  public boolean inside(int x, int y) {
    int x1 = fslot.x + itemsize/2;
    int y1 = fslot.y + itemsize/2;
    int x2 = tslot.x + itemsize/2;
    int y2 = tslot.y + itemsize/2;
    if ((x >= min(x1, x2) - 1) && (x <= max(x1, x2) + 1) &&
	(y >= min(y1, y2) - 1) && (y <= max(y1, y2) + 1)) {
      double a = Math.sqrt((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1));
      double k = ((x2-x1)*(x2-x) + (y2-y1)*(y2-y))/a;
      if (Math.sqrt((x-x2)*(x-x2) + (y-y2)*(y-y2) - k*k) < 2)
	return true;
    }
    return false;
  }

  /** 
   * Paint method for Edge. Draws an solid line between fromSlot and
   * toSlot. If traveled flag is set, the line is thicker.
   * @return allways true.
   */
  public boolean paint(Graphics g) {
    if ((fslot == null) || (tslot == null))
      return true;
    Slot fslot = getFrom().getMyRoot();
    Slot tslot = getTo().getMyRoot();
    if (fslot == tslot)
      return true;
    if (!((fslot.isVisible()) && (tslot.isVisible())))
      return true;

    /* invoked("paint it " + getFrom() + " -> " + getTo()); */

    int dcenter = itemsize/2;
    // centers (x,y) of p(arent) and t(arget)
    int cxp = fslot.x + dcenter;
    int cyp = fslot.y + dcenter;
    int cxt = tslot.x + dcenter;
    int cyt = tslot.y + dcenter;
    // length inside the slot (dx, dy)
    int xl = (cxp-cxt);
    int yl = (cyp-cyt);
    int dx = (int)Math.round(dcenter * xl / Math.sqrt(xl*xl + yl*yl));
    int dy = (int)Math.round(dcenter * yl / Math.sqrt(xl*xl + yl*yl));
    // draw the connecting lines only to the border of the slot
    g.drawLine(cxp - dx, cyp - dy, cxt + dx, cyt + dy);
    if (traveled) {
      g.drawLine(cxp - dx + 1, cyp - dy + 1, cxt + dx + 1, cyt + dy + 1);
      g.drawLine(cxp - dx + 1, cyp - dy, cxt + dx + 1, cyt + dy);
      g.drawLine(cxp - dx, cyp - dy + 1, cxt + dx, cyt + dy + 1);
    }
    return true; 
  }
  
  public String toString() {
    return "Edge " + lbl;
  }

}
