package tred;

import java.awt.*; 
import java.util.*;

/** 
 * Implementation to BSlots. BSlot is a container for multiple NodeItems
 * in B plus tree.
 * @see Slot.
 */
class BSlot extends NodeSlot implements Cloneable {
  private Edge fatherEdge = null;
  private static int nextId = 0;
  private boolean paint = true;

  /** This is the constructor for BSlot. This constructor simply do nothing
   * but invokes the super constructor. (The slot's x and y fields are also
   * set to zero.)
   */
  BSlot(EditorPanel panel, int id) {
    super(panel, id);
    this.list = new ListObject();
    fatherEdge = new Edge(this, this);
    getBounds = true;
    x = y = 0;
  }

  BSlot(EditorPanel panel) {
    this(panel, ++nextId);
    moveObject(this, null, this);
  }

  protected boolean isRootNode() {
    return (((BPlus)editorPanel).getHead() == this);
  }

  protected void setRootNode(BSlot bSlot) {
    if (bSlot == null)
      ((BPlus)editorPanel).move();
    else
      ((BPlus)editorPanel).move(bSlot, -1, -1);
  }

  protected void buildTree() {
    ((BPlus)editorPanel).setLocation(); 
  }

  // ----------- Link operations 

  public boolean isHeadSlot() {
    return (!isItemSlot() && list != null);
  }

  public boolean isItemSlot() {
    return (getItem() != null);
  }

  public boolean isLinkSlot() {
    return (super.isLinkSlot());
  }

  public boolean isEnabledLinkSlot() {
    BSlot slot = getLink();
    if (slot != null) {
      return (isLinkSlot() && 
	      (slot != this) && 
	      (slot != null) &&
	      (slot.getLink() == this) &&
	      (slot.isVisible()));
    }
    else return false;
  }

  public void setLinkSlot() {
    super.setLinkSlot();
    setLink(this);
  }

  public boolean hasLink(BSlot bSlot) {
    if (list == null) 
      return false;
    Enumeration enum = list.elements();
    while (enum.hasMoreElements()) {
      BSlot next = (BSlot)enum.nextElement();
      if (next == bSlot)
	return true;
    }
    return false;
  }

  private boolean hasLinkTo(BSlot bSlot) {
    if ((bSlot == null) || (list == null))
      return false;
    Enumeration enum = list.elements();
    while (enum.hasMoreElements()) {
      BSlot next = (BSlot)enum.nextElement();
      if ((next.isLinkSlot()) && (next.getLink() == bSlot))
	return true;
    }
    return false;
    
  }

  private boolean hasBSlotChild(BSlot bSlot) {
    if (list == null) {
      Warning("hasBSlotChild: no link list ("+this+"["+bSlot+"])");
      return false;
    }
    if (this.hasLinkTo(bSlot))
      return true;
    Enumeration enum = list.elements();
    while (enum.hasMoreElements()) {
      BSlot bs = (BSlot)enum.nextElement();
      BSlot link = bs.getLink();
      if (bs.isLinkSlot()) {
	if (link == this) {
	  Warning("hasBSlotChild: unconsistent link "+bs+" points to "+link);
	  return false;
	}
	if (link.hasBSlotChild(bSlot))
	  return true;
      }
    }
    return false;
  }

  private boolean hasItemChild(Item item) {
    if (item == null)
      return false;
    else
      return hasBSlotChild((BSlot)item.getSlot());
  }

  public boolean hasChild(EditorObject obj) {
    if (obj instanceof BSlot)
      return hasBSlotChild((BSlot)obj);
    else if (obj instanceof Item)
      return hasItemChild((Item)obj);
    else return false;
  }

  /** @param edge which points to this nodes father */
  public void setLinkEdge(Edge edge) {
    fatherEdge = edge;
  }
  /** @return Edge father */
  public Edge getLinkEdge() {
    return fatherEdge;
  }

  public void setLink(BSlot bSlot) {
    Edge edge = getLinkEdge();
    if (edge != null)
      edge.move(bSlot,0,0);
    else {
      Warning("setLink (new edge): this should never occur.");
      edge = new Edge(0, this, bSlot);
      setLinkEdge(edge);
    }
  }

  /** @return Slot slot - father slot of this slot - null if not defined */
  public BSlot getLink() {
    Edge edge = getLinkEdge();
    if (edge != null)
      return (BSlot)edge.getTo();
    else
      return null;
  }

  // ----------- Manipulating the structure

  /** Delete this slot */
  public boolean erase(int x, int y) {
    Item thisItem;
    
    hide(true);
    Enumeration enum = list.elements();
    while (enum.hasMoreElements()) { // hide also the items if exists
      BSlot slot = (BSlot)enum.nextElement();
      if ((thisItem = slot.getItem()) != null)
	thisItem.delete(x, y); 
    }
    disable();
    moveObject(this, null, this);
    return true;
  }

  /** Move into point (x,y). Invoked by GraphPanel. */
  public void move(int x, int y) {
    this.x = x;
    this.y = y;
    getBounds = true;
    super.setLocation(x,y);
  }

  /** Move into a new Link. Set bidirectional links */
  private void moveTo(BSlot newLink) {
    /* invoked("moveTo: " + newLink); */
    BSlot oldFather = (BSlot)this.getLink();
    oldFather.setLink(oldFather);
    this.setLink(newLink);
    newLink.setLink(this);
  }

  /** Trash the whole subtree and all the items in nodes. */
  private void trashSubTree(Trash trash) {
    if (list != null) {
      Enumeration enum = list.elements();
      while (enum.hasMoreElements()) {
	BSlot obj = (BSlot)enum.nextElement();
	if (obj.getItem() != null)
	  obj.getItem().move((Trash)trash, 1, 1);
	if (obj.isLinkSlot())
	  obj.getLink().move((Trash)trash, 1, 1);
      }
    }
  }

  /** This method is invoked in order to move this slot into the Trash. */
  public void move(Trash obj, int x, int y) {
    /* invoked("trash"); */
    BSlot oldFather = (BSlot)this.getLink();
    moveObject(this, oldFather, obj);
    if (this.isRootNode())
      setRootNode(null);
    trashSubTree((Trash)obj);
    oldFather.setLink(oldFather);
  }

  /** This method is invoked by the animator. */
  public void animatorMove(EditorObject obj) {
    Item thisItem;
    if (obj == null) {     // dynamic slot deleted
      hide(true); 
      if ((thisItem = getItem()) != null)
	thisItem.hide(true); // hide also the item if exists
    }
    else if (obj == this) {     // dynamic slot created
      hide(false); 
      if ((thisItem = getItem()) != null)
	thisItem.hide(false); // set also the item visible if exists
    }
    else if (obj instanceof BSlot)
      moveTo((BSlot)obj);
    else Warning("animatorMove: unknown move (" + obj + ")");
    buildTree();
  }

  /** 
   * This method is invoked in order to move this slot into a slot
   * just left side of the given item. A new link node is created. 
   */
  public void move(Item obj, int x, int y) {
    /* invoked("move into item " + obj); */
    Slot slot = obj.getSlot();
    if (slot.getMyRoot() == this) // can't move this into a node item of this
      return;
    else if (slot != null) {
      Slot linkSlot = slot.insertItemBefore(null, (Item)obj);
      /* invoked("move (Item): new link created " + linkSlot); */
      move(linkSlot, x, y);
    } else Warning("move(Item): trying to move into a null slot.");
  }

  /** This method is invoked in order to move this slot into a
   * given tree. The real insertion position (slot) is determined by the
   * tree and a move method is invoked with that slot.
   */
  public void move(BPlus obj, int x, int y) {
    BSlot slot = (BSlot)obj.getSlot(x,y);
    if (slot != null) { // move to slot of item
      Slot s = slot.insertItem(null, (Item)null);
      move(s, x, y);
    }
  }

  /** This method is invoked in order to move this subtree
   * (represented by this slot) as a subtree of given slot.
   */
  public void move(BSlot obj, int x, int y) {
    BSlot oldFather = (BSlot)this.getLink();
    BSlot newFather = (BSlot)obj;

    if (this.hasChild(newFather)) {
      Warning("move: trying to move into a child!");
      return;
    }
      
    if (this.isRootNode())
      setRootNode((BSlot)newFather.getMyRoot());

    /* if obj was not a link we must create one */
    if (!newFather.isLinkSlot()) { // moving to head node or item node
      Slot tmpSlot = null;
      if (((BSlot)obj).getItem() != null)
	tmpSlot = (BSlot)obj;  // moving to item node (tmpSlot=items slot)
      else
	tmpSlot = (Slot)newFather.list.firstElement();
      Item item = null;
      if (tmpSlot != null)
	item = tmpSlot.getItem(); // get the item or null
      Slot slot = newFather.insertItemBefore(null, item); // insrt new link
      newFather = (BSlot)slot; // newFather set to link just created
      /* invoked("move: new link created " + newFather); */
	    
    }
    moveTo(newFather);
    moveObject(this, oldFather, newFather);
  }
  
  /** 
   * This method is invoked when this slot is moved.
   * This method invokes proper move method determined by
   * the instance of the given parameter obj.
   * @param EditorObject obj - object where this is moved to 
   * @param int x, int y - unused
   */
  public void move(EditorObject obj, int x, int y) {
    if (obj == this) return;
    if (obj instanceof Item) { move((Item)obj, x, y); return; }
    if (obj instanceof BPlus) { move((BPlus)obj, x, y); return; }
    if (obj instanceof BSlot) { move((BSlot)obj, x, y); }
    if (obj instanceof Trash) { move((Trash)obj, x, y); }
    buildTree();
  }

  /**
    * This method is a shorthand notation for move(obj, 1, 1)
    */
  public void move(EditorObject obj) {
    move(obj,1,1);
  }

  public Object clone() {
    BSlot bSlot = (BSlot)super.clone();
    bSlot.fatherEdge = new Edge(this, this);
    bSlot.getBounds = true;
    return bSlot;
  }

  /** This method is invoked when this slot is splitted. */
  public boolean split(int x, int y) {
    // splitting slot with item
    if ((!isLinkSlot()) && (getItem() != null)) {
      BSlot newSlot = null, headSlot = (BSlot)getMyRoot();
      if (headSlot.sizeOfVisibleLinks(headSlot.list) == 0) {
	newSlot = new BSlot(editorPanel);
	Slot slot = headSlot.insertItem(null, (Item)null);
	newSlot.move(slot);
      }
      newSlot = new BSlot(editorPanel);
      Slot slot = headSlot.insertItem(null, (Item)null);
      newSlot.move(slot);
      return true;
    }
    BSlot oldRoot = (BSlot)this.getLink();
    if (this.isRootNode()) { // root slot splitted
      BSlot newRoot = new BSlot(editorPanel);
      BSlot newChild = new BSlot(editorPanel);
      Slot s1 = newRoot.insertItem(null, (Item)null);
      Slot s2 = newRoot.insertItem(null, (Item)null);
      this.move(s2);
      newChild.move(s1);
    }
    else { // some other slot than root splitted
      BSlot newChild = new BSlot(editorPanel);
      Slot s = oldRoot.getMyRoot().insertItemBefore(null, this.getLink());
      newChild.move(s);
    }
    return true;
  }

  /** This method is invoked when a new item is inserted into this tree. 
   * The item is inserted into this slot as it's first element.
   * @param item - item which is inserted
   */
  protected Slot putItem(Item item) {
    Slot slot = super.putItem(item);
    slot.getMyRoot().list = sortLinks(slot.getMyRoot().list);
    buildTree();
    return slot;
  }

  protected Slot insertItem(Item item, EditorObject after) {
    Slot slot = super.insertItem(item, after);
    slot.getMyRoot().list = sortLinks(slot.getMyRoot().list);
    buildTree();
    return slot;
  }
  protected Slot insertItemBefore(Item item, EditorObject before) {
    Slot slot = super.insertItemBefore(item, before);
    slot.getMyRoot().list = sortLinks(slot.getMyRoot().list);
    buildTree();
    return slot;
  }

  /* @return Slot if it's in this node at position (x,y) */
  private Slot getSlot(int x, int y) {
    if ((x > this.x) && 
	(y > this.y) && 
	(x < this.x + this.width) && 
	(y < this.y + this.height))
      return this;
    else
      if (list != null) {
	Enumeration enum = list.elements();
	while (enum.hasMoreElements()) {
	  Slot slot = (BSlot)enum.nextElement();
	  if ((x > slot.x) && 
	      (y > slot.y) && 
	      (x < slot.x + slot.width) && 
	      (y < slot.y + slot.height))
	    return slot;
	}
      }
    return null;
  }

  /* 
   * @return EditorObject type element (slot or edge) if it's in this 
   * subtree at position (x,y) 
   */
  public EditorObject hasElement(int x, int y) {
    Slot slot;
    if ((slot = getSlot(x, y)) != null)
      return slot;
    else {
      if (this.isHeadSlot()) {
	Edge edge = getLinkEdge();
	if (edge != null) {
	  if (edge.inside(x, y))
	    return edge;
	}
      }
      Enumeration enum = list.elements();
      while (enum.hasMoreElements()) {
	BSlot bSlot = (BSlot)enum.nextElement();
	if (bSlot.isEnabledLinkSlot()) {
	  BSlot lSlot = (BSlot)bSlot.getLink();
	  EditorObject obj = null;
	  if (bSlot.getMyRoot() != lSlot.getMyRoot())
	    obj = lSlot.hasElement(x, y);
	  if (obj != null)
	    return obj;
	}
      }
    } 
    return null;
  }

  /* @return Slot if it's in this subtree at position (x,y) */
  public Slot hasSlot(int x, int y) {
    Slot slot;
    if ((slot = getSlot(x, y)) != null)
      return slot;
    else {
      if (list == null) {
	Warning("hasSlot: list == null");
	return null;
      }
      Enumeration enum = list.elements();
      while (enum.hasMoreElements()) {
	BSlot bSlot = (BSlot)enum.nextElement();
	BSlot lSlot = (BSlot)bSlot.getLink();
	if (bSlot.getMyRoot() != lSlot.getMyRoot())
	  lSlot = (BSlot)lSlot.hasSlot(x, y);
	else
	  lSlot = null;
	if (lSlot != null)
	  return lSlot;
      }
    } 
    return null;
  }
  
  /* return the real amount of visible child slots (empty ones not included) */
  private int sizeOfVisibleLinks(ListObject list) {
    int i = 0;
    if (list == null) return 0;
    Enumeration enum = list.elements();
    while (enum.hasMoreElements()) {
      Object obj = enum.nextElement();
      if ((obj instanceof BSlot) && (((BSlot)obj).isLinkSlot())) {
	BSlot bSlot = ((BSlot)obj).getLink();
	if ((bSlot != this) && 
	    (bSlot.isVisible()) && 
	    (!bSlot.equals(obj)) &&
	    (obj.equals(bSlot.getLink())))
	  i++;
      }
    }
    return i;
  }

  // ---------------- Structure construction

  Rectangle sRect = null;
  public Rectangle getBounds() {
    if (getBounds) {
      Rectangle tmpRect, rect = super.getBounds(); // this slot
      if (isHeadSlot()) {
	tmpRect = getLinkEdge().getBounds(); // link into the father
	if (tmpRect != null)
	  rect = rect.union(tmpRect);
	Enumeration enum = list.elements(); // list
	while (enum.hasMoreElements()) {
	  BSlot bSlot = (BSlot)enum.nextElement();
	  if (bSlot.isEnabledLinkSlot()) { // if link, count the edge
	    tmpRect = bSlot.getLinkEdge().getBounds();
	  }
	  else if (bSlot.isItemSlot()) { // if item slot, count the slot
	    tmpRect = bSlot.getBounds();
	  }
	  if (tmpRect != null)
	    rect = rect.union(tmpRect);
	}
      }
      sRect = rect;
      getBounds = false;
      return rect;
    }
    return sRect;
  }

  private ListObject sortLinks(ListObject list) {
    if (list == null) {
      Warning("sortLinks: trying to sort null list");
      return null;
    }
    Enumeration enum = list.elements();
    Enumeration slotEnum;
    ListObject links = new ListObject();
    ListObject slots = new ListObject();
    ListObject newList = new ListObject();
    while (enum.hasMoreElements()) {
      BSlot bSlot = (BSlot)enum.nextElement();
      if (bSlot.isLinkSlot())
	links.append(bSlot);
      else {
	slots.append(bSlot);
	bSlot.list = newList;
      }
    }
    enum = links.elements();
    slotEnum = slots.elements();
    BSlot bSlot, tmpSlot;
    while ((enum.hasMoreElements()) && (slotEnum.hasMoreElements())) {
      do {
	bSlot = (BSlot)enum.nextElement();
	newList.append(bSlot);
      } while (enum.hasMoreElements() && bSlot.getLink() == bSlot);
      do {
	tmpSlot = (BSlot)slotEnum.nextElement();
	newList.append(tmpSlot);
      } while (slotEnum.hasMoreElements() && (tmpSlot.getItem() == null));
    }
    while (enum.hasMoreElements()) {
      bSlot = (BSlot)enum.nextElement();
      newList.append(bSlot);
    }
    while (slotEnum.hasMoreElements()) {
      tmpSlot = (BSlot)slotEnum.nextElement();
      newList.append(tmpSlot);
    }
    return newList;
  }

  public void setLocation(int x, int y, int width) {
    int size = sizeOfVisibleLinks(list), neww = width, j = 0;
    getBounds = true;
    if (fixed) { // picked up
      x = this.x;
      y = this.y;
    }
    if (isVisible()) {
      /* list = sortLinks(list); */
      super.setLocation(x, y);
    }
    if (list == null) { 
      Warning("setLocation: null list"); 
      return; 
    }
    if (size > 0) neww = width/size;
    Enumeration enum = list.elements();
    while (enum.hasMoreElements()) {
      Object obj = enum.nextElement();
      if ((obj instanceof BSlot) && (((BSlot)obj).isLinkSlot())) {
	BSlot bSlot =  ((BSlot)obj).getLink();
	if (bSlot.getMyRoot() != ((BSlot)obj).getMyRoot()) {
	  if ((isVisible()) && (bSlot.getMyRoot().isVisible()))
	    bSlot.setLocation(x+((2*j+++1)*neww-width)/2, 
			      y+2*itemsize,
			      neww);
	  /* else
	    bSlot.setLocation(x, y, width); */
	}
      }
    }
  }

  /* 
   * Paint this and all the child slots and the possible edge.
   * @param g Graphics context
   * @return allways true
   */
  public boolean paint(Graphics g) {
    if (isVisible()) {
      super.paint(g); // paint this 
      if (list != null) {
	Enumeration enum = list.elements(); // paint child slots
	while (enum.hasMoreElements()) {
	  Object obj = enum.nextElement();
	  if ((obj instanceof BSlot) && (((BSlot)obj).isLinkSlot())) {
	    BSlot bSlot =  ((BSlot)obj).getLink();
	    if (bSlot.getMyRoot() != this)
	      bSlot.paint(g);
	  }
	}
      }
      if (!this.isRootNode()) { // paint the edge
	Edge e = getLinkEdge();
	if (e != null)
	  e.paint(g);
      }
    }
    return true;
  }

  public boolean paint(Graphics g, Rectangle rect) {
    if (isVisible()) {
      paint = (getBounds().intersects(rect));
      if (paint) super.paint(g); // paint this 
      if (list != null) {
	Enumeration enum = list.elements(); // paint child slots
	while (enum.hasMoreElements()) {
	  Object obj = enum.nextElement();
	  if ((obj instanceof BSlot) && (((BSlot)obj).isLinkSlot())) {
	    BSlot bSlot =  ((BSlot)obj).getLink();
	    if (bSlot.getMyRoot() != this)
	      bSlot.paint(g, rect);
	  }
	}
      }
      if (paint && !this.isRootNode()) { // paint the edge
	Edge e = getLinkEdge();
	if (e != null)
	  e.paint(g);
      }
    }
    return true;
  }

  public String toString() {
    String inStr = "";
    if (item != null)
      inStr = item.toString();
    else if (list != null) {
      if (this != getMyRoot())
	inStr = list.toString() + "(" + lbl + ")";
      else 
	inStr = list.toString();
    }
    return "BSlot" + lbl + " (" + inStr + ")";
  }
}

