#!/bin/sh

CXX_DEFAULT="g++"
CXXFLAGS_DEFAULT="-O -g -Wall -Wshadow"
LDFLAGS_DEFAULT=""


function show_help
{
	echo "configure.sh [switches]

The switches:
  --help               This help.
  --disable-mysql      Disable MySQL support even if detected.
  --disable-postgresql Disable PostgreSQL support even if detected.
  --cxx=...            Use a specific C++ compiler. By default, environment
                       variable CXX is used, or ${CXX_DEFAULT} if CXX is not present.
  --cxxflags=...       Use specific compiler flags. By default, environment
                       variable CXXFLAGS is used, or 
		         \"${CXXFLAGS_DEFAULT}\"
		       if CXXFLAGS is not present.
  --linker=...         Use a specific linker. By default, the C++ compiler is
                       used.
  --ldflags=...        Use specific linker flags. By default, environment
                       variable LDFLAGS is used, or empty if LDFLAGS is not
		       present.
  --disable-deps       Disable autodependencies. Use if cannot compile otherwise.

If the dependencies cannot be found, use MYSQL_CONFIG, PG_CONFIG and
RAPTOR_CONFIG environment variables to force the location of mysql_config,
pg_config and raptor-config.
"
}


while [ $# -ge 1 ]
do
    FLAG="$1"

    shift 1
    case "$FLAG" in
	--disable-mysql)
	    MYSQL_CONFIG="disabled"
	    ;;

	--disable-postgresql)
	    PG_CONFIG="disabled"
	    ;;

	--help)
	    show_help
	    exit
	    ;;

  	--cxx=*)
	    CXX=${FLAG:6} 
	    ;;

	--cxxflags=*)
	    CXXFLAGS=${FLAG:11}
	    ;;

        --linker=*)
	    LINKER=${FLAG:9}
	    ;;

	--ldflags=*)
	    LDFLAGS=${FLAG:10}
	    ;;

	--disable-deps)
	    SKIPAUTODEPS=1
	    ;;

	*)
	    echo "Unknown switch $FLAG. Try --help"
	    exit 1
	    ;;
    esac
done

# defaults
CXX="${CXX:-${CXX_DEFAULT}}"
CXXFLAGS="${CXXFLAGS:-${CXXFLAGS_DEFAULT}}"
INCLUDES=""
LDFLAGS="${LDFLAGS:-${LDFLAGS_DEFAULT}}"
LINKER="${LINKER:-${CXX}}"

# mysql
echo -n "Checking for MySQL... "
MYSQL_CONFIG=${MYSQL_CONFIG:-$(which mysql_config)}
MYSQL_CONFIG=${MYSQL_CONFIG:-$(which mysql_config5)}
if [ "${MYSQL_CONFIG}" != "disabled" -a -f "${MYSQL_CONFIG}" ]
then
    HAS_MYSQL=1
    INCLUDES="${INCLUDES} $(${MYSQL_CONFIG} --include)"
    LDFLAGS="${LDFLAGS} $(${MYSQL_CONFIG} --libs)"
    echo "Ok"
else
    HAS_MYSQL=0
    echo "Not found"
fi


echo -n "Checking for PostgreSQL... "
PG_CONFIG=${PG_CONFIG:-$(which pg_config)}
PG_CONFIG=${PG_CONFIG:-$(which /opt/local/lib/postgresql84/bin/pg_config)}
PG_CONFIG=${PG_CONFIG:-$(which /opt/local/lib/postgresql83/bin/pg_config)}
PG_CONFIG=${PG_CONFIG:-$(which /opt/local/lib/postgresql82/bin/pg_config)}
PG_CONFIG=${PG_CONFIG:-$(which /opt/local/lib/postgresql81/bin/pg_config)}
if [ "${PG_CONFIG}" != "disabled" -a -f "${PG_CONFIG}" ]
then
    HAS_POSTGRESQL=1
    INCLUDES="${INCLUDES} -I$(${PG_CONFIG} --includedir)"
    LDFLAGS="${LDFLAGS} -L$(${PG_CONFIG} --libdir) -lpq"
    echo "Ok"
else
    HAS_POSTGRESQL=0
    echo "Not found"
fi

# raptor
echo -n "Checking for raptor... "
RAPTOR_CONFIG=${RAPTOR_CONFIG:-$(which raptor-config)}
if [ "${RAPTOR_CONFIG}" != "disabled" -a -f "${RAPTOR_CONFIG}" ]
then
    INCLUDES="${INCLUDES} $(${RAPTOR_CONFIG} --cflags)"
    LDFLAGS="${LDFLAGS} $(${RAPTOR_CONFIG} --libs)"
    echo "Ok"
else
    echo "Not found"
    echo "Raptor required for compile."
    exit 1
fi

echo -n "Writing Makefile.config... "
echo "# Generated by configure.sh" >Makefile.config
echo "CXX:=${CXX}" >>Makefile.config
echo "CXXFLAGS:=${CXXFLAGS} ${INCLUDES}" \
    "-DHAS_MYSQL=${HAS_MYSQL}" \
    "-DHAS_POSTGRESQL=${HAS_POSTGRESQL}" \
    >>Makefile.config
echo "LINKER:=${LINKER}" >>Makefile.config
echo "LDFLAGS:=${LDFLAGS}" >>Makefile.config
echo "HAS_MYSQL:=${HAS_MYSQL}" >>Makefile.config
echo "HAS_POSTGRESQL:=${HAS_POSTGRESQL}" >>Makefile.config
echo "SKIPAUTODEPS:=${SKIPAUTODEPS}" >>Makefile.config
echo "Ok"
