/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <string.h>

#include <iostream>
#include <fstream>
#include <typeinfo>
#include <vector>
#include <limits>

#include <raptor.h>

#include "AQLException.h"
#include "AQLModel.h"
#include "AQLParser.h"
#include "AQLListParser.h"
#include "AQLToSQLTranslator.h"
#include "AQLQueryExecutor.h"
#include "AQLSupport.h"
#include "FlexibleSQLLayout.h"
#include "SQLBackend.h"
#include "SPARQLParser.h"

#if HAS_MYSQL
#include "SQLBackendMySQL.h"
#endif

#if HAS_POSTGRESQL
#include "SQLBackendPostgres.h"
#endif

#include "Messages.h"
#include "N3TripleReader.h"
#include "TripleAPI.h"

#include "Version.h"

namespace {
   using namespace TypeRQ;

   struct ArgParseException : TypeRQ::AQLException {
      ArgParseException(const char *format ...) {
         va_list ap;
         va_start(ap, format);
         setFormattedReason(format, ap);
         va_end(ap);
      }
   };

   enum OPERATING_MODE_ENUM {
      OM_HELP=0,
      OM_GENERATE_CREATE_SQL,
      OM_LOAD_TRIPLES,
      OM_QUERY,
   };

   enum AQL_PARSER_ENUM {
      AP_LIST,
      AP_SPARQL,
   };

   enum SQL_LAYOUT_ENUM {
      SL_FLEXIBLE_SQL_LAYOUT,
   };

   enum SQL_BACKEND_ENUM {
      SB_MYSQL,
      SB_POSTGRES,
   };

   struct OperatingDataContainer
   {
      SQLLayout *sqlLayout;
      SQLBackend *sqlBackend;
      std::string sqlConnectionString;
      TripleAPI *tripleAPI;
      QueryExecutor *queryExecutor;

      SPARQLParser::VariableBindingMode sparqlVariableBindingMode;
      AQL_PARSER_ENUM aqlParser;
      OPERATING_MODE_ENUM operatingMode;
      SQL_LAYOUT_ENUM sqlLayoutMode;
      SQL_BACKEND_ENUM sqlBackendMode;
      int forceMaxPropertyAccessOrder;

      std::string stopAtStage;

      std::istream *is;
      bool closeIsInDestructor;

      FlexibleLayoutDescriptor flexibleLayoutDescriptor;

      std::string inputFile;


      OperatingDataContainer() :
         sqlLayout(0), sqlBackend(0), tripleAPI(0), queryExecutor(0),
         sparqlVariableBindingMode(SPARQLParser::VBM_BOTTOM_UP),
         aqlParser(AP_SPARQL), operatingMode(OM_QUERY), sqlLayoutMode(SL_FLEXIBLE_SQL_LAYOUT),
         sqlBackendMode(SB_MYSQL), forceMaxPropertyAccessOrder(-1), is(0), closeIsInDestructor(false),
         flexibleLayoutDescriptor()
      {
         FlexibleSQLLayout::useSimpleIndexedLayout(flexibleLayoutDescriptor);
      }

      ~OperatingDataContainer()
      {
         delete queryExecutor;
         delete tripleAPI;
         delete sqlLayout;
         delete sqlBackend;

         if (closeIsInDestructor) delete is;
      }

   };


   void printHelp()
   {
      std::cout <<
         "Usage: typearque [--query] [options] <input_file>\n"
         "       typearque --load [options] <input_file>\n"
         "       typearque --sqlcreate [options]\n"
         "\n"
         "  <input_file>    Input file or - for stdin\n"
         "\n"
         "Operating modes:\n"
         "  --query         (default). Perform query. Input file contains query.\n"
         "  --load          Load triples instead querying. <input_file> contains\n"
         "                  triples in N3 format.\n"
         "  --sqlcreate     Outputs statements for creating the SQL schema. No input\n"
         "                  is used and no database connection is attempted.\n"
         "\n"
         "Options:\n"
         "  --help          Display this help\n"
         "  --version       Display version information\n"
         "  --quiet         Output only the final result.\n"
         "  --verbose       Verbose output. Prints intermediate results.\n"
         "  --debug         Lots of debug stuff\n"
         "  --trace         Lots more of debug stuff\n"
         "  --parser=...    Select AQL parser front-end. One of: list, sparql.\n"
         "                  Default: sparql\n"
         "  --stop-at=...   Stops query processing after specific stage and display\n"
         "                  working data. Stages for list: aql, sql, result (default).\n"
         "                  Stages for sparql: sparql, aql, sql, result (default).\n"
         "  --sqllayout=... Selects sql layout from template. Options are: \n"
         "                  simple-inline, simple-indexed (default), versatile-indexed\n"
         "                  versatile-compact\n"
         "  --sqlbackend=.. Selects SQL backend. One of: mysql (default), postgres\n"
         "  --conn=...      Backend-specific connection string.\n"
#if HAS_MYSQL
         "                  - MySQL: connection string is of form\n"
         "                    [key1=value1[,key2=value2[...]]]. Backslash (\\) can be used\n"
         "                    to escape characters. Keys: user, pass, host, port, db.\n"
         "                    E.g., `user=aqlu,pass=aqlp\\,,db=aql,host=server,port=12345'\n"
#else
         "                  - MySQL: Support not compiled in\n"
#endif
#if HAS_POSTGRESQL
         "                  - Postgres: Postgres connection string. try something like\n"
         "                    `host=localhost user=postgres dbname=aql'. See details at\n"
         "         http://www.postgresql.org/docs/8.1/interactive/libpq.html#LIBPQ-CONNECT\n"
#else
         "                  - Postgres: Support not compiled in\n"
#endif
         "  --fflattenlj    Force flattening of nested left joins. This is a debugging\n"
         "                  option and should not be normally needed, as the translation\n"
         "                  process should be able to determine when flattening is\n"
         "                  necessary. However, in some cases, this could yield better\n"
         "                  query execution performance.\n"
         "  --noflattenlj   Disable flattening of nested left joins. This is for debugging\n"
         "                  only. Can be used to test whether variable scoping works for\n"
         "                  SPARQL to AQL transformation phase.\n"
         "  --varbind=...   Variable binding mode for SPARQL parser. Default: bottomup\n"
         "                    bottomup    Bottom-up binding. Nested match patterns have\n"
         "                                precedence in variable binding.\n"
         "                    leftright   Left-to-right. First occurrence in SPARQL query\n"
         "                                string in textual representation binds.\n"
         "                                Historical.\n"
         "";
      std::cout << std::endl;
      std::cout << getCopyrightBanner();
   }

   void parseArgs(OperatingDataContainer &odc, int argc, char **argv)
   {

      // parse switches
      char **argp=argv+1;
      for (; argp!=argv+argc; ++argp) {
         char *arg=*argp;
         if (strncmp(arg, "--", 2)==0) {
            // it's a switch

            if (strlen(arg)==2) {
               // end of switches
               ++argp;
               break;
            }

            if (strcmp(arg+2, "help")==0) {
               // help
               odc.operatingMode=OM_HELP;
               goto parse_bail_out;
            }
            else if (strcmp(arg+2, "quiet")==0) {
               outputLevel=OL_QUIET;
            }
            else if (strcmp(arg+2, "verbose")==0) {
               outputLevel=OL_VERBOSE;
            }
            else if (strcmp(arg+2, "debug")==0) {
               outputLevel=OL_DEBUG;
            }
            else if (strcmp(arg+2, "trace")==0) {
               outputLevel=OL_TRACE;
            }
            else if (strncmp(arg+2, "stop-at=", 8)==0) {
               odc.stopAtStage=arg+10;
            }
            else if (strcmp(arg+2, "parser=list")==0) {
               odc.aqlParser=AP_LIST;
            }
            else if (strcmp(arg+2, "parser=sparql")==0) {
               odc.aqlParser=AP_SPARQL;
            }
            else if (strcmp(arg+2, "sqlbackend=mysql")==0) {
               odc.sqlBackendMode=SB_MYSQL;
            }
            else if (strcmp(arg+2, "sqlbackend=postgres")==0) {
               odc.sqlBackendMode=SB_POSTGRES;
            }
            else if (strcmp(arg+2, "sqllayout=simple-inline")==0) {
               FlexibleSQLLayout::useSimpleInlineLayout(odc.flexibleLayoutDescriptor);
               odc.sqlLayoutMode=SL_FLEXIBLE_SQL_LAYOUT;
            }
            else if (strcmp(arg+2, "sqllayout=simple-indexed")==0) {
               FlexibleSQLLayout::useSimpleIndexedLayout(odc.flexibleLayoutDescriptor);
               odc.sqlLayoutMode=SL_FLEXIBLE_SQL_LAYOUT;
            }
            else if (strcmp(arg+2, "sqllayout=versatile-indexed")==0) {
               FlexibleSQLLayout::useVersatileIndexedLayout(odc.flexibleLayoutDescriptor);
               odc.sqlLayoutMode=SL_FLEXIBLE_SQL_LAYOUT;
            }
            else if (strcmp(arg+2, "sqllayout=versatile-compact")==0) {
               FlexibleSQLLayout::useVersatileCompactLayout(odc.flexibleLayoutDescriptor);
               odc.sqlLayoutMode=SL_FLEXIBLE_SQL_LAYOUT;
            }
            else if (strcmp(arg+2, "sqlcreate")==0) {
               odc.operatingMode=OM_GENERATE_CREATE_SQL;
            }
            else if (strcmp(arg+2, "load")==0) {
               odc.operatingMode=OM_LOAD_TRIPLES;
            }
            else if (strncmp(arg+2, "conn=", 5)==0) {
               odc.sqlConnectionString=arg+7;
            }
            else if (strcmp(arg+2, "fflattenlj")==0) {
               odc.forceMaxPropertyAccessOrder=0;
            }
            else if (strcmp(arg+2, "noflattenlj")==0) {
               odc.forceMaxPropertyAccessOrder=std::numeric_limits<int>::max();
            }
            else if (strcmp(arg+2, "varbind=bottomup")==0) {
               odc.sparqlVariableBindingMode=SPARQLParser::VBM_BOTTOM_UP;
            }
            else if (strcmp(arg+2, "varbind=leftright")==0) {
               odc.sparqlVariableBindingMode=SPARQLParser::VBM_LEFT_TO_RIGHT;
            }
            else {
               // no match found, this is error
               throw ArgParseException("Unknown switch %s", arg);
            }
         } else {
            // not a switch
            break;
         }
      }
     parse_bail_out:

      int expectedArgs=1;
      if (odc.operatingMode==OM_GENERATE_CREATE_SQL) expectedArgs=0;

      // There should be 2 arguments left, namely the database and input files
      if (argc - (argp-argv) < expectedArgs)
      {
         odc.operatingMode=OM_HELP;
      }

      if (argc - (argp-argv) > expectedArgs)
      {
         odc.operatingMode=OM_HELP;
      }

      if (odc.operatingMode==OM_HELP) expectedArgs=0;

      if (expectedArgs>0)
      {
         odc.inputFile=argp[0];
      }
   }

   void performQuery(OperatingDataContainer &odc)
   {
      int stopAtStage=odc.queryExecutor->getFinalStage();
      if (!odc.stopAtStage.empty())
      {
         int i;
         for (i=1; i<=odc.queryExecutor->getFinalStage(); ++i)
         {
            if (odc.stopAtStage==odc.queryExecutor->getStageName(i))
            {
               break;
            }
         }
         if (i>odc.queryExecutor->getFinalStage())
         {
            throw AQLException("Unknown stage '%s'", odc.stopAtStage.c_str());
         }
         stopAtStage=i;
      }

      odc.queryExecutor->newQuery(odc.is, false);
      for (int i=1; i<=stopAtStage; ++i)
      {
         odc.queryExecutor->advanceToStage(i);
         if (i!=stopAtStage && outputLevel>=OL_VERBOSE)
         {
            odc.queryExecutor->printCurrentStageResult(std::cout);
            std::cout << std::endl;
         }
      }
      odc.queryExecutor->printCurrentStageResult(std::cout);
   }

   struct SQLInserterTripleHandler : N3TripleHandler
   {
      OperatingDataContainer &odc;

      SQLInserterTripleHandler(OperatingDataContainer &_odc) : odc(_odc) {}

      void handleTriple(const AQLLiteralExpr &s, const AQLLiteralExpr &p, const AQLLiteralExpr &o)
      {
         if (outputLevel>=OL_DEBUG)
         {
            AQLPrinterVisitor v(std::cout);
            AQLLiteralExpr(s).accept(v);
            AQLLiteralExpr(p).accept(v);
            AQLLiteralExpr(o).accept(v);
         }

         odc.tripleAPI->insertTriple(s, p, o);
      }
   };

   void loadTriples(OperatingDataContainer &odc)
   {
      N3TripleReader tripleReader;
      SQLInserterTripleHandler tripleHandler(odc);

      if (odc.inputFile=="-")
         tripleReader.openConsoleIn();
      else
         tripleReader.openFile(odc.inputFile);

      Transaction tx=odc.sqlBackend->newTransaction();
      tripleReader.readTriples(tripleHandler);
      tx.commit();

      tripleReader.close();
   }

   int processMain(int argc, char **argv)
   {
      using namespace TypeRQ;

      OperatingDataContainer odc;

      parseArgs(odc, argc, argv);

      print(OL_NORMAL, "This is %s, the experimental SPARQL to SQL translator.\n\n", getVersionString());

      if (odc.operatingMode==OM_HELP)
      {
         printHelp();
         return 1;
      }

      // ok, and go

      switch (odc.sqlBackendMode)
      {
         case SB_MYSQL:
#if HAS_MYSQL
            odc.sqlBackend=new SQLBackendMySQL();
            break;
#else
            throw AQLException("MySQL backend disabled from build");
#endif

         case SB_POSTGRES:
#if HAS_POSTGRESQL
            odc.sqlBackend=new SQLBackendPostgres();
            break;
#else
            throw AQLException("Postgres backend disabled from build");
#endif

         default:
            throw AQLException("Bad sql backend mode: %d", (int)odc.sqlBackendMode);
      }

      switch (odc.sqlLayoutMode)
      {
         case SL_FLEXIBLE_SQL_LAYOUT:
         {
            FlexibleSQLLayout *flexibleSqlLayout=
               new FlexibleSQLLayout(odc.flexibleLayoutDescriptor, *odc.sqlBackend);

            if (odc.forceMaxPropertyAccessOrder>=0)
            {
               flexibleSqlLayout->overrideMaxPropertyAccessDepth(odc.forceMaxPropertyAccessOrder);
            }

            odc.sqlLayout=flexibleSqlLayout;
            break;
         }

         default:
            throw AQLException("Bad layout mode: %d", (int)odc.sqlLayoutMode);
      }

      if (odc.operatingMode==OM_GENERATE_CREATE_SQL)
      {
         print(OL_QUIET, "%s\n", odc.sqlLayout->generateCreateSQL().c_str());
         return 0;
      }

      odc.sqlBackend->connect(odc.sqlConnectionString);
      odc.tripleAPI=new TripleAPI(*odc.sqlLayout, *odc.sqlBackend);

      if (odc.operatingMode==OM_LOAD_TRIPLES)
      {
         loadTriples(odc);
      }
      else if (odc.operatingMode==OM_QUERY)
      {
         if (!odc.inputFile.empty())
         {
            if (odc.inputFile=="-") {
               odc.is=&std::cin;
               odc.closeIsInDestructor=false;
            } else {
               odc.is=new std::ifstream(odc.inputFile.c_str());
               odc.closeIsInDestructor=true;
               if (odc.is->fail()) {
                  throw AQLException("Could not open input file '%s'", odc.inputFile.c_str());
               }
            }
         }

         switch (odc.aqlParser)
         {
            case AP_LIST:
               odc.queryExecutor=new AQLQueryExecutor(*odc.sqlLayout, *odc.sqlBackend);
               break;

            case AP_SPARQL:
               odc.queryExecutor=new SPARQLQueryExecutor(*odc.sqlLayout, *odc.sqlBackend, odc.sparqlVariableBindingMode);
               break;
         }

         performQuery(odc);
      }
      else {
         throw AQLException("Unhandled operating mode %d", (int)odc.operatingMode);
      }

      return 0;
   }

} // end of anonymous namespace

int main(int argc, char **argv) {
   using namespace TypeRQ;
   int ret;

   raptor_init();

   try {
      ret=processMain(argc, argv);
   }
   catch (ArgParseException &ex) {
      std::cerr << "Argument exception: " << ex.getReason() << std::endl;
      ret=1;
   }
   catch (AQLException &exception) {
      std::cerr << "Translation exception";
      if (outputLevel>=OL_DEBUG)
      {
         const std::type_info &info = typeid(exception);
         std::cerr << ' ' << info.name();
      }
      std::cerr << ": " << exception.getReason() << std::endl;
      ret=1;
   }
   raptor_finish();
}
