/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <sstream>

#include "AQLException.h"
#include "SQLLayout.h"
#include "SQLBackend.h"
#include "AQLModel.h"
#include "Messages.h"
#include "TripleAPI.h"
#include "AQLSupport.h"


namespace {

}

namespace TypeRQ {

   TripleAPI::TripleAPI(SQLLayout &_layout, SQLBackend &_backend)
      : sqlLayout(_layout), sqlBackend(_backend), tripleIdSeq(), valueIdSeq()
   {
      // nothing here
   }

   TripleAPI::~TripleAPI() {}

   int TripleAPI::getReferenceForValue(const AQLLiteralExpr &literal, AQLPropertyExpr::Property property)
   {
      std::list<std::string> sqlQueries;
      sqlLayout.getReferenceForValueSQL(literal, property, sqlQueries);

      if (sqlQueries.empty()) return 0;

      std::stringstream unionQuery;
      bool first=true;

      for (std::list<std::string>::iterator i=sqlQueries.begin();
           i!=sqlQueries.end(); ++i)
      {
         if (outputLevel>=OL_DEBUG)
         {
            print(OL_DEBUG, "Query part: %s\n",  i->c_str());
         }
         if (!first)
         {
            unionQuery << " UNION ";
         }
         else {
            first=false;
         }
         unionQuery << *i;
      }

      print(OL_DEBUG, "UNION Query: %s\n", unionQuery.str().c_str());

      SQLResultIterator result=sqlBackend.executeQuery(unionQuery.str());
      if (result.nextRow())
      {
         return result.getInt(0);
      }
      else {
         return -1;
      }
   }

   int TripleAPI::insertLiteral(const AQLLiteralExpr &literalExpr, AQLPropertyExpr::Property property)
   {
      std::list<std::string> sqlStatements;
      int64_t refId=++valueIdSeq;
      sqlLayout.generateInsertLiteralSQL(refId, literalExpr, property, sqlStatements);

      for (std::list<std::string>::iterator i=sqlStatements.begin();
           i!=sqlStatements.end(); ++i)
      {
         if (outputLevel>=OL_DEBUG)
         {
            print(OL_DEBUG, "Executing statement: %s\n",  i->c_str());
         }

         sqlBackend.executeStatement(*i);
      }

      return refId;
   }

   void TripleAPI::insertTriple(const AQLLiteralExpr &_s, const AQLLiteralExpr &_p, const AQLLiteralExpr &_o)
   {
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "Inserting literals\n");
      }

      const AQLLiteralExpr &s=simplifyLiteral(_s);
      const AQLLiteralExpr &p=simplifyLiteral(_p);
      const AQLLiteralExpr &o=simplifyLiteral(_o);

      int subjId=getReferenceForValue(s, AQLPropertyExpr::SUBJECT);
      if (subjId<0) subjId=insertLiteral(s, AQLPropertyExpr::SUBJECT);

      int predId=getReferenceForValue(p, AQLPropertyExpr::PREDICATE);
      if (predId<0) predId=insertLiteral(p, AQLPropertyExpr::PREDICATE);

      int objId=getReferenceForValue(o, AQLPropertyExpr::OBJECT);
      if (objId<0) objId=insertLiteral(o, AQLPropertyExpr::OBJECT);

      print(OL_DEBUG, "Inserting triple\n");

      std::list<std::string> sqlStatements;
      sqlLayout.generateInsertTripleSQL(++tripleIdSeq, s, subjId, p, predId, o, objId, sqlStatements);
      for (std::list<std::string>::iterator i=sqlStatements.begin();
           i!=sqlStatements.end(); ++i)
      {
         if (outputLevel>=OL_DEBUG)
         {
            print(OL_DEBUG, "Executing statement: %s\n", i->c_str());
         }
         sqlBackend.executeStatement(*i);
      }
   }
}
