/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <map>
#include <string>

#include "AQLDebug.h"
#include "AQLModel.h"

namespace TypeRQ {

   struct SQLBackendCapabilities;

   class SQLLayout
   {
   public:
      SQLLayout();

      virtual ~SQLLayout();

      virtual void prepareAQLForTranslationFinal(AQLQuery &) = 0;
      virtual std::string translateCustomAQLExpr(AQLCustomExpr &) = 0;

      typedef std::map<AQLJoinGroupLike *, std::string> join_condition_map;


      /**
       * Creates SQL expression for joining join group and all variables. SQLCondition is the condition
       * for triple joining or empty, if no conditions apply.
       *
       * If join==0, this is the root join
       *
       * Returns string that can be appended after "FROM ", "INNER JOIN ", or "LEFT JOIN ".
       */
      virtual std::string createSQLForAQLJoinGroupTree(AQLQuery &aql, const join_condition_map &) = 0;


      /**
       * Returns SQL code for accessing a property. May be non-simple, e.g., could
       * contain type casts, case expressions, etc. Note that property access must be
       * created before use. This is to ensure that necessary join structures are in place.
       */
      virtual std::string getPropertyAccess(const std::string &joinName, AQLPropertyExpr::Property,
                                            AQLTypeSet) = 0;

      /**
       * Returns statements which can be used to create the SQL tables for the layout.
       */
      virtual std::string generateCreateSQL() = 0;


      /**
       * Returns SQL statement for retrieving reference (int) for literal value. Empty string
       * if int references are ignored - e.g. inline layouts
       */
      virtual void getReferenceForValueSQL(const AQLLiteralExpr &, AQLPropertyExpr::Property,
                                           std::list<std::string> &queryStatements) = 0;

      /**
       * Returns a list of SQL statements, which insert literal with given reference id.
       * Note that reference id is only used by indexed layouts - and must be unique per literal
       */
      virtual void generateInsertLiteralSQL(int refId,
                                            const AQLLiteralExpr &s,
                                            AQLPropertyExpr::Property,
                                            std::list<std::string> &sqlStatements) = 0;


      /**
       * Returns a list of SQL statements, which insert triple. Literals must already be
       * inserted.
       */
      virtual void generateInsertTripleSQL(int tripleId,
                                           const AQLLiteralExpr &s, int sRefId,
                                           const AQLLiteralExpr &p, int pRefId,
                                           const AQLLiteralExpr &o, int oRefId,
                                           std::list<std::string> &sqlStatements) = 0;


      /**
       * Returns SQL-escaped string literal
       */
      virtual std::string escapeString(const std::string &stringLiteral) = 0;

      /**
       * Returns SQL backend capability bits.
       */
      virtual const SQLBackendCapabilities &getBackendCapabilities() = 0;

      //  0: all nested left joins should be flattened
      // >0: specific max property access depth
      // std::numeric_limits<int>::max(): no limit
      virtual int getMaxPropertyAccessDepth() const = 0;

   };

}
