/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "AQLException.h"
#include "SQLJoinBuilder.h"
#include "StringUtils.h"

namespace {

   using namespace TypeRQ;

   void printIndent(std::ostream &ss, int nestingLevel)
   {
      ss << "\n     ";
      for (int i=0; i<nestingLevel; ++i)
      {
         ss << "            ";
      }
   }

   void printJoinExpressionRecursive(std::ostream &ss, const JoinExpression &join, int nestingLevel)
   {
      if (join.joinTerms.empty()) throw AQLException("Join must have at least 1 term");

      {
         SeparatedStringBuilder ssb(" CROSS JOIN ");
         for (std::list<JoinTerm>::const_iterator i=join.joinTerms.begin();
              i!=join.joinTerms.end(); ++i)
         {
            const JoinTerm &term=*i;
            ssb.newPart();
            ssb << term.table << " AS " << term.alias;
         }
         ss << ssb.getString();
      }

      // add inner joins
      for (std::list<InnerJoinExpression *>::const_iterator i=join.nestedJoins.begin();
           i!=join.nestedJoins.end(); ++i)
      {
         const InnerJoinExpression &inner=*(*i);
         printIndent(ss, nestingLevel);
         if (inner.isLeftOuter)
            ss << "  LEFT JOIN ";
         else
            ss << " INNER JOIN ";

         const int totalTerms=inner.joinTerms.size() + inner.nestedJoins.size();

         if (totalTerms > 1)
         {
            ss << '(';
         }

         printJoinExpressionRecursive(ss, inner, nestingLevel+1);

         if (totalTerms > 1)
         {
            ss << ')';
         }
         if (!inner.nestedJoins.empty())
         {
            printIndent(ss, nestingLevel);
            ss << "        ";
         }
         ss << " ON ";
         ss << inner.joinCondition;
      }
   }

}

namespace TypeRQ {

   JoinExpression::~JoinExpression()
   {
      for (std::list<InnerJoinExpression *>::iterator i=nestedJoins.begin();
           i!=nestedJoins.end(); ++i)
      {
         InnerJoinExpression *inner=*i;
         delete inner;
      }
   }

   void printJoinExpression(std::ostream &ss, const JoinExpression &join)
   {
      printJoinExpressionRecursive(ss, join, 0);
   }

}
