/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <sstream>

#include <libpq-fe.h>

#include "AQLModel.h"
#include "AQLException.h"
#include "AQLSupport.h"
#include "FormatUtils.h"
#include "SQLBackendPostgres.h"
#include "SQLBackendFunctions.h"
#include "Messages.h"
#include "ExpressionWriter.h"

namespace {

   const char *plpgsqlConvertToInt=
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_integer(t text)\n"
      "RETURNS INTEGER AS $$\n"
      "BEGIN\n"
      "    RETURN aqltosql_any_to_integer(aqltosql_any_to_double(t));\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_integer(t boolean)\n"
      "RETURNS INTEGER AS $$\n"
      "BEGIN\n"
      "    RETURN CASE WHEN t THEN 1 ELSE 0 END;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_integer(t double precision)\n"
      "RETURNS INTEGER AS $$\n"
      "BEGIN\n"
      "    BEGIN\n"
      "        RETURN CAST(t AS INTEGER);\n"
      "    EXCEPTION WHEN OTHERS THEN\n"
      "        RETURN NULL;\n"
      "    END;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n";

   const char *plpgsqlConvertToDouble=
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_double(t text)\n"
      "RETURNS DOUBLE PRECISION AS $$\n"
      "BEGIN\n"
      "    BEGIN\n"
      "        RETURN CAST(t AS DOUBLE PRECISION);\n"
      "    EXCEPTION WHEN OTHERS THEN\n"
      "        RETURN NULL;\n"
      "    END;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_double(t integer)\n"
      "RETURNS DOUBLE PRECISION AS $$\n"
      "BEGIN\n"
      "    RETURN t;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_double(t boolean)\n"
      "RETURNS DOUBLE PRECISION AS $$\n"
      "BEGIN\n"
      "    RETURN CASE WHEN t THEN 1 ELSE 0 END;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n";

   const char *plpgsqlConvertBooleanToText=
      "CREATE OR REPLACE FUNCTION aqltosql_boolean_to_text(t boolean)\n"
      "RETURNS TEXT AS $$\n"
      "BEGIN\n"
      "    RETURN CAST(aqltosql_any_to_integer(t) AS TEXT);\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n";

   const char *plpgsqlConvertTimestampToText=
      "CREATE OR REPLACE FUNCTION aqltosql_timestamp_to_text(t timestamp)\n"
      "RETURNS TEXT AS $$\n"
      "BEGIN\n"
      "    RETURN to_char(t, 'YYYY-MM-DD\"T\"HH24:MI:SS.MS\"Z\"');\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n";

   const char *plpgsqlConvertToBoolean=
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_boolean(t text)\n"
      "RETURNS BOOLEAN AS $$\n"
      "BEGIN\n"
      "    RETURN CASE WHEN t='1' THEN TRUE WHEN t='0' THEN FALSE ELSE NULL END;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_boolean(t double precision)\n"
      "RETURNS BOOLEAN AS $$\n"
      "BEGIN\n"
      "    RETURN CASE WHEN t=1 THEN TRUE WHEN t=0 THEN FALSE ELSE NULL END;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_boolean(t integer)\n"
      "RETURNS BOOLEAN AS $$\n"
      "BEGIN\n"
      "    RETURN CASE WHEN t=1 THEN TRUE WHEN t=0 THEN FALSE ELSE NULL END;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n";

   const char *plpgsqlConvertToTimestamp=
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_timestamp(t text)\n"
      "RETURNS TIMESTAMP AS $$\n"
      "BEGIN\n"
      "    RETURN to_timestamp(t, 'YYYY-MM-DD\"T\"HH24:MI:SS.MS\"Z\"'); \n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_timestamp(t double precision)\n"
      "RETURNS TIMESTAMP AS $$\n"
      "BEGIN\n"
      "    RETURN to_timestamp(t) AT TIME ZONE 'UTC';\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_timestamp(t integer)\n"
      "RETURNS TIMESTAMP AS $$\n"
      "BEGIN\n"
      "    RETURN to_timestamp(t) AT TIME ZONE 'UTC';\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_timestamp(t boolean)\n"
      "RETURNS TIMESTAMP AS $$\n"
      "BEGIN\n"
      "    RETURN CASE WHEN t THEN (to_timestamp(0) AT TIME ZONE 'UTC') ELSE NULL END;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n";

   const char *plpgsqlConvertToIri=
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_iri(t text)\n"
      "RETURNS TEXT AS $$\n"
      "BEGIN\n"
      "    RETURN t;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_iri(t timestamp)\n"
      "RETURNS TEXT AS $$\n"
      "BEGIN\n"
      "    RETURN NULL;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_iri(t double precision)\n"
      "RETURNS TEXT AS $$\n"
      "BEGIN\n"
      "    RETURN NULL;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n"
      "CREATE OR REPLACE FUNCTION aqltosql_any_to_iri(t integer)\n"
      "RETURNS TEXT AS $$\n"
      "BEGIN\n"
      "    RETURN NULL;\n"
      "END;\n"
      "$$ LANGUAGE plpgsql IMMUTABLE RETURNS NULL ON NULL INPUT;\n";
}

namespace TypeRQInternal
{
   using namespace TypeRQ;

   struct PostgresContext
   {
      PGconn *conn;
      SQLBackendCapabilities caps;


      PostgresContext() : conn(0)
      {
      }

      PGresult *execute(const std::string &statement, bool isQuery)
      {
         PGresult *result=PQexec(conn, statement.c_str());

         if (result==NULL)
         {
            // fatal error
            throw AQLException("Fatal SQL execution error: %s", PQerrorMessage(conn));
         }

         ExecStatusType resultStatus=PQresultStatus(result);
         if (resultStatus==PGRES_FATAL_ERROR)
         {
            AQLException e=AQLException("Error in SQL execution: %s", PQresultErrorMessage(result));
            PQclear(result);
            throw e;
         }

         const ExecStatusType expectedStatus=(isQuery? PGRES_TUPLES_OK : PGRES_COMMAND_OK);

         if (resultStatus!=expectedStatus)
         {
            PQclear(result);
            throw AQLException("Expected SQL result status %d(%s) but got %d(%s)",
                               static_cast<int>(expectedStatus), PQresStatus(expectedStatus),
                               static_cast<int>(resultStatus), PQresStatus(resultStatus));
         }

         if (isQuery)
         {
            return result;
         }
         else {
            // statement
            PQclear(result);
            return 0;
         }
      }

   };
}

namespace {
   using namespace TypeRQ;
   using namespace TypeRQInternal;

   class ToTextMapping : public SQLFunctionSelectionMapping
   {
   protected:
      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         AQLTypeSet::ExprType sourceType=expr.arguments.front()->getExprTypeSet().singularType();

         switch (sourceType)
         {
            case AQLTypeSet::IRI:
            case AQLTypeSet::STRING:
               break;

            case AQLTypeSet::INTEGER:
            case AQLTypeSet::DOUBLE:
               ew.startFunction("CAST");
               break;

            case AQLTypeSet::DATETIME:
               ew.startFunction("aqltosql_timestamp_to_text");
               break;

            case AQLTypeSet::BOOLEAN:
               ew.startFunction("aqltosql_boolean_to_text");
               break;

            default:
               throw AQLException("No cast to text for %s", getNameForExprType(sourceType));
         }
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         AQLTypeSet::ExprType sourceType=expr.arguments.front()->getExprTypeSet().singularType();

         switch (sourceType)
         {
            case AQLTypeSet::IRI:
            case AQLTypeSet::STRING:
               break;

            case AQLTypeSet::INTEGER:
            case AQLTypeSet::DOUBLE:
               ew << " AS TEXT";
               ew.endOp();
               break;

            case AQLTypeSet::DATETIME:
               ew.endOp();
               break;

            case AQLTypeSet::BOOLEAN:
               ew.endOp();
               break;

            default:
               throw AQLException("No cast to text for %s", getNameForExprType(sourceType));
         }
      }

   public:
      ToTextMapping(const std::string _sparqlFn, AQLTypeSet::ExprType _et) :
         SQLFunctionSelectionMapping(_sparqlFn)
      {
         *this << &(*new SQLFunctionType(AQLTypeSet(_et)) << AQLTypeSet(AQLTypeSet::ANY));
      }
   };


   SQLFunctionMap *instantiatePostgresFunctionMap()
   {
      SQLFunctionMapping *postgresFunctions[]={
         // typecastings
         &(*new SQLFixedFunctionMapping("builtin:to-integer", "aqltosql_any_to_integer")
           << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::INTEGER)) << AQLTypeSet(AQLTypeSet::ANY))),
         &(*new SQLFixedFunctionMapping("builtin:to-double", "aqltosql_any_to_double")
           << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::DOUBLE)) << AQLTypeSet(AQLTypeSet::ANY))),
         &(*new SQLFixedFunctionMapping("builtin:to-boolean", "aqltosql_any_to_boolean")
           << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN)) << AQLTypeSet(AQLTypeSet::ANY))),
         &(*new SQLFixedFunctionMapping("builtin:to-datetime", "aqltosql_any_to_timestamp")
           << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::DATETIME)) << AQLTypeSet(AQLTypeSet::ANY))),
         &(*new SQLFixedFunctionMapping("builtin:to-iri", "aqltosql_any_to_iri")
           << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::IRI)) << AQLTypeSet(AQLTypeSet::ANY))),
         new ToTextMapping("builtin:to-string", AQLTypeSet::STRING),
         0
      };

      return new SQLDelegatingFunctionMap(getStandardSQLFunctionMap(), postgresFunctions);
   }

   class PostgresResultIteratorImpl : public SQLResultIteratorImpl
   {
   protected:
      PostgresContext &ctx;
      PGresult *const result;

      const int cols, rows;
      int currentRow;

      std::string tmp;

   public:
      PostgresResultIteratorImpl(PostgresContext &_ctx, PGresult *_result) :
         ctx(_ctx), result(_result), cols(PQnfields(result)), rows(PQntuples(result)),
         currentRow(-1)
      {
      }

      ~PostgresResultIteratorImpl()
      {
         PQclear(result);
      }

      int getColCount()
      {
         return cols;
      }

      bool nextRow()
      {
         ++currentRow;
         return currentRow < rows;
      }

      const std::string &getString(int col)
      {
         char *s=PQgetvalue(result, currentRow, col);
         tmp=s;
         return tmp;
      }

      int64_t getInt(int col)
      {
         char *s=PQgetvalue(result, currentRow, col);
         std::stringstream ss(s);
         int64_t ret;
         ss >> ret;
         return ret;
      }

      double getDouble(int col)
      {
         char *s=PQgetvalue(result, currentRow, col);
         std::stringstream ss(s);
         double ret;
         ss >> ret;
         return ret;
      }

      bool isNull(int col)
      {
         return PQgetisnull(result, currentRow, col) != 0;
      }
   };

   class PostgresTransactionImpl : public TransactionImpl
   {
   protected:
      PostgresContext &ctx;
      bool active;
   public:
      PostgresTransactionImpl(PostgresContext &_ctx) : ctx(_ctx), active(false) {}

      bool isActive() const
      {
         return active;
      }

      void begin()
      {
         ctx.execute("START TRANSACTION", false);
         active=true;
      }

      void commit()
      {
         ctx.execute("COMMIT", false);
         active=false;
      }

      void rollback()
      {
         ctx.execute("ROLLBACK", false);
         active=false;
      }
   };
}

namespace TypeRQ {

   using namespace TypeRQInternal;

   SQLBackendPostgres::SQLBackendPostgres() :
      functionMap(instantiatePostgresFunctionMap()), postgresContext(new PostgresContext) {}

   SQLBackendPostgres::~SQLBackendPostgres()
   {
      disconnect();
      delete functionMap;
      delete postgresContext;
   }

   const std::string &SQLBackendPostgres::getName()
   {
      static const std::string name("postgres");
      return name;
   }

   SQLFunctionMap &SQLBackendPostgres::getFunctionMap()
   {
      return *functionMap;
   }

   void SQLBackendPostgres::connect(const std::string &connectionString)
   {
      disconnect();

      // try to connect
      PGconn *conn=PQconnectdb(connectionString.c_str());
      if (!conn) throw AQLException("Connection failed, error message not available.");

      if (PQstatus(conn)!=CONNECTION_OK)
      {
         // connection failed
         throw AQLException("Connection failed: %s", PQerrorMessage(conn));
      }

      postgresContext->conn=conn;

      try {
         executeStatement(plpgsqlConvertToInt);
         executeStatement(plpgsqlConvertToDouble);
         executeStatement(plpgsqlConvertBooleanToText);
         executeStatement(plpgsqlConvertTimestampToText);
         executeStatement(plpgsqlConvertToBoolean);
         executeStatement(plpgsqlConvertToTimestamp);
         executeStatement(plpgsqlConvertToIri);
      }
      catch (...) {
         disconnect();
         throw;
      }
   }

   void SQLBackendPostgres::disconnect()
   {
      if (postgresContext->conn)
      {
         PQfinish(postgresContext->conn);
         postgresContext->conn=0;
      }
   }

   void SQLBackendPostgres::executeStatement(const std::string &statement)
   {
      postgresContext->execute(statement, false);
   }

   SQLResultIterator SQLBackendPostgres::executeQuery(const std::string &query)
   {
      PGresult *const result=postgresContext->execute(query, true);

      SQLResultIteratorImpl *impl=new PostgresResultIteratorImpl(*postgresContext, result);
      return SQLResultIterator(impl);
   }

   std::string SQLBackendPostgres::escapeString(const std::string &stringLiteral)
   {
      std::stringstream ss;

      for (std::string::const_iterator i=stringLiteral.begin(); i!=stringLiteral.end(); ++i)
      {
         char c=*i;
         if (c!='\'')
         {
            ss << c;
         }
         else {
            ss << "''";
         }
      }

      return ss.str();
   }

   Transaction SQLBackendPostgres::newTransaction()
   {
      PostgresTransactionImpl *impl=new PostgresTransactionImpl(*postgresContext);
      impl->begin();
      return Transaction(impl);
   }

   const SQLBackendCapabilities &SQLBackendPostgres::getCapabilities()
   {
      return postgresContext->caps;
   }
}
