/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <string>
#include <map>

#include "AQLDebug.h"
#include "AQLModel.h"


namespace TypeRQ {

   class SQLFunctionMap;
   class SQLExpressionWriter;


   // helper template to find last element of the array
   template <typename E>
   const E *getArrayTerminator(const E *start, E terminator=E())
   {
      while (*start != terminator) ++start;
      return start;
   }

   struct SQLFunctionType : protected AQLDebugBase
   {
      AQLTypeSet returnType;
      typedef std::vector<AQLTypeSet> param_type_vec;
      param_type_vec paramTypes;

      SQLFunctionType(AQLTypeSet returnType);
      SQLFunctionType(AQLTypeSet returnType, const std::vector<AQLTypeSet> &paramTypes);
      SQLFunctionType(const AQLFunctionExpr &fexpr);
      SQLFunctionType(const SQLFunctionType &);

      virtual ~SQLFunctionType();

      SQLFunctionType &operator << (AQLTypeSet paramType);
      bool operator == (const SQLFunctionType &other) const;
      bool operator != (const SQLFunctionType &other) const;

      // Returns true if AQLFunctionExpr is applicable to this function type. If exactMatch is
      // true we require exact match, i.e., f(double) does not match for int parameter.
      //
      // exactMatch should be true for type inference purposes, but false when choosing
      // concrete functions after type inference passes.
      bool isApplicableTo(const AQLFunctionExpr &, bool exactMatch) const;

      SQLFunctionType &intersect(const SQLFunctionType &);
   };

   std::ostream &operator << (std::ostream &, const SQLFunctionType &);
   std::string toString(const SQLFunctionType &);


   /**
    * Represents a function mapping.
    */
   class SQLFunctionMapping : protected AQLDebugBase
   {
   protected:
      const std::string sparqlFunctionName;

   public:
      SQLFunctionMapping(const std::string &sparqlFunctionName);
      virtual ~SQLFunctionMapping();

      // the fully qualified function name, such as
      // http://www.w3.org/2005/02/xpath-functions#string-length
      virtual const std::string &getSparqlFunctionName() const;

      // output SQL for AQL function expr
      virtual void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew) = 0;
      virtual void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew) = 0;

      // This is used by type inference pass, aka, the transfer function. 
      virtual void determineFunctionTypeSet(const AQLFunctionExpr &, SQLFunctionType &) = 0;

      virtual void determineVariant(const AQLFunctionExpr &, SQLFunctionType &) = 0;
   };

   class SQLFunctionSelectionMapping : public SQLFunctionMapping
   {
   protected:
      typedef std::vector<SQLFunctionType *> alternative_vec;
      alternative_vec typeAlternatives;

      virtual void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &) = 0;
      virtual void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &) = 0;

   public:
      SQLFunctionSelectionMapping(const std::string &sparqlFn);
      SQLFunctionSelectionMapping(const std::string &sparqlFn, std::vector<SQLFunctionType *> &);
      virtual ~SQLFunctionSelectionMapping();

      virtual void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew);
      virtual void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew);

      virtual void determineFunctionTypeSet(const AQLFunctionExpr &, SQLFunctionType &);

      virtual void determineVariant(const AQLFunctionExpr &, SQLFunctionType &);

      SQLFunctionSelectionMapping & operator <<(SQLFunctionType *);
   };

   class SQLFixedFunctionMapping : public SQLFunctionSelectionMapping
   {
   protected:
      const std::string sqlFn;

      virtual void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &);
      virtual void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &);

   public:
      SQLFixedFunctionMapping(const std::string &sparqlFn, const std::string &sqlFn);
   };

   class SQLNumericFunctionMapping : public SQLFunctionMapping
   {
      const std::string sqlFn;
      const int parameters;
   public:
      SQLNumericFunctionMapping(const std::string &sparqlFn, const std::string &sqlFn, int parameters);

      virtual void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew);
      virtual void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew);

      virtual void determineFunctionTypeSet(const AQLFunctionExpr &, SQLFunctionType &);
      virtual void determineVariant(const AQLFunctionExpr &, SQLFunctionType &);
   };

   class SQLFunctionMap : protected AQLDebugBase
   {
   public:
      SQLFunctionMap();
      virtual ~SQLFunctionMap();

      virtual SQLFunctionMapping &getFunctionMapping(const std::string &functionName) = 0;
   };

   class SQLDelegatingFunctionMap : public SQLFunctionMap
   {
   public:
      // No-fallback version, mappings is a zero-terminated list of function mappings.
      // If mappings does not contain requested function, exception is raised.
      SQLDelegatingFunctionMap(SQLFunctionMapping *const *mappings);

      // Fallback-version, mappings is a zero-terminated list of function mappings.
      // If mappings does not contain requested function, the request is delegated to fallback.
      SQLDelegatingFunctionMap(SQLFunctionMap &fallback, SQLFunctionMapping *const *mappings);

      virtual ~SQLDelegatingFunctionMap();

      virtual SQLFunctionMapping &getFunctionMapping(const std::string &functionName);

   protected:
      SQLFunctionMap *fallback;

      typedef std::map<std::string, SQLFunctionMapping *> mappings_type;
      mappings_type mappings;

   private:
      void insertMappings(SQLFunctionMapping *const *mappings);
   };

   // returns reference to standard SQL function map. Useful as a fallback for default implementations.
   SQLFunctionMap &getStandardSQLFunctionMap();

   SQLFunctionMapping *instantiateTypecastMapping(const char *sparqlFnName,
                                                  const char *sqlFnName,
                                                  AQLTypeSet::ExprType retType);
}
