/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <algorithm>
#include <sstream>

#include "AQLException.h"
#include "SQLBackendFunctions.h"
#include "AQLSupport.h"
#include "ExpressionWriter.h"
#include "Messages.h"

namespace {

   using namespace TypeRQ;


   template<class T> void deleteObject(T *o)
   {
      delete o;
   }

   class ConcatMapping : public SQLFunctionMapping
   {
   public:
      ConcatMapping() :  SQLFunctionMapping("http://www.w3.org/2005/xpath-functions#concat")
      {
         // no-op
      }

      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
      {
         if (expr.arguments.empty())
         {
            ew << "''";
         }
         else if (expr.arguments.size()>=2)
         {
            ew.startConcat();
         }
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
      {
         if (expr.arguments.size()>=2)
         {
            ew.endOp();
         }
      }

      virtual void determineFunctionTypeSet(const AQLFunctionExpr &fexpr, SQLFunctionType &type)
      {
         type.returnType=AQLTypeSet(AQLTypeSet::STRING);
         type.paramTypes.resize(fexpr.arguments.size());
         std::fill(type.paramTypes.begin(), type.paramTypes.end(), AQLTypeSet(AQLTypeSet::ANY));
      }

      virtual void determineVariant(const AQLFunctionExpr &fexpr, SQLFunctionType &type)
      {
         type.returnType=AQLTypeSet(AQLTypeSet::STRING);
         type.paramTypes.resize(fexpr.arguments.size());
         std::fill(type.paramTypes.begin(), type.paramTypes.end(), AQLTypeSet(AQLTypeSet::STRING));
      }
   };

   class CoalesceMapping : public SQLFunctionMapping
   {
   public:
      CoalesceMapping() : SQLFunctionMapping("builtin:coalesce") {}

      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
      {
         ew.startFunction("COALESCE");
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
      {
         ew.endOp();
      }

      virtual void determineFunctionTypeSet(const AQLFunctionExpr &fexpr, SQLFunctionType &type)
      {
         if (fexpr.arguments.empty()) throw AQLException("%s expects at least 1 parameter",
                                                         sparqlFunctionName.c_str());

         // determine return type
         AQLTypeSet paramUnion=AQLTypeSet(AQLTypeSet::UNSET);

         for (AQLFunctionExpr::arg_list_type::const_iterator i=fexpr.arguments.begin();
              i!=fexpr.arguments.end(); ++i)
         {
            const AQLExpr *param=*i;
            paramUnion.setTypes(param->getExprTypeSet());
         }
         type.returnType=makeIntersection(fexpr.functionType, paramUnion);

         // determine parameter types
         type.paramTypes.clear();
         type.paramTypes.reserve(fexpr.arguments.size());
         for (AQLFunctionExpr::arg_list_type::const_iterator i=fexpr.arguments.begin();
              i!=fexpr.arguments.end(); ++i)
         {
            const AQLExpr *param=*i;
            type.paramTypes.push_back(makeIntersection(param->getExprTypeSet(), type.returnType));
         }
      }

      virtual void determineVariant(const AQLFunctionExpr &fexpr, SQLFunctionType &type)
      {
         AQLTypeSet t=fexpr.functionType;
         AQLTypeSet paramMask;

         switch (t.singularType())
         {
            case AQLTypeSet::STRING:
               t=AQLTypeSet::STRING;
               paramMask=AQLTypeSet::STRING;
               break;

            case AQLTypeSet::IRI:
               t=AQLTypeSet::STRING;
               paramMask=AQLTypeSet::STRING;
               break;

            case AQLTypeSet::DOUBLE:
               t=AQLTypeSet::DOUBLE;
               paramMask=AQLTypeSet::DOUBLE;
               paramMask.setType(AQLTypeSet::INTEGER);
               break;

            case AQLTypeSet::INTEGER:
               t=AQLTypeSet::INTEGER;
               paramMask=AQLTypeSet::INTEGER;
               break;

            case AQLTypeSet::BOOLEAN:
               t=AQLTypeSet::BOOLEAN;
               paramMask=AQLTypeSet::BOOLEAN;
               break;

            case AQLTypeSet::DATETIME:
               t=AQLTypeSet::DATETIME;
               paramMask=AQLTypeSet::DATETIME;
               break;

            case AQLTypeSet::ANY:
               t=AQLTypeSet::STRING;
               paramMask=AQLTypeSet::STRING;
               break;

            default:
               throw AQLException("CoalesceMapping: Unhandled return type %s", toString(t).c_str());
         }

         type.returnType=t;
         type.paramTypes.resize(fexpr.arguments.size());
         std::fill(type.paramTypes.begin(), type.paramTypes.end(), paramMask);
      }
   };

   class TypecastMapping : public SQLFunctionSelectionMapping
   {
   protected:
      const std::string sqlTypeName;

      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         ew.startFunction("CAST");
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         ew << " AS " << sqlTypeName;
         ew.endOp();
      }

   public:
      TypecastMapping(const std::string &_sparqlFunctionName,
                      const std::string &_sqlTypeName,
                      AQLTypeSet::ExprType _exprType)
         : SQLFunctionSelectionMapping(_sparqlFunctionName), sqlTypeName(_sqlTypeName)
      {
         *this << &(*new SQLFunctionType(AQLTypeSet(_exprType)) << AQLTypeSet(AQLTypeSet::ANY));
      }
   };

   class DoubleTypecastMapping : public TypecastMapping
   {
   protected:
      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         ew.startPlus();
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         ew.nextTerm();
         ew << "0.0";
         ew.endOp();
      }
   public:
      DoubleTypecastMapping(const std::string &sparqlFn) :
         TypecastMapping(sparqlFn, std::string(), AQLTypeSet::DOUBLE) {}

      ~DoubleTypecastMapping() {}
   };

   class IsNullMapping : public SQLFunctionSelectionMapping
   {
   protected:
      const bool isNull;

      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         if (isNull)
            ew.startUnaryIs(" IS NULL");
         else
            ew.startUnaryIs(" IS NOT NULL");
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         ew.endOp();
      }

   public:
      IsNullMapping(const std::string &sparqlFn, bool _isNull) :
         SQLFunctionSelectionMapping(sparqlFn), isNull(_isNull)
      {
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN)) << AQLTypeSet(AQLTypeSet::ANY));
      }

      void prepare(AQLFunctionExpr &expr, SQLFunctionMap &)
      {
         expr.functionType=AQLTypeSet::BOOLEAN;
      }
   };

   class NumericBinaryOperatorMapping : public SQLFunctionSelectionMapping
   {
   protected:
      void (SQLExpressionWriter::*pstartOp)();

      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         (ew.*pstartOp)();
      }

      void endSQLExp(const AQLFunctionExpr &fexpr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         ew.endOp();
      }
   public:
      NumericBinaryOperatorMapping(const std::string &sparqlFn, void (SQLExpressionWriter::*_pstartOp)()) :
         SQLFunctionSelectionMapping(sparqlFn), pstartOp(_pstartOp)
      {
         // the alternatives are:
         AQLTypeSet doubleAndInt=AQLTypeSet(AQLTypeSet::DOUBLE);
         doubleAndInt.setType(AQLTypeSet::INTEGER);

         // (DOUBLE, DOUBLE) -> DOUBLE, INTEGER
         *this << &(*new SQLFunctionType(doubleAndInt)
                    << AQLTypeSet(AQLTypeSet::DOUBLE) << AQLTypeSet(AQLTypeSet::DOUBLE));
         // (DOUBLE, INTEGER) -> DOUBLE, INTEGER
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::DOUBLE))
                    << AQLTypeSet(AQLTypeSet::DOUBLE) << AQLTypeSet(AQLTypeSet::INTEGER));
         // (INTEGER, DOUBLE) -> DOUBLE, INTEGER
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::DOUBLE))
                    << AQLTypeSet(AQLTypeSet::INTEGER) << AQLTypeSet(AQLTypeSet::DOUBLE));
         // (INTEGER, INTEGER) -> INTEGER
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::INTEGER))
                    << AQLTypeSet(AQLTypeSet::INTEGER) << AQLTypeSet(AQLTypeSet::INTEGER));
      }

   };

   class JunctionMapping : public SQLFunctionMapping
   {
   protected:
      void (SQLExpressionWriter::*pstartOp)();
      const std::string noParamsReplace;
   public:
      JunctionMapping(const std::string &_sparqlFn, void (SQLExpressionWriter::*_pstartOp)(),
                      const std::string &_noParamsReplace) :
         SQLFunctionMapping(_sparqlFn), pstartOp(_pstartOp), noParamsReplace(_noParamsReplace)
      {}

      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
      {
         if (expr.arguments.empty())
         {
            ew << noParamsReplace;
         }
         else if (expr.arguments.size()>=2)
         {
            (ew.*pstartOp)();
         }
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
      {
         if (expr.arguments.size()>=2)
         {
            ew.endOp();
         }
      }

      void determineFunctionTypeSet(const AQLFunctionExpr &fexpr, SQLFunctionType &type)
      {
         type.returnType=AQLTypeSet(AQLTypeSet::BOOLEAN);
         type.paramTypes.resize(fexpr.arguments.size());
         std::fill(type.paramTypes.begin(), type.paramTypes.end(), AQLTypeSet(AQLTypeSet::ANY));
      }      

      void determineVariant(const AQLFunctionExpr &fexpr, SQLFunctionType &type)
      {
         type.returnType=AQLTypeSet(AQLTypeSet::BOOLEAN);
         type.paramTypes.resize(fexpr.arguments.size());
         std::fill(type.paramTypes.begin(), type.paramTypes.end(), AQLTypeSet(AQLTypeSet::BOOLEAN));
      }      
   };

   class NotMapping : public SQLFunctionMapping
   {
   public:
      NotMapping(const std::string &_sparqlFn) : SQLFunctionMapping(_sparqlFn)
      {}

      void prepare(AQLFunctionExpr &expr, SQLFunctionMap &)
      {
         expr.functionType=AQLTypeSet::BOOLEAN;
      }

      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
      {
         ew.startNot();
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
      {
         ew.endOp();
      }

      void determineFunctionTypeSet(const AQLFunctionExpr &fexpr, SQLFunctionType &type)
      {
         type.returnType=AQLTypeSet(AQLTypeSet::BOOLEAN);
         type.paramTypes.resize(1);
         type.paramTypes.at(0)=AQLTypeSet::ANY;
      }

      void determineVariant(const AQLFunctionExpr &fexpr, SQLFunctionType &type)
      {
         type.returnType=AQLTypeSet(AQLTypeSet::BOOLEAN);
         type.paramTypes.resize(1);
         type.paramTypes.at(0)=AQLTypeSet::BOOLEAN;
      }

   };

   class ComparisonMapping : public SQLFunctionSelectionMapping
   {
   protected:
      const std::string op;

      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         if (op=="=")
         {
            ew.startEq();
         }
         else {
            ew.startComparison(op.c_str());
         }
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         ew.endOp();
      }

   public:
      ComparisonMapping(const char *_sparqlFn, const char *_op) :
         SQLFunctionSelectionMapping(_sparqlFn), op(_op)
      {
         // the alternatives are:

         // (STRING, STRING) -> BOOLEAN
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN))
                    << AQLTypeSet(AQLTypeSet::STRING) << AQLTypeSet(AQLTypeSet::STRING));
         // (IRI, IRI) -> BOOLEAN
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN))
                    << AQLTypeSet(AQLTypeSet::IRI) << AQLTypeSet(AQLTypeSet::IRI));

         // and the numeric types:

         // (DOUBLE, DOUBLE) -> BOOLEAN
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN))
                    << AQLTypeSet(AQLTypeSet::DOUBLE) << AQLTypeSet(AQLTypeSet::DOUBLE));
         if (op!="=")
         {
            // (DOUBLE, INTEGER) -> BOOLEAN
            *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN))
                       << AQLTypeSet(AQLTypeSet::DOUBLE) << AQLTypeSet(AQLTypeSet::INTEGER));
            // (INTEGER, DOUBLE) -> BOOLEAN
            *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN))
                       << AQLTypeSet(AQLTypeSet::INTEGER) << AQLTypeSet(AQLTypeSet::DOUBLE));
         }
         // (INTEGER, INTEGER) -> BOOLEAN
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN))
                    << AQLTypeSet(AQLTypeSet::INTEGER) << AQLTypeSet(AQLTypeSet::INTEGER));

         // and the rest
         // (DATETIME, DATETIME) -> BOOLEAN
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN))
                    << AQLTypeSet(AQLTypeSet::DATETIME) << AQLTypeSet(AQLTypeSet::DATETIME));
         // (BOOLEAN, BOOLEAN) -> BOOLEAN
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN))
                    << AQLTypeSet(AQLTypeSet::BOOLEAN) << AQLTypeSet(AQLTypeSet::BOOLEAN));

         // finally, references
         if (op=="=" || op=="!=")
         {
            *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::BOOLEAN))
                       << AQLTypeSet(AQLTypeSet::REFERENCE) << AQLTypeSet(AQLTypeSet::REFERENCE));
         }
      }
   };

   class NegateMapping : public SQLFunctionSelectionMapping
   {
   protected:
      void startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         ew.startUnaryMinus();
      }

      void endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
      {
         ew.endOp();
      }


   public:
      NegateMapping(const std::string &_sparqlFn) :
         SQLFunctionSelectionMapping(_sparqlFn)
      {
         // (DOUBLE) -> DOUBLE
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::DOUBLE))
                    << AQLTypeSet(AQLTypeSet::DOUBLE));
         // (INTEGER) -> INTEGER
         *this << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::INTEGER))
                    << AQLTypeSet(AQLTypeSet::INTEGER));
      }
   };


   SQLFunctionMapping **getStandardBackendFunctionMappings()
   {
      static SQLFunctionMapping *functions[]={

         // math functions
         new SQLNumericFunctionMapping("http://www.w3.org/2005/xpath-functions#abs", "ABS", 1),
         new SQLNumericFunctionMapping("http://www.w3.org/2005/xpath-functions#ceiling", "CEIL", 1),
         new SQLNumericFunctionMapping("http://www.w3.org/2005/xpath-functions#round", "ROUND", 1),
         new SQLNumericFunctionMapping("http://www.w3.org/2005/xpath-functions#floor", "FLOOR", 1),

         // string functions
         &(*new SQLFixedFunctionMapping("http://www.w3.org/2005/xpath-functions#string-length",
                                        "CHAR_LENGTH")
           << &(*new SQLFunctionType(AQLTypeSet(AQLTypeSet::INTEGER)) << AQLTypeSet(AQLTypeSet::STRING))),
         new ConcatMapping(),

         new TypecastMapping("builtin:to-integer", "SIGNED INTEGER", AQLTypeSet::INTEGER),
         new DoubleTypecastMapping("builtin:to-double"),
         new TypecastMapping("builtin:to-iri", "CHAR", AQLTypeSet::IRI),
         new TypecastMapping("builtin:to-string", "CHAR", AQLTypeSet::STRING),

         new IsNullMapping("builtin:is-null", true),
         new IsNullMapping("builtin:is-not-null", false),

         new NumericBinaryOperatorMapping("builtin:add", &SQLExpressionWriter::startPlus),
         new NumericBinaryOperatorMapping("builtin:substract", &SQLExpressionWriter::startMinus),
         new NumericBinaryOperatorMapping("builtin:multiply", &SQLExpressionWriter::startMult),
         new NumericBinaryOperatorMapping("builtin:divide", &SQLExpressionWriter::startDiv),
         new NumericBinaryOperatorMapping("builtin:modulo", &SQLExpressionWriter::startMod),

         new JunctionMapping("builtin:and", &SQLExpressionWriter::startAnd, "TRUE"),
         new JunctionMapping("builtin:or", &SQLExpressionWriter::startOr, "FALSE"),
         new NotMapping("builtin:not"),

         new ComparisonMapping("builtin:comp-lt", "<"),
         new ComparisonMapping("builtin:comp-le", "<="),
         new ComparisonMapping("builtin:comp-eq", "="),
         new ComparisonMapping("builtin:comp-ne", "<>"),
         new ComparisonMapping("builtin:comp-ge", ">="),
         new ComparisonMapping("builtin:comp-gt", ">"),

         new NegateMapping("builtin:negate"),

         new CoalesceMapping(),

         0,
      };
      return functions;
   }

}

namespace TypeRQ {

   std::ostream &operator << (std::ostream &os, const SQLFunctionType &ft)
   {
      os << '[';
      bool first=true;
      for (SQLFunctionType::param_type_vec::const_iterator i=ft.paramTypes.begin();
           i!=ft.paramTypes.end(); ++i)
      {
         if (first) first=false; else os << ' ';
         os << *i;
      }
      os << "] : ";
      os << ft.returnType;
      return os;
   }

   std::string toString(const SQLFunctionType &ft)
   {
      std::stringstream ss;
      ss << ft;
      return ss.str();
   }

   SQLFunctionMapping *instantiateTypecastMapping(const char *sparqlFnName,
                                                  const char *sqlFnName,
                                                  AQLTypeSet::ExprType retType)
   {
      return new TypecastMapping(sparqlFnName, sqlFnName, retType);
   }

   SQLFunctionType::SQLFunctionType(AQLTypeSet _returnType) : returnType(_returnType)
   {}

   SQLFunctionType::SQLFunctionType(AQLTypeSet _returnType, const std::vector<AQLTypeSet> &_paramTypes) :
      returnType(_returnType), paramTypes(_paramTypes)
   {}

   SQLFunctionType::SQLFunctionType(const AQLFunctionExpr &fexpr) : 
      returnType(fexpr.functionType)
   {
      paramTypes.reserve(fexpr.arguments.size());
      for (AQLFunctionExpr::arg_list_type::const_iterator i=fexpr.arguments.begin(); i!=fexpr.arguments.end(); ++i)
      {
         const AQLExpr *arg=*i;
         paramTypes.push_back(arg->getExprTypeSet());
      }
   }

   SQLFunctionType::SQLFunctionType(const SQLFunctionType &o) :
      returnType(o.returnType), paramTypes(o.paramTypes)
   {}

   SQLFunctionType::~SQLFunctionType()
   {
   }

   SQLFunctionType &SQLFunctionType::operator << (AQLTypeSet paramType)
   {
      paramTypes.push_back(paramType);
      return *this;
   }

   bool SQLFunctionType::operator == (const SQLFunctionType &other) const
   {
      if (returnType!=other.returnType) return false;
      if (paramTypes.size()!=other.paramTypes.size()) return false;
      for (size_t i=0; i<paramTypes.size(); ++i)
      {
         if (paramTypes.at(i)!=other.paramTypes.at(i)) return false;
      }
      return true;
   }

   bool SQLFunctionType::operator != (const SQLFunctionType &other) const
   {
      return !(this->operator == (other));
   }

   bool SQLFunctionType::isApplicableTo(const AQLFunctionExpr &expr, bool exactMatch) const
   {
      if (expr.functionType!=AQLTypeSet::UNSET)
      {
         // return type mismatch
         if (makeIntersection(expr.functionType, returnType)==AQLTypeSet(AQLTypeSet::UNSET)) return false;
      }
      if (expr.arguments.size()!=paramTypes.size()) return false;

      size_t j=0;
      for (AQLFunctionExpr::arg_list_type::const_iterator i=expr.arguments.begin(); i!=expr.arguments.end();
           ++i, ++j)
      {
         const AQLExpr &param=**i;
         AQLTypeSet argType=param.getExprTypeSet();
         const AQLTypeSet paramType=paramTypes.at(j);

         if (dynamic_cast<const AQLNullExpr *>(&param)) continue; // null passes always

         if (!exactMatch)
         {
            argType.addSuperTypes();
         }

         if (makeIntersection(argType, paramType)==AQLTypeSet(AQLTypeSet::UNSET)) return false;
      }

      return true;
   }

   SQLFunctionType &SQLFunctionType::intersect(const SQLFunctionType &ft)
   {
      if (ft.paramTypes.size()!=paramTypes.size())
      {
         throw AQLException("SQLFunctionType::intersect: mismatch in param type list sizes");
      }

      returnType=makeIntersection(returnType, ft.returnType);
      for (size_t i=0; i<paramTypes.size(); ++i)
      {
         AQLTypeSet &tset=paramTypes.at(i);
         tset=makeIntersection(tset, ft.paramTypes.at(i));
      }
      return *this;
   }


   SQLFunctionMapping::SQLFunctionMapping(const std::string &_sparqlFn) : sparqlFunctionName(_sparqlFn) {}

   SQLFunctionMapping::~SQLFunctionMapping() {}

   const std::string &SQLFunctionMapping::getSparqlFunctionName() const
   {
      return sparqlFunctionName;
   }

   SQLFunctionSelectionMapping::SQLFunctionSelectionMapping(const std::string &sparqlFn) :
      SQLFunctionMapping(sparqlFn)
   {
   }

   SQLFunctionSelectionMapping::SQLFunctionSelectionMapping(const std::string &sparqlFn,
                                                            std::vector<SQLFunctionType *> &selection) :
      SQLFunctionMapping(sparqlFn), typeAlternatives(selection)
   {
   }

   SQLFunctionSelectionMapping::~SQLFunctionSelectionMapping()
   {
      std::for_each(typeAlternatives.begin(), typeAlternatives.end(), deleteObject<SQLFunctionType>);
   }

   void
   SQLFunctionSelectionMapping::determineVariant(const AQLFunctionExpr &expr, SQLFunctionType &ft)
   {
      SQLFunctionType *bestNonExactMatch=0;
      AQLTypeSet doubleAndInteger=AQLTypeSet(AQLTypeSet::DOUBLE);
      doubleAndInteger.setType(AQLTypeSet::INTEGER);

      if (outputLevel>=OL_DEBUG)
      {
         SQLFunctionType tmp(expr);
         print(OL_DEBUG, "Determining concrete alternative for %s %s...\n",
               expr.functionName.c_str(), toString(tmp).c_str());
      }
      for (alternative_vec::iterator i=typeAlternatives.begin(); i!=typeAlternatives.end(); ++i)
      {
         SQLFunctionType *type=*i;
         if (outputLevel>=OL_DEBUG)
         {
            print(OL_DEBUG, "- testing %s... ", toString(*type).c_str());
         }
         if (type->isApplicableTo(expr, false))
         {
            // check if exact match (applicable can require subtype->supertype conversion)
            bool exactMatch=type->isApplicableTo(expr, true);

            if (exactMatch)
            {
               print(OL_DEBUG, "exact match\n");
               ft=*type;
               return;
            }
            else {
               print(OL_DEBUG, "non-exact match\n");
               if (!bestNonExactMatch) bestNonExactMatch=type;
            }
         }
         else {
            print(OL_DEBUG, "incompatible\n");
         }
      }

      if (bestNonExactMatch)
      {
         ft=*bestNonExactMatch;
         print(OL_DEBUG, "Chosen best non-exact match %s\n", toString(ft).c_str());
         return;
      }
      ft=*typeAlternatives.at(0);
      print(OL_DEBUG, "All incompatible, fallbacking to default variant %s\n", toString(ft).c_str());
   }

   void SQLFunctionSelectionMapping::startSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
   {
      startSQLExp(expr, ew, *expr.chosenVariant);
   }

   void SQLFunctionSelectionMapping::endSQLExp(const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
   {
      endSQLExp(expr, ew, *expr.chosenVariant);
   }

   void SQLFunctionSelectionMapping::determineFunctionTypeSet(const AQLFunctionExpr &expr, SQLFunctionType &type)
   {
      type.returnType.clear();
      type.paramTypes.resize(expr.arguments.size());
      std::fill(type.paramTypes.begin(), type.paramTypes.end(), AQLTypeSet(AQLTypeSet::UNSET));

      // the type set is union of all applicable alternatives
      for (alternative_vec::iterator i=typeAlternatives.begin(); i!=typeAlternatives.end(); ++i)
      {
         SQLFunctionType &alternative=**i;
         if (!alternative.isApplicableTo(expr, true)) continue; // not applicable

         type.returnType.setTypes(alternative.returnType);
         for (size_t j=0; j<alternative.paramTypes.size(); ++j)
         {
            type.paramTypes.at(j).setTypes(alternative.paramTypes.at(j));
         }
      }
   }

   SQLFunctionSelectionMapping &SQLFunctionSelectionMapping::operator <<(SQLFunctionType *type)
   {
      typeAlternatives.push_back(type);

      return *this;
   }

   SQLFixedFunctionMapping::SQLFixedFunctionMapping(const std::string &sparqlFn, const std::string &_sqlFn) :
      SQLFunctionSelectionMapping(sparqlFn), sqlFn(_sqlFn)
   {
   }

   void SQLFixedFunctionMapping::startSQLExp(
      const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
   {
      ew.startFunction(sqlFn);
   }

   void SQLFixedFunctionMapping::endSQLExp(
      const AQLFunctionExpr &expr, SQLExpressionWriter &ew, const SQLFunctionType &)
   {
      ew.endOp();
   }

   SQLNumericFunctionMapping::SQLNumericFunctionMapping(const std::string &_sparqlFn, const std::string &_sqlFn,
                                                        int _parameters) :
      SQLFunctionMapping(_sparqlFn), sqlFn(_sqlFn), parameters(_parameters)
   {
   }

   void SQLNumericFunctionMapping::startSQLExp(
      const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
   {
      ew.startFunction(sqlFn);
   }

   void SQLNumericFunctionMapping::endSQLExp(
      const AQLFunctionExpr &expr, SQLExpressionWriter &ew)
   {
      ew.endOp();
   }

   void SQLNumericFunctionMapping::determineFunctionTypeSet(const AQLFunctionExpr &expr, SQLFunctionType &type)
   {
      AQLTypeSet t=AQLTypeSet::UNSET;
      AQLTypeSet numeric=AQLTypeSet::INTEGER;
      numeric.setType(AQLTypeSet::DOUBLE);

      t.setTypes(type.returnType);
      for (AQLFunctionExpr::arg_list_type::const_iterator i=expr.arguments.begin();
           i!=expr.arguments.end(); ++i)
      {
         const AQLExpr *aqlExpr=*i;
         t.setTypes(aqlExpr->getExprTypeSet());
      }
      t=makeIntersection(t, numeric);

      type.returnType=t;
      type.paramTypes.resize(expr.arguments.size());
      std::fill(type.paramTypes.begin(), type.paramTypes.end(), t);
   }

   void SQLNumericFunctionMapping::determineVariant(const AQLFunctionExpr &expr, SQLFunctionType &type)
   {
      determineFunctionTypeSet(expr, type);
   }


   SQLFunctionMap::SQLFunctionMap() {}

   SQLFunctionMap::~SQLFunctionMap() {}

   SQLDelegatingFunctionMap::SQLDelegatingFunctionMap(SQLFunctionMapping *const *_mappings) : fallback(0)
   {
      insertMappings(_mappings);
   }

   SQLDelegatingFunctionMap::SQLDelegatingFunctionMap(SQLFunctionMap &_fallback,
                                                      SQLFunctionMapping *const *_mappings) : fallback(&_fallback)
   {
      insertMappings(_mappings);
   }

   SQLDelegatingFunctionMap::~SQLDelegatingFunctionMap()
   {
      mappings_type::iterator i;
      for (i=mappings.begin(); i!=mappings.end(); ++i)
      {
         SQLFunctionMapping *mapping=i->second;
         delete mapping;
      }
   }

   void SQLDelegatingFunctionMap::insertMappings(SQLFunctionMapping *const *_mappings)
   {
      while (*_mappings)
      {
         SQLFunctionMapping *mapping=*_mappings;
         mappings.insert(std::make_pair(mapping->getSparqlFunctionName(), mapping));
         ++_mappings;
      }
   }

   SQLFunctionMapping &SQLDelegatingFunctionMap::getFunctionMapping(const std::string &functionName)
   {
      mappings_type::iterator i=mappings.find(functionName);
      if (i!=mappings.end())
      {
         return *(i->second);
      }

      if (!fallback)
      {
         throw AQLException("SQLFunctionMapping: no mapping for %s", functionName.c_str());
      }

      return fallback->getFunctionMapping(functionName);
   }

   SQLFunctionMap &getStandardSQLFunctionMap()
   {
      static SQLDelegatingFunctionMap functionMap(getStandardBackendFunctionMappings());

      return functionMap;
   }
}
