/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <map>
#include <stdarg.h>
#include <vector>
#include <set>

#include "AQLDebug.h"
#include "AQLModel.h"


namespace TypeRQ {
   struct SQLFunctionMap;
}

namespace TypeRQInternal {

   struct SourceReference
   {
   private:
      int row, col;
   public:
      SourceReference();
      SourceReference(const SourceReference &sref);
      SourceReference(int row, int col);
      int getSourceRow() const;
      int getSourceCol() const;
      void setSourceReference(const SourceReference &ref);
   };

   struct Node : public TypeRQ::AQLDebugBase {
      enum Type {
         NONE=0,
         STRING_LITERAL,
         INTEGER,
         DOUBLE,
         VARIABLE,
         BOOLEAN,
         IRI,
      };
      Type type;
      std::string str;
      int64_t i;
      double d;
      bool b;

      Node();
      Node(const Node &);
      ~Node();
   };

   struct TripleOrGraphGroup : TypeRQ::AQLDebugBase
   {
      TripleOrGraphGroup() {}
      virtual ~TripleOrGraphGroup() {}
   };

   struct Triple : TripleOrGraphGroup {
      Node s, p, o;

      Triple();
      Triple(const Triple &);
   };

   struct Expression : public SourceReference {
      enum Type {
         NONE=0,
         NODE,
         EQUAL,
         NOT_EQUAL,
         LESS,
         LESS_OR_EQUAL,
         GREATER,
         GREATER_OR_EQUAL,
         FUNCTION,
         AND,
         OR,
         NOT
      };
      Type type;

      // depending on expression, either node or params is populated
      Node node; // used when type is NODE or FUNCTION
      std::vector<Expression> params; // used when type is anything but NODE

      Expression(const SourceReference &sref) : SourceReference(sref), type(NONE) {}
      Expression(const Expression &exp) : SourceReference(exp), type(exp.type), node(exp.node), params(exp.params) {}
   };

   struct GraphGroup : TripleOrGraphGroup {
      bool isOptional;

      std::list<TripleOrGraphGroup *> triplesAndGraphGroups; 
      std::list<Expression> filters;

      // populated by putParentLinks()
      GraphGroup *parentGraphGroup;

      GraphGroup() : isOptional(false), parentGraphGroup(0) {}

      ~GraphGroup() {
         for (std::list<TripleOrGraphGroup *>::iterator i=triplesAndGraphGroups.begin();
              i!=triplesAndGraphGroups.end(); ++i) {
            TripleOrGraphGroup *t=*i;
            delete t;
         }
      }
   };

   struct OrderCondition {
      bool ascending;
      Expression exp;

      OrderCondition() : ascending(true), exp(SourceReference()) {}
   };

   struct SparqlVariableTarget
   {
      std::string triple;
      TypeRQ::AQLPropertyExpr::Property property;
      SparqlVariableTarget() {}
      SparqlVariableTarget(const std::string &_t, TypeRQ::AQLPropertyExpr::Property _p)
         : triple(_t), property(_p) {}
   };

   struct SparqlVariableDesc : protected TypeRQ::AQLDebugBase
   {
      typedef std::list<SparqlVariableTarget> target_list;

      /*
       * Note: Usually there is 1 element in the target list.
       * However, in case of parallel OPTIONAL graph groups introducing the variable,
       * the common parent has both property expressions in the target list. This means
       * that coalesce expression is produced in createAQLAccessExpr() instead of typical
       * AQLPropertyExpr.
       */
      target_list aqlTargets;
      bool authoritative;
      TypeRQ::AQLExpr *createAQLAccessExpr() const;

      SparqlVariableDesc(const std::string &triple, TypeRQ::AQLPropertyExpr::Property p, bool authoritative);
      SparqlVariableDesc(const SparqlVariableDesc &);

      virtual ~SparqlVariableDesc();
   };

   struct VariableScope : protected TypeRQ::AQLDebugBase
   {
      typedef std::map<std::string, SparqlVariableDesc> map_type;
      map_type variables;

      VariableScope();
      virtual ~VariableScope();
   private:
      VariableScope(const VariableScope &);
   };

   class ScopedVariableBindings
   {
   protected:
      typedef std::map<const GraphGroup *, VariableScope *> map_type;
      map_type scopeMap;
   public:


      ScopedVariableBindings();
      virtual ~ScopedVariableBindings();

      VariableScope &getVariableScope(const GraphGroup &) const;

      // Returns true if scope or its parent (recursive) has an authoritative definition for variable vname,
      // i.e., definition that cannot be null in the scope
      bool hasAuthoritativeDefinition(const GraphGroup &scope, const std::string &vname) const;

      // returns true if vname is defined in the scope (recursively or not)
      bool isDefined(const GraphGroup &scope, const std::string &vname, bool recurse) const;

      void defineVariableScope(const GraphGroup &);
      void defineVariable(const GraphGroup &scope, const std::string &vname,
                          const std::string &triple, TypeRQ::AQLPropertyExpr::Property p,
                          bool authoritative);

      const SparqlVariableDesc &getVariableDesc(const GraphGroup &scope, const std::string &vname);
      TypeRQ::AQLExpr *createVariableAccessExpr(const GraphGroup &scope, const std::string &vname);
   };

   // The intermediate query object for parsing. This will
   // be used to create the final query object with AQL and all
   struct SPARQLIntermediateQuery : protected TypeRQ::AQLDebugBase {

      /*** THE FOLLOWING ARE POPULATED BY THE PARSER ***/

      std::list<std::string> selects;
      GraphGroup rootGraphGroup;
      int limit, offset;
      bool distinct;

      typedef std::list<OrderCondition> OrderList;
      OrderList orders;

      TypeRQ::SQLFunctionMap &functionMap;

      // this is the "current" graph group
      GraphGroup *currentGraphGroup;

      /*** THE FOLLOWING ARE POPULATED BY VARIOUS FUNCTIONS IN SPARQLToAQL.cpp ***/

      // populated by createAQLJoinGroups()
      typedef std::map<GraphGroup *, TypeRQ::AQLJoinGroupLike *> GraphGroupJoinMap;
      GraphGroupJoinMap graphGroupJoins;

      typedef std::map<Triple *, std::string> TripleJoinNameMap;
      TripleJoinNameMap tripleJoinNames;

      // populated by createVariableBindings()
      ScopedVariableBindings variableBindings;


      SPARQLIntermediateQuery(TypeRQ::SQLFunctionMap &_functionMap) : limit(-1), offset(-1), distinct(false),
                                                                      functionMap(_functionMap),
                                                                      currentGraphGroup(0)
      {}
   };


   void printExpression(const Expression &exp);
   void printGraphGroup(SPARQLIntermediateQuery &iq, GraphGroup &gg, int level, bool checkParentLinks);

   // translates intermediate query to AQL
   TypeRQ::AQLQuery *translateToAQL(SPARQLIntermediateQuery &iq, bool doBottomUpVariableChecking);

}
