/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdexcept>
#include <stdlib.h>
#include <stdio.h>
#include <sstream>

#include "AQLException.h"
#include "SPARQLToAQL.h"
#include "AQLSupport.h"
#include "SQLBackendFunctions.h"
#include "Messages.h"
#include "StringUtils.h"

namespace {
   using namespace TypeRQInternal;
   using namespace TypeRQ;

   AQLExpr *expressionToAQLExpr(SPARQLIntermediateQuery &iq, GraphGroup &gg, const Expression &exp);


   void translationError(const SourceReference &sref, const char *fmt ...)
   {
      va_list ap;
      va_start(ap, fmt);

      std::string reason=formatToString("Line %d Column %d: ", sref.getSourceRow(), sref.getSourceCol());
      reason+=formatToString(fmt, ap);

      va_end(ap);

      throw AQLException(reason);
   }

   void printIndent(int level) {
      for (int i=0; i<level; ++i) {
         printf("  ");
      }
   }

   void printNode(Node &n) {
      switch (n.type) {
         case Node::STRING_LITERAL:
            printf("STRING_LITERAL(%s)", n.str.c_str());
            break;
         case Node::IRI:
            printf("IRI(%s)", n.str.c_str());
            break;
         case Node::INTEGER:
            printf("INTEGER(%lld)", static_cast<long long>(n.i));
            break;
         case Node::DOUBLE:
            printf("DOUBLE(%f)", n.d);
            break;
         case Node::VARIABLE:
            printf("VARIABLE(%s)", n.str.c_str());
            break;
         case Node::BOOLEAN:
            printf("BOOLEAN(%d)", static_cast<int>(n.b));
            break;
         default:
            // prevent interpreting ??) as a tri-graph
            printf("UnknownType%d(?" "?)", static_cast<int>(n.type));
      }
   }

   void possiblyAddKnownVariable(std::set<std::string> &knownVariables, const Node &n)
   {
      if (n.type==Node::VARIABLE)
      {
         knownVariables.insert(n.str);
      }
   }

   void checkVariableAccesses(const std::set<std::string> &knownVariables, const Expression &exp)
   {
      if (exp.type==Expression::NODE && exp.node.type==Node::VARIABLE)
      {
         if (knownVariables.find(exp.node.str)==knownVariables.end())
         {
            translationError(exp, "Variable %s is unknown or cannot be used at this context. Maybe try left-to-right variable binding?", exp.node.str.c_str());
         }
      }
      for (std::vector<Expression>::const_iterator i=exp.params.begin(); i!=exp.params.end(); ++i) {
         checkVariableAccesses(knownVariables, *i);
      }
   }

   std::set<std::string> checkBottomUpVariableAccesses(GraphGroup &gg, int level)
   {
      ++level;

      std::set<std::string> knownVariables;
      

      for (std::list<TripleOrGraphGroup *>::iterator i=gg.triplesAndGraphGroups.begin();
           i!=gg.triplesAndGraphGroups.end(); ++i)
      {
         GraphGroup *const nested=dynamic_cast<GraphGroup *>(*i);
         if (nested)
         {
            const std::set<std::string> ret=checkBottomUpVariableAccesses(*nested, level);
            knownVariables.insert(ret.begin(), ret.end());
         }
      }      

      print(OL_DEBUG, "- level %d: adding variables of scope %p...\n", level, &gg);
      for (std::list<TripleOrGraphGroup *>::iterator i=gg.triplesAndGraphGroups.begin();
           i!=gg.triplesAndGraphGroups.end(); ++i)
      {
         Triple *t=dynamic_cast<Triple *>(*i);
         if (t)
         {
            possiblyAddKnownVariable(knownVariables, t->s);
            possiblyAddKnownVariable(knownVariables, t->p);
            possiblyAddKnownVariable(knownVariables, t->o);
         }
      }      

      print(OL_DEBUG, "- level %d: checking filters of scope %p...\n", level, &gg);
      for (std::list<Expression>::iterator i=gg.filters.begin(); i!=gg.filters.end(); ++i) {
         checkVariableAccesses(knownVariables, *i);
      }

      --level;
      return knownVariables;
   }

   void checkBottomUpVariableAccesses(GraphGroup &gg)
   {
      checkBottomUpVariableAccesses(gg, 0);
   }

   void putParentLinks(GraphGroup &gg) {
      for (std::list<TripleOrGraphGroup *>::iterator i=gg.triplesAndGraphGroups.begin();
           i!=gg.triplesAndGraphGroups.end(); ++i)
      {
         GraphGroup *nested=dynamic_cast<GraphGroup *>(*i);
         if (nested)
         {
            nested->parentGraphGroup=&gg;
            putParentLinks(*nested);
         }
      }
   }

   void mergeNestedMandatoryGraphsWithParents(GraphGroup &gg) {
      for (std::list<TripleOrGraphGroup *>::iterator i=gg.triplesAndGraphGroups.begin();
           i!=gg.triplesAndGraphGroups.end(); )
      {
         GraphGroup *const nested=dynamic_cast<GraphGroup *>(*i);
         if (!nested)
         {
            // not a graph group
            ++i;
            continue;
         }

         if (nested->isOptional)
         {
            // don't merge, but recurse only
            mergeNestedMandatoryGraphsWithParents(*nested);
            ++i;
         }
         else {
            // merge
            std::list<TripleOrGraphGroup *>::iterator j=i;
            ++j;
            gg.filters.insert(gg.filters.end(), nested->filters.begin(), nested->filters.end());
            nested->filters.clear();

            gg.triplesAndGraphGroups.insert(j,
                                            nested->triplesAndGraphGroups.begin(),
                                            nested->triplesAndGraphGroups.end());

            gg.triplesAndGraphGroups.erase(i++);

            // delete nested, but prevent deleting its children
            nested->triplesAndGraphGroups.clear();
            delete nested;
            ++i;
         }
      }
   }

   void createAQLJoinGroups(SPARQLIntermediateQuery &iq, GraphGroup &gg, AQLJoinGroupLike &join) {
      int tripleNum=0; // triple num within this join

      // join for GraphGroup
      iq.graphGroupJoins[&gg]=&join;
      const size_t graphGroupNum=iq.graphGroupJoins.size();

      for (std::list<TripleOrGraphGroup *>::iterator i=
              gg.triplesAndGraphGroups.begin(); i!=gg.triplesAndGraphGroups.end(); ++i) {

         Triple *const t=dynamic_cast<Triple *>(*i);
         if (t)
         {
            ++tripleNum;
            std::stringstream ss;

            ss << "triple_";
            ss << graphGroupNum;
            ss << '_';
            ss << tripleNum;

            join.names.push_back(ss.str());
            iq.tripleJoinNames[t]=ss.str();
         }

         GraphGroup *const nested=dynamic_cast<GraphGroup *>(*i);
         if (nested)
         {
            AQLJoinGroup *const joinForNested=new AQLJoinGroup;
            joinForNested->joinType=(nested->isOptional? AQLJoinGroup::LEFT_OUTER : AQLJoinGroup::INNER);

            createAQLJoinGroups(iq, *nested, *joinForNested);
            join.nestedJoins.push_back(joinForNested);
         }
      }
   }

   void addSelects(SPARQLIntermediateQuery &iq, AQLQuery &aql) {
      for (std::list<std::string>::const_iterator i=iq.selects.begin(); i!=iq.selects.end(); ++i) {
         const std::string &var=*i;
         AQLSelect *aqlSelect=new AQLSelect;
         AQLExpr *propertyExpr=0;
         try {
            propertyExpr=iq.variableBindings.createVariableAccessExpr(iq.rootGraphGroup, var);
         } catch (std::out_of_range) {
            delete aqlSelect;
            throw AQLException("Cannot select unbound variable '%s'", var.c_str());
         }
         aqlSelect->label=var;
         aqlSelect->expr=propertyExpr;

         aql.selects.push_back(aqlSelect);

      }
   }

   void addConditionForProperty(SPARQLIntermediateQuery &iq, GraphGroup &gg, Triple *t, AQLPropertyExpr::Property p) {
      const Node *node;
      switch (p) {
         case AQLPropertyExpr::SUBJECT:   node=&t->s; break;
         case AQLPropertyExpr::PREDICATE: node=&t->p; break;
         case AQLPropertyExpr::OBJECT:    node=&t->o; break;
         default:
            throw AQLException("addConditionForProperty: Internal error, aql property code (%d)", (int)p);
      }

      AQLJoinGroupLike *join=iq.graphGroupJoins.at(&gg);
      const std::string &tripleJoinName=iq.tripleJoinNames.at(t);

      AQLExpr *condition=0;

      switch (node->type) {
         case Node::STRING_LITERAL:
         {
            AQLPropertyExpr *left=new AQLPropertyExpr(tripleJoinName, p, AQLTypeSet(AQLTypeSet(AQLTypeSet::ANY)));
            AQLLiteralExpr *right=new AQLLiteralExpr(node->str, AQLTypeSet::STRING);
            condition=new AQLComparisonCriterion(left, right, AQLComparisonCriterion::EQUAL);
            break;
         }
         case Node::IRI:
         {
            AQLPropertyExpr *left=new AQLPropertyExpr(tripleJoinName, p, AQLTypeSet(AQLTypeSet(AQLTypeSet::ANY)));
            AQLLiteralExpr *right=new AQLLiteralExpr(node->str, AQLTypeSet::IRI);
            condition=new AQLComparisonCriterion(left, right, AQLComparisonCriterion::EQUAL);
            break;
         }
         case Node::VARIABLE: {
            if (iq.variableBindings.isDefined(gg, node->str, true))
            {
               print(OL_DEBUG, "- Variable %s already defined, adding constraint...\n",
                     node->str.c_str());

               // variable already defined ==> add constraint
               const SparqlVariableDesc &svd=iq.variableBindings.getVariableDesc(gg, node->str);

               AQLPropertyExpr *left=new AQLPropertyExpr(tripleJoinName, p,
                                                         AQLTypeSet(AQLTypeSet(AQLTypeSet::ANY)));
               AQLExpr *right=svd.createAQLAccessExpr();
               condition=new AQLComparisonCriterion(left, right, AQLComparisonCriterion::EQUAL);

               if (!iq.variableBindings.hasAuthoritativeDefinition(gg, node->str))
               {
                  print(OL_DEBUG, "- Variable %s can be null...\n",
                        node->str.c_str());

                  AQLFunctionExpr *rightIsNull=new AQLFunctionExpr;
                  rightIsNull->functionName="builtin:is-null";
                  rightIsNull->arguments.push_back(svd.createAQLAccessExpr());

                  AQLJunctionCriterion *disjunction=new AQLJunctionCriterion;
                  disjunction->junctionType=AQLJunctionCriterion::DISJUNCTION;
                  disjunction->terms.push_back(rightIsNull);
                  disjunction->terms.push_back(condition);

                  condition=disjunction;
               }
               else {
                  print(OL_DEBUG, "- Variable %s can not be null...\n",
                        node->str.c_str());
               }
            }
            else {
               print(OL_DEBUG, "- Variable %s not defined, skipping constraint...\n",
                     node->str.c_str());
            }

            if (!iq.variableBindings.hasAuthoritativeDefinition(gg, node->str) ||
                !iq.variableBindings.isDefined(gg, node->str, false)) {

               print(OL_DEBUG, "- Defining variable %s as %s:%s\n",
                     node->str.c_str(),
                     tripleJoinName.c_str(), getNameForAQLTripleProperty(p));

               // not defined yet in this scope, (re)define
               iq.variableBindings.defineVariable(gg, node->str, tripleJoinName, p, true);
            }

            break;
         }
         default:
            throw AQLException("addConditionForProperty: Internal error, unhandled node type (%d)", (int)node->type);
      }

      if (condition)
         join->addCriterion(condition);

   }

   void createVariableBindingsAndAddJoinConditions(SPARQLIntermediateQuery &iq, GraphGroup &gg, AQLQuery &aql)
   {
      print(OL_DEBUG, "Considering graph group...\n");
      iq.variableBindings.defineVariableScope(gg);
      for (std::list<TripleOrGraphGroup *>::iterator i=gg.triplesAndGraphGroups.begin();
           i!=gg.triplesAndGraphGroups.end(); ++i)
      {
         Triple *t=dynamic_cast<Triple *>(*i);
         if (t)
         {
            addConditionForProperty(iq, gg, t, AQLPropertyExpr::SUBJECT);
            addConditionForProperty(iq, gg, t, AQLPropertyExpr::PREDICATE);
            addConditionForProperty(iq, gg, t, AQLPropertyExpr::OBJECT);
         }

         GraphGroup *const nested=dynamic_cast<GraphGroup *>(*i);
         if (nested)
         {
            createVariableBindingsAndAddJoinConditions(iq, *nested, aql);
         }
      }      
      print(OL_DEBUG, "Graph group complete\n");
   }

   AQLFunctionExpr *translateFunction(SPARQLIntermediateQuery &iq,
                                      GraphGroup &gg, const std::string &name,
                                      const std::vector<Expression> &params) {

      AQLFunctionExpr *f=new AQLFunctionExpr;
      try {
         f->functionName=name;
         for (size_t i=0; i<params.size(); ++i)
         {
            f->arguments.push_back(expressionToAQLExpr(iq, gg, params.at(i)));
         }
         return f;
      }
      catch (...) {
         delete f;
         throw;
      }
   }

   AQLExpr *expressionToAQLExpr(SPARQLIntermediateQuery &iq, GraphGroup &gg, const Expression &exp) {
      switch (exp.type) {
         case Expression::NODE:
            switch (exp.node.type) {
               case Node::STRING_LITERAL: {
                  return new AQLLiteralExpr(exp.node.str, AQLTypeSet::STRING);
               }

               case Node::IRI: {
                  return new AQLLiteralExpr(exp.node.str, AQLTypeSet::IRI);
               }

               case Node::INTEGER: {
                  return new AQLLiteralExpr(exp.node.i);
               }

               case Node::DOUBLE: {
                  return new AQLLiteralExpr(exp.node.d);
               }

               case Node::BOOLEAN: {
                  return new AQLLiteralExpr(exp.node.b);
               }

               case Node::VARIABLE: {
                  try {
                     AQLExpr *e=iq.variableBindings.createVariableAccessExpr(gg, exp.node.str);
                     return e;
                  } catch (std::out_of_range) {
                     throw AQLException("Cannot use unbound variable '%s' in expression", exp.node.str.c_str());
                  }
               }

               default:
                  throw AQLException("expressionToAQLExpr: Internal error, unhandled node type (%d)", (int)exp.node.type);
            }

         case Expression::EQUAL:
         case Expression::NOT_EQUAL:
         case Expression::LESS:
         case Expression::LESS_OR_EQUAL:
         case Expression::GREATER_OR_EQUAL:
         case Expression::GREATER: {
            AQLComparisonCriterion *e=new AQLComparisonCriterion;
            switch (exp.type) {
               case Expression::EQUAL:            e->comparisonType=AQLComparisonCriterion::EQUAL; break;
               case Expression::NOT_EQUAL:        e->comparisonType=AQLComparisonCriterion::NOT_EQUAL; break;
               case Expression::LESS:             e->comparisonType=AQLComparisonCriterion::LESS; break;
               case Expression::LESS_OR_EQUAL:    e->comparisonType=AQLComparisonCriterion::LESS_OR_EQUAL; break;
               case Expression::GREATER_OR_EQUAL: e->comparisonType=AQLComparisonCriterion::GREATER_OR_EQUAL; break;
               case Expression::GREATER:          e->comparisonType=AQLComparisonCriterion::GREATER; break;
               default:
                  throw AQLException("expressionToAQLExpr(): Internal error, unhandled expression type %d",
                                     static_cast<int>(exp.type));
            }
            try {
               e->left=expressionToAQLExpr(iq, gg, exp.params.at(0));
               e->right=expressionToAQLExpr(iq, gg, exp.params.at(1));
            }
            catch (...) {
               delete e;
               throw;
            }
            return e;
         }

         case Expression::FUNCTION: {
            AQLFunctionExpr *fexpr=translateFunction(iq, gg, exp.node.str, exp.params);
            return fexpr;
         }

         case Expression::AND:
         case Expression::OR: {
            AQLJunctionCriterion *junction=new AQLJunctionCriterion;
            try {
               junction->junctionType=
                  exp.type==Expression::AND?
                  AQLJunctionCriterion::CONJUNCTION : AQLJunctionCriterion::DISJUNCTION;

               junction->terms.push_back(expressionToAQLExpr(iq, gg, exp.params.at(0)));
               junction->terms.push_back(expressionToAQLExpr(iq, gg, exp.params.at(1)));
               return junction;
            }
            catch (...) {
               delete junction;
               throw;
            }
         }

         case Expression::NOT: {
            AQLExpr *expr=expressionToAQLExpr(iq, gg, exp.params.at(0));
            AQLNotExpression *nexpr=new AQLNotExpression;
            nexpr->expr=expr;
            return nexpr;
         }

         default:
            throw AQLException("expressionToAQLExpr: Internal error, unhandled expression type (%d)",
                               (int)exp.type);
      }

   }

   void addFilterConditions(SPARQLIntermediateQuery &iq, GraphGroup &gg, AQLQuery &aql)
   {
      AQLJoinGroupLike *const join=iq.graphGroupJoins.at(&gg);

      for (std::list<Expression>::iterator i=gg.filters.begin(); i!=gg.filters.end(); ++i) {
         const Expression &exp=*i;

         AQLExpr *aqlExpr=expressionToAQLExpr(iq, gg, exp);
         join->addCriterion(aqlExpr);
      }

      for (std::list<TripleOrGraphGroup *>::iterator i=gg.triplesAndGraphGroups.begin();
           i!=gg.triplesAndGraphGroups.end();
           ++i)
      {
         GraphGroup *nested=dynamic_cast<GraphGroup *>(*i);
         if (nested)
         {
            addFilterConditions(iq, *nested, aql);
         }
      }      
   }

   void addOrderBys(SPARQLIntermediateQuery &iq, AQLQuery &aql) {
      for (SPARQLIntermediateQuery::OrderList::const_iterator i=iq.orders.begin(); i!=iq.orders.end(); ++i) {
         const OrderCondition &oc=*i;
         AQLSort *aqlSort=new AQLSort;
         aql.sorts.push_back(aqlSort);
         aqlSort->ascending=oc.ascending;
         aqlSort->expr=expressionToAQLExpr(iq, iq.rootGraphGroup, oc.exp);
      }
   }
}


namespace TypeRQInternal {

   SourceReference::SourceReference() : row(-1), col(-1) {}

   SourceReference::SourceReference(int _row, int _col) : row(_row), col(_col) {}

   SourceReference::SourceReference(const SourceReference &sref) : row(sref.row), col(sref.col) {}

   int SourceReference::getSourceRow() const
   {
      return row;
   }

   int SourceReference::getSourceCol() const
   {
      return col;
   }

   void SourceReference::setSourceReference(const SourceReference &sref)
   {
      row=sref.row;
      col=sref.col;
   }


   Node::Node() : type(NONE) {}
   Node::Node(const Node &n) : type(n.type), str(n.str), i(n.i), d(n.d), b(n.b) {}
   Node::~Node() {
      type=NONE;
   }

   Triple::Triple() {}
   Triple::Triple(const Triple &t) : s(t.s), p(t.p), o(t.o) {}


   void printExpression(const Expression &exp) {
      printf("[%d,%d]", exp.getSourceRow(), exp.getSourceCol());
      if (exp.type!=Expression::NODE) {
         std::string n;
         switch (exp.type) {
            case Expression::EQUAL:           n="="; break;
            case Expression::NOT_EQUAL:       n="!="; break;
            case Expression::LESS:            n="<"; break;
            case Expression::LESS_OR_EQUAL:   n="<="; break;
            case Expression::GREATER_OR_EQUAL:n=">="; break;
            case Expression::GREATER:         n=">"; break;
            case Expression::FUNCTION:        n="function '"; n+=exp.node.str; n+='\''; break;
            case Expression::AND:             n="AND"; break;
            case Expression::OR:              n="OR"; break;
            case Expression::NOT:             n="NOT"; break;
            default:                          n="UNKNOWN"; break;
         }
         printf("(%s(%d)", n.c_str(), (int)exp.type);
         for (std::vector<Expression>::const_iterator i=exp.params.begin(); i!=exp.params.end(); ++i) {
            printf(" ");
            printExpression(*i);
         }
         printf(")");
      } else {
         // NODE expression
         switch (exp.node.type) {
            case Node::STRING_LITERAL: printf("\"%s\"", exp.node.str.c_str()); break;
            case Node::INTEGER:        printf("%lld", static_cast<long long>(exp.node.i)); break;
            case Node::DOUBLE:         printf("%f", exp.node.d); break;
            case Node::VARIABLE:       printf("?%s", exp.node.str.c_str()); break;
            case Node::BOOLEAN:        printf("%s", exp.node.b? "true":"false"); break;
            case Node::IRI:            printf("<%s>", exp.node.str.c_str()); break;
            default:
               printf("???");
         }
      }
   }

   void printGraphGroup(SPARQLIntermediateQuery &iq, GraphGroup &gg, int level, bool checkParentLinks) {
      printIndent(level);
      printf("GRAPH GROUP");
      if (gg.isOptional) printf(" OPTIONAL");
      printf(" {\n");
      ++level;

      for (std::list<TripleOrGraphGroup *>::iterator i=gg.triplesAndGraphGroups.begin();
           i!=gg.triplesAndGraphGroups.end();
           ++i)
      {
         Triple *t=dynamic_cast<Triple *>(*i);
         if (t)
         {
            printIndent(level);
            printf("Triple(");
            printNode(t->s);
            printf(", ");
            printNode(t->p);
            printf(", ");
            printNode(t->o);
            printf(")");

            SPARQLIntermediateQuery::TripleJoinNameMap::iterator j=iq.tripleJoinNames.find(t);
            if (j!=iq.tripleJoinNames.end())
            {
               printf(" bound to join %s", j->second.c_str());
            }
            else {
               printf(" is unbound");
            }
            printf("\n");
         }


         GraphGroup *nested=dynamic_cast<GraphGroup *>(*i);
         if (nested)
         {
            if (checkParentLinks && nested->parentGraphGroup!=&gg)
            {
               printIndent(level);
               printf("WARNING!!! Parent-link of the following GRAPH GROUP is bad or missing!\n");
            }
            printGraphGroup(iq, *nested, level, checkParentLinks);
         }
      }

      for (std::list<Expression>::iterator i=gg.filters.begin(); i!=gg.filters.end(); ++i) {
         printIndent(level);
         printf("Filter: ");
         printExpression(*i);
         printf("\n");
      }

      --level;
      printIndent(level);
      printf("}\n");
   }

   void dumpVariableBindings(GraphGroup &gg, ScopedVariableBindings &svb, int indentLevel)
   {
      if (indentLevel>0) printf("\n");
      printIndent(indentLevel);
      printf("SCOPE { (%p)\n", &gg);

      VariableScope &vscope=svb.getVariableScope(gg);
      for (VariableScope::map_type::iterator i=vscope.variables.begin();
           i!=vscope.variables.end(); ++i)
      {
         const std::string &vname=i->first;
         const SparqlVariableDesc &svd=i->second;
         printIndent(indentLevel+1);
         printf("Var %s: aql targets={", vname.c_str());
         for (SparqlVariableDesc::target_list::const_iterator j=svd.aqlTargets.begin();
              j!=svd.aqlTargets.end(); ++j)
         {
            const SparqlVariableTarget &svt=*j;
            printf(" %s:%s", svt.triple.c_str(), TypeRQ::getNameForAQLTripleProperty(svt.property));
         }
         printf(" }\n");
      }

      for (std::list<TripleOrGraphGroup *>::iterator i=gg.triplesAndGraphGroups.begin();
           i!=gg.triplesAndGraphGroups.end();
           ++i)
      {
         GraphGroup *const nested=dynamic_cast<GraphGroup *>(*i);
         if (nested)
         {
            dumpVariableBindings(*nested, svb, indentLevel+1);
         }
      }

      printIndent(indentLevel);
      printf("}\n");
   }

   AQLQuery *translateToAQL(SPARQLIntermediateQuery &iq, bool doBottomUpVariableChecking)
   {
      AQLQuery *aql=0;

      try {
         if (doBottomUpVariableChecking)
         {
            print(OL_DEBUG, "checkBottomUpVariableAccesses()\n");
            checkBottomUpVariableAccesses(iq.rootGraphGroup);            
         }

         // simplify graph group structure
         print(OL_DEBUG, "mergeNestedMandatoryGraphsWithParents()\n");
         mergeNestedMandatoryGraphsWithParents(iq.rootGraphGroup);
         printGraphGroup(iq, iq.rootGraphGroup, 0, false);

         // set parent links between graph groups
         print(OL_DEBUG, "putParentLinks()\n");
         putParentLinks(iq.rootGraphGroup);

         // create AQL

         aql=new AQLQuery();

         // DISTINCT
         aql->distinct=iq.distinct;

         // ROW LIMIT
         if (iq.limit>=0) {
            aql->maxRows=iq.limit;
         }

         // ROW OFFSET
         if (iq.offset>=0) {
            aql->rowOffset=iq.offset;
         }

         print(OL_DEBUG, "createAQLJoinGroups()\n");
         createAQLJoinGroups(iq, iq.rootGraphGroup, *aql);
         print(OL_DEBUG, "createVariableBindingsAndAddJoinConditions()\n");
         createVariableBindingsAndAddJoinConditions(iq, iq.rootGraphGroup, *aql);

         // variables are bound, now we can add selects to the AQL
         print(OL_DEBUG, "addSelects()\n");
         addSelects(iq, *aql);

         print(OL_DEBUG, "addFilterConditions()\n");
         addFilterConditions(iq, iq.rootGraphGroup, *aql);

         print(OL_DEBUG, "addOrderBys()\n");
         addOrderBys(iq, *aql);


         if (outputLevel>=OL_DEBUG) {
            printf("Variable bindings:\n");
            if (outputLevel>=OL_DEBUG)
            {
               dumpVariableBindings(iq.rootGraphGroup, iq.variableBindings, 0);
            }
         }

         return aql;
      } catch (...) {
         delete aql;
         throw;
      }
   }

   SparqlVariableDesc::SparqlVariableDesc(const std::string &_triple,
                                          TypeRQ::AQLPropertyExpr::Property _p,
                                          bool _authoritative) : authoritative(_authoritative)
   {
      aqlTargets.push_back(SparqlVariableTarget(_triple, _p));
   }

   SparqlVariableDesc::SparqlVariableDesc(const SparqlVariableDesc &svd) :
      aqlTargets(svd.aqlTargets), authoritative(svd.authoritative) {}

   SparqlVariableDesc::~SparqlVariableDesc() {}

   TypeRQ::AQLExpr *SparqlVariableDesc::createAQLAccessExpr() const
   {
      if (aqlTargets.empty())
         throw AQLException("SparqlVariableDesc::createAQLAccessExpr(): somehow target list is empty");

      if (aqlTargets.size()==1)
      {
         return new AQLPropertyExpr(aqlTargets.front().triple, aqlTargets.front().property, AQLTypeSet::ANY);
      }

      AQLFunctionExpr *fexpr=new AQLFunctionExpr;

      fexpr->functionName="builtin:coalesce";

      for (target_list::const_iterator i=aqlTargets.begin(); i!=aqlTargets.end(); ++i)
      {
         fexpr->arguments.push_back(new AQLPropertyExpr(i->triple, i->property, AQLTypeSet::ANY));
      }

      return fexpr;
   }

   VariableScope::VariableScope()
   {
   }

   VariableScope::~VariableScope()
   {
   }

   ScopedVariableBindings::ScopedVariableBindings()
   {
   }

   ScopedVariableBindings::~ScopedVariableBindings()
   {
      for (map_type::iterator i=scopeMap.begin(); i!=scopeMap.end(); ++i)
      {
         VariableScope *vscope=i->second;
         delete vscope;
      }
   }

   VariableScope &ScopedVariableBindings::getVariableScope(const GraphGroup &gg) const
   {
      return *scopeMap.at(&gg);
   }

   const SparqlVariableDesc &
   ScopedVariableBindings::getVariableDesc(const GraphGroup &scope, const std::string &vname)
   {
      const GraphGroup *i=&scope;
      while (i->parentGraphGroup)
      {
         VariableScope &vscope=*scopeMap.at(i);
         VariableScope::map_type::iterator j=vscope.variables.find(vname);
         if (j!=vscope.variables.end())
            return j->second;

         i=i->parentGraphGroup;
      }
      return scopeMap.at(i)->variables.at(vname);
   }

   TypeRQ::AQLExpr *
   ScopedVariableBindings::createVariableAccessExpr(const GraphGroup &scope, const std::string &vname)
   {
      return getVariableDesc(scope, vname).createAQLAccessExpr();
   }

   bool ScopedVariableBindings::hasAuthoritativeDefinition(const GraphGroup &scope, const std::string &vname) const
   {
      const GraphGroup *i=&scope;
      do
      {
         const VariableScope &vscope=getVariableScope(*i);
         VariableScope::map_type::const_iterator j=vscope.variables.find(vname);
         if (j!=vscope.variables.end())
         {
            return j->second.authoritative;
         }

         i=i->parentGraphGroup;
      } while (i);
      return false;
   }

   bool ScopedVariableBindings::isDefined(const GraphGroup &scope, const std::string &vname, bool recurse) const
   {
      const GraphGroup *i=&scope;
      do
      {
         const VariableScope &vscope=getVariableScope(*i);
         VariableScope::map_type::const_iterator j=vscope.variables.find(vname);
         if (j!=vscope.variables.end())
         {
            return true;
         }

         i=i->parentGraphGroup;
         print(OL_DEBUG, "-- parent=%p\n", i);
      } while (recurse && i);
      return false;
   }

   void ScopedVariableBindings::defineVariableScope(const GraphGroup &gg)
   {
      scopeMap.insert(std::make_pair(&gg, new VariableScope));
   }

   void ScopedVariableBindings::defineVariable(const GraphGroup &scope, const std::string &vname,
                                               const std::string &triple, AQLPropertyExpr::Property p,
                                               bool authoritative)
   {
      // ensure there's entry in scope map
      VariableScope &vscope=getVariableScope(scope);

      if (!authoritative && hasAuthoritativeDefinition(scope, vname))
         return; // don't override authoritative definition with non-authoritative

      if (authoritative || vscope.variables.find(vname)==vscope.variables.end())
      {
         // create entry
         vscope.variables.erase(vname); // erase possible previous definition
         vscope.variables.insert(std::make_pair(vname,
                                                SparqlVariableDesc(triple, p, authoritative)));
      }
      else {
         // update entry
         vscope.variables.at(vname).aqlTargets.push_back(SparqlVariableTarget(triple, p));
      }

      // update parent scopes
      if (scope.parentGraphGroup)
      {
         defineVariable(*scope.parentGraphGroup, vname, triple, p, false);
      }
   }
}
