/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <list>
#include <map>
#include <set>
#include <string>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "AQLException.h"
#include "AQLModel.h"
#include "SPARQLParser.h"
#include "SPARQLToAQL.h"
#include "SQLBackend.h"
#include "SPARQLTokenizer.h"
#include "Messages.h"

namespace TypeRQInternal {
   using namespace TypeRQ;

   class SPARQLParserException : public AQLException {
      public:
         SPARQLParserException(int line, int col, const char *message) : AQLException("Line %d Column %d: %s", line, col, message)  {}
         SPARQLParserException(int line, int col, const std::string &message) : AQLException("Line %d Column %d: %s", line, col, message.c_str())  {}
   };

   struct GraphGroupStacker {
         SPARQLIntermediateQuery &iq;
         GraphGroup *const old;

         GraphGroupStacker(SPARQLIntermediateQuery &_iq) : iq(_iq), old(iq.currentGraphGroup) {
            if (old) {
               GraphGroup *gg=new GraphGroup;
               iq.currentGraphGroup->triplesAndGraphGroups.push_back(gg);
               iq.currentGraphGroup=gg;
            } else {
               iq.currentGraphGroup=&iq.rootGraphGroup;
            }
         }

         ~GraphGroupStacker() {
            iq.currentGraphGroup=old;
         }
   };

   class SPARQLParserImpl
   {
      private:
         SPARQLTokenizer tokenizer;
         SPARQLIntermediateQuery iq;
         SQLFunctionMap &functionMap;
         const SPARQLParser::VariableBindingMode vbm;
         int anonymousNodeSeq; // used to generate variable names for anonymous nodes

         // namespace prefix to URI map
         std::map<std::string, std::string> namespaces;

         bool selectAll;
         std::set<std::string> autoselectedVariables; // variables selected by SELECT *

         std::string iriBase; // this can be set by BASE

         std::list<Token> tokenStack;
         typedef std::list<Token>::iterator TokenStackIterator;
         typedef std::list<Token>::reverse_iterator TokenStackRIterator;

         Token &tokenStackAt(int i) {
            if (i>=static_cast<int>(tokenStack.size())) {
               throwParseException("Internal error: tried to peek at token %d, but stack size is only %d",
                                   i, static_cast<int>(tokenStack.size()));
            }
            TokenStackRIterator j;
            for (j=tokenStack.rbegin(); i>0; --i, ++j) {
               //
            }
            return *j;
         }

         void throwParseException(const char *fmt, ...) __attribute__ ((noreturn))
         {
            va_list ap;
            va_start(ap, fmt);
            char *message=NULL;
            int ret=vasprintf(&message, fmt, ap);
            va_end(ap);

            std::string _message;

            if (ret>=0) {
               _message=message;
               free(message);
            } else {
               _message="SPARQLParserImpl::throwParseException(): Format failure!";
            }

            throw SPARQLParserException(tokenizer.getCurrentLine(), tokenizer.getCurrentCol(), _message.c_str());
         }

         void unsupported(const char *operation)
         {
            std::string message="Unsupported operation: ";
            message+=operation;
            throw SPARQLParserException(tokenizer.getCurrentLine(), tokenizer.getCurrentCol(), message.c_str());
         }

         void todo(const char *todoText)
         {
            std::string message="Unimplemented: ";
            message+=todoText;
            throw SPARQLParserException(tokenizer.getCurrentLine(), tokenizer.getCurrentCol(), message.c_str());
         }

         void tokenToNode(Token &t, Node &n) {
            switch (t.tokenCode) {
               case TOKEN_VAR:
                  n.type=Node::VARIABLE;
                  n.str=t.lastTokenString;

                  if (selectAll && n.str.at(0)!='$') {
                     if (autoselectedVariables.find(n.str)==autoselectedVariables.end()) {
                        autoselectedVariables.insert(n.str);
                        iq.selects.push_back(n.str);
                     }
                  }

                  break;

               case TOKEN_IRI_REF:
                  n.type=Node::IRI;
                  n.str=t.lastTokenString;
                  break;

               case TOKEN_PNAME_LN:
                  n.type=Node::IRI;
                  n.str=t.lastTokenString;
                  break;

               default:
                  throwParseException("tokenToNode: Unhandled token code %s", tokenizer.getTokenName(t.tokenCode));
            }
         }

         void createTriple(Token &s, Token &p, Token &o) {
            print(OL_DEBUG, "CreateTriple('%s', '%s', '%s')\n", s.lastTokenRaw.c_str(), p.lastTokenRaw.c_str(), o.lastTokenRaw.c_str());
            Triple *t=new Triple;
            tokenToNode(s, t->s);
            tokenToNode(p, t->p);
            tokenToNode(o, t->o);
            iq.currentGraphGroup->triplesAndGraphGroups.push_back(t);
         }


// [1]  	Query	  ::=  	Prologue
// ( SelectQuery | ConstructQuery | DescribeQuery | AskQuery )
         void parseQuery()
         {
            bool gotPrologue=parsePrologue();

            if (parseSelectQuery()) {
               // no semantic action here
            } else if (parseConstructQuery()) {
               // no semantic action here
            } else if (parseDescribeQuery()) {
               // no semantic action here
            } else if (parseAskQuery()) {
               // no semantic action here
            } else {
               if (gotPrologue)
                  throwParseException("Expected 'SELECT', 'CONSTRUCT', 'DESCRIBE' or 'ASK'");
               else
                  throwParseException("Expected query prologue, 'SELECT', 'CONSTRUCT', 'DESCRIBE' or 'ASK'");
            }
            tokenizer.nextToken(TOKEN_END);
         }

// [2]  	Prologue	  ::=  	BaseDecl? PrefixDecl*
         bool parsePrologue() {
            if (parseBaseDecl()) {
               while (parsePrefixDecl()) {}
               return true;
            } else if (parsePrefixDecl()) {
               while (parsePrefixDecl()) {}
               return true;
            }
            return false;
         }

// [3]  	BaseDecl	  ::=  	'BASE' IRI_REF
         bool parseBaseDecl() {
            if (tokenizer.eatIfNext(TOKEN_KEYWORD_BASE)) {
               tokenizer.nextToken(TOKEN_IRI_REF);
               iriBase=tokenizer.getLastToken().lastTokenString;
               print(OL_DEBUG, "Using base: \"%s\"\n", iriBase.c_str());
               unsupported("BASE declaration and local IRIs");
               return true;
            }
            return false;
         }

// [4]  	PrefixDecl	  ::=  	'PREFIX' PNAME_NS IRI_REF
         bool parsePrefixDecl() {
            if (tokenizer.eatIfNext(TOKEN_KEYWORD_PREFIX)) {
               tokenizer.nextToken(TOKEN_PNAME_NS);
               const std::string ns=tokenizer.getLastToken().lastTokenNS;
               tokenizer.nextToken(TOKEN_IRI_REF);
               const std::string iri=tokenizer.getLastToken().lastTokenString;
               print(OL_DEBUG, "Adding namespace prefix %s -> %s\n", ns.c_str(), iri.c_str());
               namespaces[ns]=iri;
               return true;
            }
            return false;
         }

// [5]  	SelectQuery	  ::=  	'SELECT' ( 'DISTINCT' | 'REDUCED' )? ( Var+ | '*' ) DatasetClause* WhereClause SolutionModifier
         bool parseSelectQuery() {
            if (tokenizer.eatIfNext(TOKEN_KEYWORD_SELECT)) {

               if (tokenizer.eatIfNext(TOKEN_KEYWORD_DISTINCT)) {
                  iq.distinct=true;
               } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_REDUCED)) {
                  iq.distinct=true;
               }

               if (tokenizer.eatIfNext(TOKEN_ASTERISK)) {
                  selectAll=true;
               } else if (parseVar()) {
                  do {
                     while (!tokenStack.empty()) {
                        iq.selects.push_back(tokenStack.front().lastTokenString);
                        tokenStack.pop_front();
                     }
                  } while (parseVar());
               } else {
                  throwParseException("Expected Var or '*'");
               }

               while (parseDatasetClause()) {
                  // nothing here
               }

               parseWhereClause();
               parseSolutionModifier();

               return true;
            }
            return false;
         }

// [6]  	ConstructQuery	  ::=  	'CONSTRUCT' ConstructTemplate DatasetClause* WhereClause SolutionModifier
         bool parseConstructQuery() {
            if (tokenizer.eatIfNext(TOKEN_KEYWORD_CONSTRUCT)) {
               if (parseConstructTemplate()) {

               } else {
                  throwParseException("Expected ConstructTemplate");
               }

               while (parseDatasetClause()) {
                  //
               }

               parseWhereClause();
               parseSolutionModifier();

               todo("CONSTRUCT query");
               return true;
            }

            return false;
         }

// [7]  	DescribeQuery	  ::=  	'DESCRIBE' ( VarOrIRIref+ | '*' ) DatasetClause* WhereClause? SolutionModifier
         bool parseDescribeQuery() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_DESCRIBE)) return false;

            if (tokenizer.eatIfNext(TOKEN_ASTERISK)) {
               //
            } else if (parseVarOrIRIref()) {
               do {
                  //
               } while (parseVarOrIRIref());
            } else {
               throwParseException("Expected VarOrIRIref or '*'");
            }

            while (parseDatasetClause()) {
               //
            }

            parseWhereClause();
            parseSolutionModifier();

            todo("DESCRIBE query");

            return true;
         }


// [8]  	AskQuery	  ::=  	'ASK' DatasetClause* WhereClause
         bool parseAskQuery() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_ASK)) return false;

            while (parseDatasetClause()) {
               //
            }

            parseWhereClause();
            todo("ASK query");
            return true;
         }

// [9]  	DatasetClause	  ::=  	'FROM' ( DefaultGraphClause | NamedGraphClause )
         bool parseDatasetClause() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_FROM)) return false;

            todo("Dataset clause, i.e., FROM ...");

            if (parseDefaultGraphClause()) {
               //
            } else if (parseNamedGraphClause()) {
               //
            } else {
               throwParseException("Expected DefaultGraphClause or NamedGraphClause");
            }

            return true;
         }

// [10]  	DefaultGraphClause	  ::=  	SourceSelector
         bool parseDefaultGraphClause() {
            return parseSourceSelector();
         }

// [11]  	NamedGraphClause	  ::=  	'NAMED' SourceSelector
         bool parseNamedGraphClause() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_NAMED)) return false;

            if (parseSourceSelector()) {
               //
            } else {
               throwParseException("Expected SourceSelector");
            }

            return true;
         }


// [12]  	SourceSelector	  ::=  	IRIref
         bool parseSourceSelector() {
            return parseIRIref();
         }

// [13]  	WhereClause	  ::=  	'WHERE'? GroupGraphPattern
         bool parseWhereClause() {
            if (tokenizer.eatIfNext(TOKEN_KEYWORD_WHERE)) {
               if (parseGroupGraphPattern()) {
                  //
               } else {
                  throwParseException("Expected GroupGraphPattern");
               }
               return true;
            } else {
               return parseGroupGraphPattern();
            }
         }

// [14]  	SolutionModifier	  ::=  	OrderClause? LimitOffsetClauses?
         bool parseSolutionModifier() {
            parseOrderClause();
            parseLimitOffsetClauses();
            return true;
         }


// [15]  	LimitOffsetClauses	  ::=  	( LimitClause OffsetClause? | OffsetClause LimitClause? )
         bool parseLimitOffsetClauses() {
            if (parseLimitClause()) {
               parseOffsetClause();
               return true;
            } else if (parseOffsetClause()) {
               parseLimitClause();
               return true;
            }
            return false;
         }

// [16]  	OrderClause	  ::=  	'ORDER' 'BY' OrderCondition+
         bool parseOrderClause() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_ORDER)) return false;

            tokenizer.nextToken(TOKEN_KEYWORD_BY);

            if (!parseOrderCondition()) {
               throwParseException("Expected OrderCondition");
            }
            do {
               //
            } while (parseOrderCondition());

            return true;
         }

// [17]  	OrderCondition	  ::=  	( ( 'ASC' | 'DESC' ) BrackettedExpression )
//                                      | ( Constraint | Var )
         bool parseOrderCondition() {
            OrderCondition oc;

            if (tokenizer.eatIfNext(TOKEN_ASC)) {
               if (!parseBrackettedExpression()) {
                  throwParseException("Expected BrackettedExpression");
               }
               Token &var=tokenStackAt(0);
               oc.exp=var.exp;
            } else if (tokenizer.eatIfNext(TOKEN_DESC)) {
               if (!parseBrackettedExpression()) {
                  throwParseException("Expected BrackettedExpression");
               }
               Token &var=tokenStackAt(0);
               oc.exp=var.exp;
               oc.ascending=false;
            } else if (parseConstraint()) {
               Token &var=tokenStackAt(0);
               oc.exp=var.exp;
            } else if (parseVar()) {
               Token &var=tokenStackAt(0);
               oc.exp.type=Expression::NODE;
               tokenToNode(var, oc.exp.node);
            } else {
               return false;
            }
            tokenStack.pop_back();
            iq.orders.push_back(oc);
            return true;
         }

// [18]  	LimitClause	  ::=  	'LIMIT' INTEGER
         bool parseLimitClause() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_LIMIT)) return false;
            tokenizer.nextToken(TOKEN_NUMBER);

            if (tokenizer.getLastToken().lastTokenNumberType!=Token::INTEGER) {
               throwParseException("Expected unsigned integer");
            }

            iq.limit=tokenizer.getLastToken().lastTokenInt;

            if (iq.limit!=tokenizer.getLastToken().lastTokenInt) {
               throwParseException("Numeric overflow");
            }

            return true;
         }


// [19]  	OffsetClause	  ::=  	'OFFSET' INTEGER
         bool parseOffsetClause() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_OFFSET)) return false;
            tokenizer.nextToken(TOKEN_NUMBER);

            if (tokenizer.getLastToken().lastTokenNumberType!=Token::INTEGER) {
               throwParseException("Expected integer");
            }

            iq.offset=tokenizer.getLastToken().lastTokenInt;

            if (iq.offset!=tokenizer.getLastToken().lastTokenInt) {
               throwParseException("Numeric overflow");
            }

            return true;
         }


// [20]  	GroupGraphPattern	  ::=  	'{' TriplesBlock? ( ( GraphPatternNotTriples | Filter ) '.'? TriplesBlock? )* '}'
         bool parseGroupGraphPattern() {
            if (!tokenizer.eatIfNext(TOKEN_LBRACE)) return false;

            GraphGroupStacker ggs(iq);

            bool anyMatch=false;
            do {
               anyMatch=false;
               if (parseTriplesBlock()) anyMatch=true;
               if (parseGraphPatternNotTriples()) {
                  anyMatch=true;
               } else {
                  if (parseFilter()) anyMatch=true;
               }
               if (tokenizer.eatIfNext(TOKEN_PERIOD)) {
                  anyMatch=true;
               }
               if (parseTriplesBlock()) anyMatch=true;

               if (!tokenStack.empty()) {
                  printTokenStack();
                  throwParseException("Internal: token stack not empty");
               }
            } while (anyMatch);

            tokenizer.nextToken(TOKEN_RBRACE);

            return true;
         }

// [21]  	TriplesBlock	  ::=  	TriplesSameSubject ( '.' TriplesBlock? )?
         bool parseTriplesBlock() {
            if (!parseTriplesSameSubject()) return false;

            if (tokenizer.eatIfNext(TOKEN_PERIOD)) {
               parseTriplesBlock();
            }

            return true;
         }


// [22]  	GraphPatternNotTriples	  ::=  	OptionalGraphPattern | GroupOrUnionGraphPattern | GraphGraphPattern
         bool parseGraphPatternNotTriples() {
            if (parseOptionalGraphPattern()) {
               return true;
            } else if (parseGroupOrUnionGraphPattern()) {
               return true;
            } else if (parseGraphGraphPattern()) {
               return true;
            } else {
               return false;
            }
         }


// [23]  	OptionalGraphPattern	  ::=  	'OPTIONAL' GroupGraphPattern
         bool parseOptionalGraphPattern() {
            if (tokenizer.eatIfNext(TOKEN_KEYWORD_OPTIONAL)) {
               if (parseGroupGraphPattern()) {
                  print(OL_DEBUG, "OptionalGraphPattern\n");
                  // WARNING: POTENTIALLY UNSAFE!
                  static_cast<GraphGroup *>(iq.currentGraphGroup->triplesAndGraphGroups.back())->isOptional=true;
                  return true;
               } else {
                  throwParseException("Expected GroupGraphPattern after '%s'", tokenizer.getTokenName(TOKEN_KEYWORD_OPTIONAL));
               }
            } else {
               return false;
            }
         }


// [24]  	GraphGraphPattern	  ::=  	'GRAPH' VarOrIRIref GroupGraphPattern
         bool parseGraphGraphPattern() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_GRAPH)) return false;
            if (!parseVarOrIRIref()) {
               throwParseException("Expected VarOrIRIref after '%s'", TOKEN_KEYWORD_GRAPH);
            }
            if (!parseGroupGraphPattern()) {
               throwParseException("Expected GroupGraphPattern after '%s VarOrIRIref'", tokenizer.getTokenName(TOKEN_KEYWORD_GRAPH));
            }
            unsupported("GraphGraphPattern");
            return true;
         }


// [25]  	GroupOrUnionGraphPattern	  ::=  	GroupGraphPattern ( 'UNION' GroupGraphPattern )*
         bool parseGroupOrUnionGraphPattern() {
            if (!parseGroupGraphPattern()) return false;

            while (tokenizer.eatIfNext(TOKEN_KEYWORD_UNION)) {
               if (!parseGroupGraphPattern()) {
                  throwParseException("Expected GroupGraphPattern after '%s'", tokenizer.getTokenName(TOKEN_KEYWORD_UNION));
               }
            }

            return true;
         }


// [26]  	Filter	  ::=  	'FILTER' Constraint
         bool parseFilter() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_FILTER)) return false;

            if (!parseConstraint()) {
               throwParseException("Expected Constraint after '%s'", TOKEN_KEYWORD_FILTER);
            }

            if (tokenStackAt(0).tokenCode!=TOKEN_EXPR) {
               printTokenStack();
               throwParseException("Expected TOKEN_EXPR in token stack");
            }

            iq.currentGraphGroup->filters.push_back(tokenStackAt(0).exp);
            tokenStack.pop_back();

            return true;
         }


// [27]  	Constraint	  ::=  	BrackettedExpression | BuiltInCall | FunctionCall
         bool parseConstraint() {
            if (parseBrackettedExpression()) {
               return true;
            } else if (parseBuiltInCall()) {
               return true;
            } else if (parseFunctionCall()) {
               return true;
            } else {
               return false;
            }
         }

// transform stack tokens representing function call into
// single stack token, which is function call expression
         void transformFunctionCallStackToExpr(int args) {
            if (args<0) args=0; // map NIL to 0 params
            Token &f=tokenStackAt(args);
            f.tokenCode=TOKEN_EXPR;
            f.exp.type=Expression::FUNCTION;
            f.exp.node.str=f.lastTokenString;
            f.exp.setSourceReference(f.sourceReference);

            for (int i=0; i<args; ++i) {
               int entry=args-1-i;
               Token &param=tokenStackAt(entry);
               if (param.tokenCode!=TOKEN_EXPR) {
                  printTokenStack();
                  throw AQLException("Token stack entry %d type is not TOKEN_EXPR", entry);
               }
               f.exp.params.push_back(param.exp);
            }
            for (int i=0; i<args; ++i) {
               tokenStack.pop_back();
            }
         }

// [28]  	FunctionCall	  ::=  	IRIref ArgList
         bool parseFunctionCall() {
            if (!parseIRIref())  return false;

            int n=parseArgList();

            if (!n) {
               throwParseException("Expected ArgList after IRIref in FunctionCall");
            }

            transformFunctionCallStackToExpr(n);

            return true;
         }

// [29]  	ArgList	  ::=  	( NIL | '(' Expression ( ',' Expression )* ')' )
         // returns value:
         // >=1: the number of parsed arguments
         // 0: didn't match
         // -1: matched NIL
         int parseArgList() {
            if (tokenizer.eatIfNext(TOKEN_NIL)) {
               return -1;
            } else {
               if (tokenizer.eatIfNext(TOKEN_LPAREN)) {
                  int n=0;
                  do {
                     if (!parseExpression()) {
                        throwParseException("Expected Expression in ArgList");
                     }
                     ++n;
                  } while (tokenizer.eatIfNext(TOKEN_COMMA));

                  tokenizer.nextToken(TOKEN_RPAREN);

                  return n;

               } else {
                  return 0;
               }
            }
         }

// [30]  	ConstructTemplate	  ::=  	'{' ConstructTriples? '}'
         bool parseConstructTemplate() {
            if (tokenizer.eatIfNext(TOKEN_LBRACE)) {

               parseConstructTriples();
               tokenizer.nextToken(TOKEN_RBRACE);

               return true;
            }

            return false;
         }

// [31]  	ConstructTriples	  ::=  	TriplesSameSubject ( '.' ConstructTriples? )?
         bool parseConstructTriples() {
            if (!parseTriplesSameSubject()) return false;

            if (tokenizer.eatIfNext(TOKEN_PERIOD)) {
               parseConstructTriples();
               todo("Construct triples");
            }

            return true;
         }


// [32]  	TriplesSameSubject	  ::=  	VarOrTerm PropertyListNotEmpty | TriplesNode PropertyList
         bool parseTriplesSameSubject() {
            if (parseVarOrTerm()) {
               if (!parsePropertyListNotEmpty()) {
                  throwParseException("Expected PropertyListNotEmpty after VarOrTerm in TriplesSameSubject");
               }
               tokenStack.pop_back();
               return true;
            } else if (parseTriplesNode()) {
               if (!parsePropertyList()) {
                  throwParseException("Expected PropertyList after TriplesNode in TriplesSameSubject");
               }
               throwParseException("Unimplemented: parseTriplesSameSubject: TriplesNode");
               return true;
            }
            return false;
         }


// [33]  	PropertyListNotEmpty	  ::=  	Verb ObjectList ( ';' ( Verb ObjectList )? )*
         bool parsePropertyListNotEmpty() {
            if (!parseVerb()) return false;

            if (!parseObjectList()) {
               throwParseException("Expected ObjectList after Vern in PropertyListNotEmpty");
            }

            tokenStack.pop_back();

            while (tokenizer.eatIfNext(TOKEN_SEMICOLON)) {
               if (parseVerb()) {
                  if (!parseObjectList()) {
                     throwParseException("Expected ObjectList after Verb in PropertyListNotEmpty");
                  }
                  tokenStack.pop_back();
               }
            }

            return true;
         }

// [34]  	PropertyList	  ::=  	PropertyListNotEmpty?
         bool parsePropertyList() {
            parsePropertyListNotEmpty();
            return true;
         }


// [35]  	ObjectList	  ::=  	Object ( ',' Object )*
         bool parseObjectList() {
            if (!parseObject()) return false;

            createTriple(tokenStackAt(2), tokenStackAt(1), tokenStackAt(0));
            tokenStack.pop_back();

            while (tokenizer.eatIfNext(TOKEN_COMMA)) {

               createTriple(tokenStackAt(2), tokenStackAt(1), tokenStackAt(0));
               tokenStack.pop_back();

               if (!parseObject()) {
                  throwParseException("Expected Object after '%s'in ObjectList", tokenizer.getTokenName(TOKEN_COMMA));
               }
            }

            return true;
         }


// [36]  	Object	  ::=  	GraphNode
         bool parseObject() {
            return parseGraphNode();
         }


// [37]  	Verb	  ::=  	VarOrIRIref | 'a'
         bool parseVerb() {
            SourceReference sref(tokenizer.getCurrentLine(), tokenizer.getCurrentCol());
            if (tokenizer.eatIfNext(TOKEN_KEYWORD_a)) {
               Token t(sref);
               t.tokenCode=TOKEN_IRI_REF;
               t.lastTokenString="http://www.w3.org/1999/02/22-rdf-syntax-ns#type";
               tokenStack.push_back(t);
               return true;
            } else {
               return parseVarOrIRIref();
            }
         }


// [38]  	TriplesNode	  ::=  	Collection | BlankNodePropertyList
         bool parseTriplesNode() {
            if (parseCollection()) {
               return true;
            } else if (parseBlankNodePropertyList()) {
               return true;
            } else {
               return false;
            }
         }


// [39]  	BlankNodePropertyList	  ::=  	'[' PropertyListNotEmpty ']'
         bool parseBlankNodePropertyList() {
            if (!tokenizer.eatIfNext(TOKEN_LBRACKET)) return false;

            if (!parsePropertyListNotEmpty()) {
               throwParseException("Expected PropertyListNotEmpty after '%s' in BlankNodePropertyList",
                                   tokenizer.getTokenName(TOKEN_LBRACKET));
            }
            tokenizer.nextToken(TOKEN_RBRACKET);

            unsupported("Blank node property list, i.e., [ ... ]");

            return true;
         }


// [40]  	Collection	  ::=  	'(' GraphNode+ ')'
         bool parseCollection() {
            if (!tokenizer.eatIfNext(TOKEN_LPAREN)) return false;

            unsupported("RDF Collection, i.e., ( ... )");

            if (!parseGraphNode()) {
               throwParseException("Expected GraphNode after '%s' in Collection", tokenizer.getTokenName(TOKEN_LPAREN));
            }
            tokenizer.nextToken(TOKEN_RPAREN);
            return true;
         }


// [41]  	GraphNode	  ::=  	VarOrTerm | TriplesNode
         bool parseGraphNode() {
            if (parseVarOrTerm()) {
               return true;
            } else if (parseTriplesNode()) {
               return true;
            } else {
               return false;
            }
         }


// [42]  	VarOrTerm	  ::=  	Var | GraphTerm
         bool parseVarOrTerm() {
            if (parseVar()) {
               return true;
            } else if (parseGraphTerm()) {
               return true;
            } else {
               return false;
            }
         }


// [43]  	VarOrIRIref	  ::=  	Var | IRIref
         bool parseVarOrIRIref() {
            if (parseVar()) {
               return true;
            } else if (parseIRIref()) {
               return true;
            } else {
               return false;
            }
         }


// [44]  	Var	  ::=  	VAR1 | VAR2
         bool parseVar() {
            if (tokenizer.eatIfNext(TOKEN_VAR)) {
               tokenStack.push_back(tokenizer.getLastToken());
               return true;
            }
            return false;
         }


// [45]  	GraphTerm	  ::=  	IRIref | RDFLiteral | NumericLiteral | BooleanLiteral | BlankNode | NIL
         bool parseGraphTerm() {
            if (parseIRIref()) {
               return true;
            } else if (parseRDFLiteral()) {
               return true;
            } else if (parseNumericLiteral()) {
               return true;
            } else if (parseBooleanLiteral()) {
               return true;
            } else if (parseBlankNode()) {
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_NIL)) {
               tokenStack.push_back(tokenizer.getLastToken());
               return true;
            } else {
               return false;
            }
         }


// [46]  	Expression	  ::=  	ConditionalOrExpression
         bool parseExpression() {
            return parseConditionalOrExpression();
         }


// [47]  	ConditionalOrExpression	  ::=  	ConditionalAndExpression ( '||' ConditionalAndExpression )*
         bool parseConditionalOrExpression() {
            SourceReference sref(tokenizer.getCurrentLine(), tokenizer.getCurrentCol());
            if (!parseConditionalAndExpression()) return false;

            while (tokenizer.eatIfNext(TOKEN_OR)) {
               if (!parseConditionalAndExpression()) {
                  throwParseException("Expected ConditionalAndExpression after '%s'", TOKEN_OR);
               }
               Expression exp(sref);
               exp.type=Expression::OR;
               exp.params.push_back(tokenStackAt(1).exp);
               exp.params.push_back(tokenStackAt(0).exp);
               tokenStack.pop_back();
               tokenStack.pop_back();
               tokenStack.push_back(exp);
            }

            return true;
         }


// [48]  	ConditionalAndExpression	  ::=  	ValueLogical ( '&&' ValueLogical )*
         bool parseConditionalAndExpression() {
            SourceReference sref(tokenizer.getCurrentLine(), tokenizer.getCurrentCol());
            if (!parseValueLogical()) return false;

            while (tokenizer.eatIfNext(TOKEN_AND)) {
               if (!parseValueLogical()) {
                  throwParseException("Expected ValueLogical after '%s'", TOKEN_AND);
               }
               Expression exp(sref);
               exp.type=Expression::AND;
               exp.params.push_back(tokenStackAt(1).exp);
               exp.params.push_back(tokenStackAt(0).exp);
               tokenStack.pop_back();
               tokenStack.pop_back();
               tokenStack.push_back(exp);
            }

            return true;
         }


// [49]  	ValueLogical	  ::=  	RelationalExpression
         bool parseValueLogical() {
            return parseRelationalExpression();
         }


// [50]  	RelationalExpression	  ::=  	NumericExpression ( '=' NumericExpression | '!=' NumericExpression | '<' NumericExpression | '>' NumericExpression | '<=' NumericExpression | '>=' NumericExpression )?
         bool parseRelationalExpression() {
            SourceReference sref(tokenizer.getCurrentLine(), tokenizer.getCurrentCol());
            if (!parseNumericExpression()) return false;

            if (tokenizer.eatIfNext(TOKEN_REL_OP)) {
               Expression exp(sref);
               switch (tokenizer.getLastToken().lastTokenInt) {
                  case TOKEN_REL_OPCODE_EQ:
                     exp.type=Expression::EQUAL;
                     break;
                  case TOKEN_REL_OPCODE_NE:
                     exp.type=Expression::NOT_EQUAL;
                     break;
                  case TOKEN_REL_OPCODE_LT:
                     exp.type=Expression::LESS;
                     break;
                  case TOKEN_REL_OPCODE_LE:
                     exp.type=Expression::LESS_OR_EQUAL;
                     break;
                  case TOKEN_REL_OPCODE_GE:
                     exp.type=Expression::GREATER_OR_EQUAL;
                     break;
                  case TOKEN_REL_OPCODE_GT:
                     exp.type=Expression::GREATER;
                     break;
                  default:
                     throwParseException("parseRelationalExpression() Internal Error: unhandled relop code %d",
                                         tokenizer.getLastToken().lastTokenInt);
               }

               if (!parseNumericExpression()) {
                  throwParseException("Expected NumericExpression after '%s'", tokenizer.getTokenName(TOKEN_REL_OP));
               }

               exp.params.push_back(tokenStackAt(1).exp);
               exp.params.push_back(tokenStackAt(0).exp);
               tokenStack.pop_back();
               tokenStack.pop_back();
               tokenStack.push_back(exp);
            }

            return true;
         }


// [51]  	NumericExpression	  ::=  	AdditiveExpression
         bool parseNumericExpression() {
            return parseAdditiveExpression();
         }


      void transformConstantFunctionCallStackToExpr(const char *sparqlFn, int args)
      {
         SourceReference sref(tokenizer.getCurrentLine(), tokenizer.getCurrentCol());
         // insert function call token
         std::list<Token>::iterator it=tokenStack.end();
         for (int i=0; i<args; ++i)
         {
            --it;
         }
         Token &fn=*(tokenStack.insert(it, Token(sref)));
         fn.tokenCode=TOKEN_EXPR;
         fn.exp.type=Expression::FUNCTION;
         fn.exp.node.type=Node::IRI;
         fn.exp.node.str=sparqlFn;
         fn.exp.setSourceReference(sref);

         // transform stack expression to parameters
         for (int i=0; i<args; ++i)
         {
            Token &param=*it++;
            if (param.tokenCode!=TOKEN_EXPR)
            {
               printTokenStack();
               throw AQLException("Function parameter %d not an TOKEN_EXPR, but %d", i+1, (int)param.tokenCode);
            }

            fn.exp.params.push_back(param.exp);
         }

         for (int i=0; i<args; ++i)
         {
            tokenStack.pop_back();
         }
      }

// [52]  	AdditiveExpression	  ::=  	MultiplicativeExpression ( '+' MultiplicativeExpression | '-' MultiplicativeExpression | NumericLiteralPositive | NumericLiteralNegative )*
// NOTE: this is reduced to
//       	AdditiveExpression	  ::=  	MultiplicativeExpression ( '+' MultiplicativeExpression | '-' MultiplicativeExpression )*
         bool parseAdditiveExpression() {
            if (!parseMultiplicativeExpression()) return false;

            while (true)
            {
               const char *fnName;
               if (tokenizer.eatIfNext(TOKEN_PLUS))
               {
                  fnName="builtin:add";
               }
               else if (tokenizer.eatIfNext(TOKEN_MINUS))
               {
                  fnName="builtin:substract";
               }
               else {
                  return true;
               }

               if (!parseMultiplicativeExpression()) {
                  throwParseException("Expected MultiplicativeExpression after '%s' or '%s'",
                                      tokenizer.getTokenName(TOKEN_PLUS), tokenizer.getTokenName(TOKEN_MINUS));
               }

               transformConstantFunctionCallStackToExpr(fnName, 2);
            }

            return true;
         }


// [53]  	MultiplicativeExpression	  ::=  	UnaryExpression ( '*' UnaryExpression | '/' UnaryExpression )*
         bool parseMultiplicativeExpression() {
            if (!parseUnaryExpression()) return false;

            while (true)
            {
               const char *fnName;
               if (tokenizer.eatIfNext(TOKEN_ASTERISK))
               {
                  fnName="builtin:multiply";
               }
               else if (tokenizer.eatIfNext(TOKEN_SLASH))
               {
                  fnName="builtin:divide";
               }
               else {
                  return true;
               }

               if (!parseUnaryExpression()) {
                  throwParseException("Expected UnaryExpression after '%s' or '%s'",
                                      tokenizer.getTokenName(TOKEN_ASTERISK), tokenizer.getTokenName(TOKEN_SLASH));
               }
               transformConstantFunctionCallStackToExpr(fnName, 2);
            }

            return true;
         }

// [54]  	UnaryExpression	  ::=  	  '!' PrimaryExpression | '+' PrimaryExpression | '-' PrimaryExpression | PrimaryExpression
         bool parseUnaryExpression() {
            SourceReference sref(tokenizer.getCurrentLine(), tokenizer.getCurrentCol());
            if (tokenizer.eatIfNext(TOKEN_EXCLAMATION)) {
               if (parsePrimaryExpression()) {
                  Expression exp(sref);
                  exp.type=Expression::NOT;
                  exp.params.push_back(tokenStackAt(0).exp);
                  tokenStack.pop_back();
                  tokenStack.push_back(exp);
                  return true;
               }
               throwParseException("Expected PrimaryExpression");
            } else if (tokenizer.eatIfNext(TOKEN_PLUS)) {
               if (parsePrimaryExpression()) {
                  return true;
               }
               throwParseException("Expected PrimaryExpression");
            } else if (tokenizer.eatIfNext(TOKEN_MINUS)) {
               if (parsePrimaryExpression()) {
                  Token &tok=tokenStackAt(0);
                  if (tok.tokenCode==TOKEN_EXPR && tok.exp.type==Expression::NODE &&
                      tok.exp.node.type==Node::INTEGER) {
                     tok.exp.node.i=-tok.exp.node.i;
                  } else if (tok.tokenCode==TOKEN_EXPR && tok.exp.type==Expression::NODE &&
                             tok.exp.node.type==Node::DOUBLE) {
                     tok.exp.node.d=-tok.exp.node.d;
                  } else {
                     transformConstantFunctionCallStackToExpr("builtin:negate", 1);
                  }

                  return true;
               }
               throwParseException("Expected PrimaryExpression");
            } else {
               return parsePrimaryExpression();
            }
         }

      void transformVarToVarExpression()
      {
         Token &var=tokenStackAt(0);
         var.tokenCode=TOKEN_EXPR;
         var.exp.type=Expression::NODE;
         var.exp.node.type=Node::VARIABLE;
         var.exp.node.str=var.lastTokenString;
         var.exp.setSourceReference(var.sourceReference);
      }


// [55]  	PrimaryExpression	  ::=  	BrackettedExpression | BuiltInCall | IRIrefOrFunction | RDFLiteral | NumericLiteral | BooleanLiteral | Var
         bool parsePrimaryExpression() {
            if (parseBrackettedExpression()) {
               return true;
            } else if (parseBuiltInCall()) {
               return true;
            } else if (parseIRIrefOrFunction()) {
               return true;
            } else if (parseRDFLiteral()) {
               Token &var=tokenStackAt(0);
               var.tokenCode=TOKEN_EXPR;
               var.exp.type=Expression::NODE;
               var.exp.node.type=Node::STRING_LITERAL;
               var.exp.node.str=var.lastTokenString;
               return true;
            } else if (parseNumericLiteral()) {
               return true;
            } else if (parseBooleanLiteral()) {
               return true;
            } else if (parseVar()) {
               transformVarToVarExpression();
               return true;
            } else {
               return false;
            }
         }


// [56]  	BrackettedExpression	  ::=  	'(' Expression ')'
         bool parseBrackettedExpression() {
            if (!tokenizer.eatIfNext(TOKEN_LPAREN)) return false;

            if (!parseExpression()) {
               throwParseException("BrackettedExpression: Expected Expression after '%s'", tokenizer.getTokenName(TOKEN_LPAREN));
            }

            tokenizer.nextToken(TOKEN_RPAREN);

            return true;
         }

// [57]  	BuiltInCall	  ::=  	  'STR' '(' Expression ')' 
// | 'LANG' '(' Expression ')' 
// | 'LANGMATCHES' '(' Expression ',' Expression ')' 
// | 'DATATYPE' '(' Expression ')' 
// | 'BOUND' '(' Var ')' 
// | 'sameTerm' '(' Expression ',' Expression ')' 
// | 'isIRI' '(' Expression ')' 
// | 'isURI' '(' Expression ')' 
// | 'isBLANK' '(' Expression ')' 
// | 'isLITERAL' '(' Expression ')' 
// | RegexExpression

         void parseRequiredExpression() {
            if (!parseExpression()) {
               throwParseException("Expected expression");
            }
         }

         bool parseBuiltInCall() {
            SourceReference sref(tokenizer.getCurrentLine(), tokenizer.getCurrentCol());
            if (tokenizer.eatIfNext(TOKEN_KEYWORD_STR)) {
               tokenizer.nextToken(TOKEN_LPAREN);
               Token t(sref);
               t.tokenCode=TOKEN_STRING_LITERAL;
               t.lastTokenString="builtin:to-string";
               tokenStack.push_back(t);
               parseRequiredExpression();
               transformFunctionCallStackToExpr(1);
               tokenizer.nextToken(TOKEN_RPAREN);
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_LANG)) {
               tokenizer.nextToken(TOKEN_LPAREN);
               parseRequiredExpression();
               tokenizer.nextToken(TOKEN_RPAREN);
               unsupported("Built-in function: LANG");
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_LANGMATCHES)) {
               tokenizer.nextToken(TOKEN_LPAREN);
               parseRequiredExpression();
               tokenizer.nextToken(TOKEN_COMMA);
               parseRequiredExpression();
               tokenizer.nextToken(TOKEN_RPAREN);
               unsupported("Built-in function: LANGMATCHES");
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_DATATYPE)) {
               tokenizer.nextToken(TOKEN_LPAREN);
               parseRequiredExpression();
               tokenizer.nextToken(TOKEN_RPAREN);
               unsupported("Built-in function: DATATYPE");
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_BOUND)) {
               tokenizer.nextToken(TOKEN_LPAREN);
               if (!parseVar()) throwParseException("Expected Var");
               tokenizer.nextToken(TOKEN_RPAREN);
               transformVarToVarExpression();
               transformConstantFunctionCallStackToExpr("builtin:is-not-null", 1);
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_sameTERM)) {
               tokenizer.nextToken(TOKEN_LPAREN);
               parseRequiredExpression();
               tokenizer.nextToken(TOKEN_COMMA);
               parseRequiredExpression();
               tokenizer.nextToken(TOKEN_RPAREN);
               unsupported("Built-in function: sameTERM");
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_isURI) || tokenizer.eatIfNext(TOKEN_KEYWORD_isIRI)) {
               tokenizer.nextToken(TOKEN_LPAREN);
               parseRequiredExpression();
               tokenizer.nextToken(TOKEN_RPAREN);
               unsupported("Built-in function: isURI/isIRI");
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_isBLANK)) {
               tokenizer.nextToken(TOKEN_LPAREN);
               parseRequiredExpression();
               tokenizer.nextToken(TOKEN_RPAREN);
               unsupported("Built-in function: isBLANK");
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_isLITERAL)) {
               tokenizer.nextToken(TOKEN_LPAREN);
               parseRequiredExpression();
               tokenizer.nextToken(TOKEN_RPAREN);
               unsupported("Built-in function: isLITERAL");
               return true;
            } else {
               return parseRegexExpression();
            }
         }

// [58]  	RegexExpression	  ::=  	'REGEX' '(' Expression ',' Expression ( ',' Expression )? ')'
         bool parseRegexExpression() {
            if (!tokenizer.eatIfNext(TOKEN_KEYWORD_REGEX)) return false;

            tokenizer.nextToken(TOKEN_LPAREN);
            parseRequiredExpression();
            tokenizer.nextToken(TOKEN_COMMA);
            parseRequiredExpression();
            if (tokenizer.eatIfNext(TOKEN_COMMA)) {
               parseRequiredExpression();
            }
            tokenizer.nextToken(TOKEN_RPAREN);

            unsupported("Built-in function: REGEX");

            return true;
         }


// [59]  	IRIrefOrFunction	  ::=  	IRIref ArgList?
         bool parseIRIrefOrFunction() {
            if (!parseIRIref()) return false;

            int n;
            n=parseArgList();
            if (n) {
               transformFunctionCallStackToExpr(n);
            } else {
               Token &var=tokenStackAt(0);
               var.tokenCode=TOKEN_EXPR;
               var.exp.type=Expression::NODE;
               var.exp.node.type=Node::IRI;
               var.exp.node.str=var.lastTokenString;
               var.exp.setSourceReference(var.sourceReference);
            }
            return true;
         }


// [60]  	RDFLiteral	  ::=  	String ( LANGTAG | ( '^^' IRIref ) )?
         bool parseRDFLiteral() {
            if (!parseString()) return false;

            if (tokenizer.eatIfNext(TOKEN_LANGTAG)) {
               unsupported("LANGTAG in RDFLiteral");
            } else if (tokenizer.eatIfNext(TOKEN_DOUBLE_CARET)) {
               unsupported("Type definition in RDFLiteral");
               if (!parseIRIref()) {
                  throwParseException("Expected IRIref after '%s'", TOKEN_DOUBLE_CARET);
               }
            }

            return true;
         }


// [61]  	NumericLiteral	  ::=  	NumericLiteralUnsigned | NumericLiteralPositive | NumericLiteralNegative
         bool parseNumericLiteral() {
            if (parseNumericLiteralUnsigned()) {
               // nothing here
            } else if (parseNumericLiteralPositive()) {
               // nothing here
            } else if (parseNumericLiteralNegative()) {
               // nothing here
            } else {
               return false;
            }

            Token &var=tokenStackAt(0);
            var.tokenCode=TOKEN_EXPR;
            var.exp.type=Expression::NODE;

            if (var.lastTokenNumberType==Token::INTEGER) {
               var.exp.node.type=Node::INTEGER;
               var.exp.node.i=var.lastTokenInt;
            } else {
               var.exp.node.type=Node::DOUBLE;
               var.exp.node.d=var.lastTokenDouble;
            }
            var.exp.setSourceReference(var.sourceReference);

            return true;
         }

// [62]  	NumericLiteralUnsigned	  ::=  	INTEGER | DECIMAL | DOUBLE
// note, this is changed to
// [62]  	NumericLiteralUnsigned	  ::=  	NUMBER
         bool parseNumericLiteralUnsigned() {
            if (tokenizer.eatIfNext(TOKEN_NUMBER)) {
               tokenStack.push_back(tokenizer.getLastToken());
               return true;
            } else {
               return false;
            }
         }


// [63]  	NumericLiteralPositive	  ::=  	INTEGER_POSITIVE | DECIMAL_POSITIVE | DOUBLE_POSITIVE
// note, this is changed to
// [63]  	NumericLiteralPositive	  ::=  	NUMBER_POSITIVE
         bool parseNumericLiteralPositive() {
            if (tokenizer.eatIfNext(TOKEN_NUMBER_POSITIVE)) {
               tokenStack.push_back(tokenizer.getLastToken());
               return true;
            } else {
               return false;
            }
         }

// [64]  	NumericLiteralNegative	  ::=  	INTEGER_NEGATIVE | DECIMAL_NEGATIVE | DOUBLE_NEGATIVE
// note, this is changed to
// [64]  	NumericLiteralNegative	  ::=  	NUMBER_NEGATIVE
         bool parseNumericLiteralNegative() {
            if (tokenizer.eatIfNext(TOKEN_NUMBER_NEGATIVE)) {
               tokenStack.push_back(tokenizer.getLastToken());
               return true;
            } else {
               return false;
            }
         }


// [65]  	BooleanLiteral	  ::=  	'true' | 'false'
         bool parseBooleanLiteral() {
            SourceReference sref(tokenizer.getCurrentLine(), tokenizer.getCurrentCol());
            bool bValue;
            if (tokenizer.eatIfNext(TOKEN_KEYWORD_true)) {
               bValue=true;
            } else if (tokenizer.eatIfNext(TOKEN_KEYWORD_false)) {
               bValue=false;
            } else {
               return false;
            }

            Token token(sref);
            token.tokenCode=TOKEN_EXPR;
            token.exp.type=Expression::NODE;
            token.exp.node.b=bValue;
            token.exp.node.type=Node::BOOLEAN;
            tokenStack.push_back(token);

            return true;
         }

// [66]  	String	  ::=  	STRING_LITERAL1 | STRING_LITERAL2 | STRING_LITERAL_LONG1 | STRING_LITERAL_LONG2
         bool parseString() {
            if (tokenizer.eatIfNext(TOKEN_STRING_LITERAL)) {
               Token token(tokenizer.getLastToken());
               token.exp.setSourceReference(token.sourceReference);
               tokenStack.push_back(token);
               return true;
            } else {
               return false;
            }
         }


// [67]  	IRIref	  ::=  	IRI_REF | PrefixedName
         bool parseIRIref() {
            if (tokenizer.eatIfNext(TOKEN_IRI_REF)) {
               Token token(tokenizer.getLastToken());
               token.exp.setSourceReference(token.sourceReference);
               tokenStack.push_back(token);
               return true;
            } else if (parsePrefixedName()) {
               return true;
            } else {
               return false;
            }
         }

// [68]  	PrefixedName	  ::=  	PNAME_LN | PNAME_NS
         bool parsePrefixedName() {
            if (tokenizer.eatIfNext(TOKEN_PNAME_LN)) {
               //
            } else if (tokenizer.eatIfNext(TOKEN_PNAME_NS)) {
               //
            } else {
               return false;
            }

            // resolve namespace
            const std::string &ns=tokenizer.getLastToken().lastTokenNS;
            if (namespaces.find(ns)==namespaces.end()) {
               throwParseException("Namespace prefix '%s' not defined.", ns.c_str());
            }
            Token token=tokenizer.getLastToken();
            token.lastTokenString=namespaces.at(ns);
            token.lastTokenString+=tokenizer.getLastToken().lastTokenLN;
            tokenStack.push_back(token);

            return true;
         }

// [69]  	BlankNode	  ::=  	BLANK_NODE_LABEL | ANON
         bool parseBlankNode() {
            SourceReference sref(tokenizer.getCurrentLine(), tokenizer.getCurrentCol());
            if (tokenizer.eatIfNext(TOKEN_BLANK_NODE_LABEL)) {
               Token token=tokenizer.getLastToken();
               token.tokenCode=TOKEN_VAR;
               token.lastTokenString.insert(0, "$blank$");
               tokenStack.push_back(token);
               return true;
            } else if (tokenizer.eatIfNext(TOKEN_ANON)) {
               char *s=NULL;
               if (asprintf(&s, "$anon$%d", anonymousNodeSeq++)>=0) {
                  Token token(sref);
                  token.tokenCode=TOKEN_VAR;
                  token.lastTokenString=s;
                  free(s);
                  tokenStack.push_back(token);
                  return true;
               } else {
                  throw AQLException("Error while generating name for blank node");
               }
            } else {
               return false;
            }
         }

         void printTokenStack() {
            printf("Token stack: {\n");
            for (std::list<Token>::const_reverse_iterator i=tokenStack.rbegin();
                 i!=tokenStack.rend();
                 ++i) {
               const Token &token=*i;
               printf("- %s(%d) ", tokenizer.getTokenName(token.tokenCode), (int)token.tokenCode);
               switch (token.tokenCode) {
                  case TOKEN_VAR:  printf("'%s'", token.lastTokenString.c_str()); break;
                  case TOKEN_EXPR: printExpression(token.exp); break;

                  default:
                     printf("Unhandled token code %d", static_cast<int>(token.tokenCode));
                     break;
               }
               printf("\n");
            }
            printf("}\n");
         }

   public:
      SPARQLParserImpl(std::istream &_is, SQLFunctionMap &_functionMap, SPARQLParser::VariableBindingMode _vbm)
         : tokenizer(_is), iq(_functionMap), functionMap(_functionMap), vbm(_vbm),
           anonymousNodeSeq(0), selectAll(false)
      {
      }

      void parse()
      {
         parseQuery();
         if (!tokenStack.empty()) {
            printTokenStack();
            throwParseException("Internal: token stack is not empty, %d objects remain", static_cast<int>(tokenStack.size()));
         }

         //return createSPARQLQueryImpl(iq);
      }

      void printIntermediateQuery(std::ostream &os)
      {
         os << "SELECTS:";
         if (iq.distinct) os << " DISTINCT";
         for (std::list<std::string>::const_iterator i=iq.selects.begin(); i!=iq.selects.end(); ++i) {
            os << ' ' << *i;
         }
         os << std::endl;

         printGraphGroup(iq, iq.rootGraphGroup, 0, false);

         os << "Orderings: {" << std::endl;
         for (SPARQLIntermediateQuery::OrderList::iterator i=iq.orders.begin(); i!=iq.orders.end(); ++i) {
            os << "- ";
            if (i->ascending) {
               os << "ASC:  ";
            } else {
               os << "DESC: ";
            }
            printExpression(i->exp);
            os << std::endl;
         }
         os << "}" << std::endl;
      }

      SPARQLIntermediateQuery &getIq()
      {
         return iq;
      }

      SPARQLParser::VariableBindingMode getVbm() const
      {
         return vbm;
      }
   };
}


namespace TypeRQ {
   using namespace TypeRQInternal;

   SPARQLParser::SPARQLParser() : parserImpl()
   {
   }

   void SPARQLParser::parseQuery(std::istream &is, SQLFunctionMap &functionMap, VariableBindingMode vbm)
   {
      delete parserImpl;
      parserImpl=0;

      parserImpl=new SPARQLParserImpl(is, functionMap, vbm);
      parserImpl->parse();
   }

   void SPARQLParser::printIntermediateQuery(std::ostream &os)
   {
      parserImpl->printIntermediateQuery(os);
   }

   SPARQLParser::~SPARQLParser()
   {
      delete parserImpl;
      parserImpl=0;
   }

   AQLQuery *SPARQLParser::translateToAQL()
   {
      return TypeRQInternal::translateToAQL(parserImpl->getIq(),
                                            parserImpl->getVbm()==SPARQLParser::VBM_BOTTOM_UP);
   }
}
