/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <sstream>

#include <string.h>
#include <strings.h>
#include <errno.h>
#include <raptor.h>
#include <stdlib.h>

#include "AQLException.h"
#include "AQLModel.h"
#include "N3TripleReader.h"
#include "Messages.h"

namespace TypeRQInternal {
   using namespace TypeRQ;

   struct N3TripleReaderContext
   {
      raptor_parser* rdf_parser;
      N3TripleHandler *tripleHandler;

      bool aborted;
      std::string exceptionMessage;
   };

}

namespace {
   using namespace TypeRQ;
   using namespace TypeRQInternal;

   AQLLiteralExpr raptorNodeToAQLLiteral(const void *node, const raptor_identifier_type type,
                                         raptor_uri *literal_datatype,
                                         const unsigned char *lang)
   {
      switch (type)
      {
         case RAPTOR_IDENTIFIER_TYPE_RESOURCE:
            return AQLLiteralExpr(static_cast<const char *>(node), AQLTypeSet::IRI);
            break;

         case RAPTOR_IDENTIFIER_TYPE_ANONYMOUS:
            return AQLLiteralExpr(std::string("_:")+static_cast<const char *>(node), AQLTypeSet::IRI);
            break;

         case RAPTOR_IDENTIFIER_TYPE_LITERAL:
            if (!literal_datatype)
            {
               // basic string literal
               return AQLLiteralExpr(static_cast<const char *>(node), AQLTypeSet::STRING);
            }
            else {
               unsigned char *udatatype=raptor_uri_to_string(literal_datatype);
               std::string datatype=(char *)udatatype;
               raptor_free_memory(udatatype);

               if (datatype=="http://www.w3.org/2000/01/rdf-schema#XMLLiteral" ||
                   datatype=="http://www.w3.org/2001/XMLSchema#string")
               {
                  // mapped as string literal
                  return AQLLiteralExpr(static_cast<const char *>(node), AQLTypeSet::STRING);
               }
               else if (datatype=="http://www.w3.org/2001/XMLSchema#datetime")
               {
                  // TODO: format validation
                  return AQLLiteralExpr(static_cast<const char *>(node), AQLTypeSet::DATETIME);
               }

               else if (datatype=="http://www.w3.org/2001/XMLSchema#integer")
               {
                  std::istringstream iss(static_cast<const char *>(node));
                  int64_t intLiteral;
                  iss >> intLiteral;
                  return AQLLiteralExpr(intLiteral);
               }
               else if (datatype=="http://www.w3.org/2001/XMLSchema#double" || datatype=="http://www.w3.org/2001/XMLSchema#decimal")
               {
                  std::istringstream iss(static_cast<const char *>(node));
                  double dblLiteral;
                  iss >> dblLiteral;
                  return AQLLiteralExpr(dblLiteral);
               }
               else if (datatype=="http://www.w3.org/2001/XMLSchema#boolean")
               {
                  const char *token=static_cast<const char *>(node);
                  bool value;
                  if (strcasecmp(token, "false")==0 || strcmp(token, "0")==0)
                     value=false;
                  else if (strcasecmp(token, "true")==0 || strcmp(token, "1")==0)
                     value=true;
                  else
                     throw AQLException("Bad token string for boolean value: '%s'", token);

                  return AQLLiteralExpr(value);
               }

               throw AQLException("Unknown RDF data type %s", datatype.c_str());
            }
            break;

         default:
            throw AQLException("Unhandled raptor identifier type %d", static_cast<int>(type));
      }

      throw AQLException("Unhandled raptor identified type %d", type);
   }

   void triples_handler(void* user_data, const raptor_statement* triple) 
   {
      N3TripleReaderContext *context=static_cast<N3TripleReaderContext *>(user_data);      

      try {
         if (!context->aborted)
         {
            const AQLLiteralExpr &s=
               raptorNodeToAQLLiteral(triple->subject, triple->subject_type, 0, 0);
            const AQLLiteralExpr &p=
               raptorNodeToAQLLiteral(triple->predicate, triple->predicate_type, 0, 0);
            const AQLLiteralExpr &o=
               raptorNodeToAQLLiteral(triple->object, triple->object_type,
                                      triple->object_literal_datatype,
                                      triple->object_literal_language);

            context->tripleHandler->handleTriple(s, p, o);
         }
      }
      catch (AQLException &ex) {
         context->aborted=true;
         context->exceptionMessage=ex.getReason();
         raptor_parse_abort(context->rdf_parser);
      }
   }

   void print_raptor_message(OutputLevel ol,
                             const char *severity,
                             raptor_locator* locator, 
                             const char *message)
   {
      print(ol, "%s line %d %s: %s\n",
            locator->file, locator->line, severity, message);
   }

   void fatal_message_handler(void *user_data, raptor_locator* locator, 
                              const char *message)
   {
      print_raptor_message(OL_QUIET, "fatal", locator, message);
   }

   void error_message_handler(void *user_data, raptor_locator* locator, 
                              const char *message)
   {
      print_raptor_message(OL_QUIET, "error", locator, message);
   }

   void warning_message_handler(void *user_data, raptor_locator* locator, 
                                const char *message)
   {
      print_raptor_message(OL_NORMAL, "warning", locator, message);
   }

}

namespace TypeRQ {
   using namespace TypeRQInternal;

   N3TripleHandler::~N3TripleHandler() {}

   N3TripleReader::N3TripleReader() : in(0), invokeClose(false), context(0) {}

   N3TripleReader::~N3TripleReader()
   {
      close();
   }

   void N3TripleReader::openFile(const std::string &_filename)
   {
      close();
      in=fopen(_filename.c_str(), "rb");
      if (in==NULL)
      {
         throw AQLException("Error opening file %s: %s", _filename.c_str(), strerror(errno));
      }
      invokeClose=true;
      filename=_filename;

      openParserContext();
   }

   void N3TripleReader::openConsoleIn()
   {
      close();
      in=stdin;
      invokeClose=false;
      filename="stdin";
      openParserContext();
   }

   void N3TripleReader::close()
   {
      closeParserContext();

      if (in!=NULL && invokeClose)
      {
         fclose(in);
      }
      in=NULL;
      invokeClose=false;
   }

   void N3TripleReader::openParserContext()
   {
      context=new N3TripleReaderContext;
      try {
         context->rdf_parser = raptor_new_parser_for_content(NULL, "application/n3", NULL, 0, NULL);
         raptor_set_fatal_error_handler(context->rdf_parser, context, fatal_message_handler);
         raptor_set_error_handler(context->rdf_parser, context, error_message_handler);
         raptor_set_warning_handler(context->rdf_parser, context, warning_message_handler);
         raptor_set_statement_handler(context->rdf_parser, context, triples_handler);
      }
      catch (...) {
         closeParserContext();
         throw;
      }
   }

   void N3TripleReader::closeParserContext()
   {
      if (context)
      {
         if (context->rdf_parser)
         {
            raptor_free_parser(context->rdf_parser);
            context->rdf_parser=0;
         }
         delete context;
      }
      context=0;
   }

   void N3TripleReader::readTriples(N3TripleHandler &tripleHandler)
   {
      print(OL_DEBUG, "Reading triples...\n");
      context->tripleHandler=&tripleHandler;

      unsigned char *uri_string=raptor_uri_filename_to_uri_string(filename.c_str());
      raptor_uri *uri=raptor_new_uri(uri_string);

      context->aborted=false;
      context->exceptionMessage.clear();

      raptor_parse_file_stream(context->rdf_parser, in, filename.c_str(), uri);

      raptor_free_uri(uri);
      raptor_free_memory(uri_string);

      context->tripleHandler=0;

      if (context->aborted)
      {
         throw AQLException(context->exceptionMessage);
      }
   }

}
