/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>

#include "ListPrettyPrinter.h"

namespace TypeRQ {

   ListPrettyPrinter::ListPrettyPrinter(std::ostream &_os) : os(_os), requireSpaceBeforeTerm(false),
                                                             currentIndent(), cursorPos() {}

   ListPrettyPrinter::~ListPrettyPrinter()
   {
      if (!levelIndents.empty())
         os << "WARNING: level indents size is " << levelIndents.size() << ", should be 0" << std::endl;
   }

   void ListPrettyPrinter::openLevel(const char *sz)
   {
      updateIndent();
      levelIndents.push_back(cursorPos);
      (*this) << '(';
      (*this) << sz;
      requireSpaceBeforeTerm=(*sz!=0);
   }

   void ListPrettyPrinter::newTerm()
   {
      updateIndent();
      requireSpaceBeforeTerm=true;
   }

   void ListPrettyPrinter::closeLevel()
   {
      (*this) << ')';
      currentIndent=cursorPos;
      cursorPos=levelIndents.back();
      levelIndents.pop_back();
      if (levelIndents.empty())
      {
         os << std::endl;
         requireSpaceBeforeTerm=false;
         currentIndent=0;
         cursorPos=0;
      }
   }

   void ListPrettyPrinter::updateIndent()
   {
      if (cursorPos<currentIndent)
      {
         //os << "Update indent[" << cursorPos << "," << currentIndent << ']';
         os << std::endl;
         currentIndent=cursorPos;
         printIndent();
      }
      else {
         if (requireSpaceBeforeTerm) (*this) << ' ';
         currentIndent=cursorPos;
      }
   }

   void ListPrettyPrinter::printIndent()
   {
      for (int i=0; i<currentIndent; ++i) os << ' ';
      cursorPos=currentIndent;
   }

   ListPrettyPrinter &ListPrettyPrinter::operator << (char c)
   {
      os << c;
      ++cursorPos;
      return *this;
   }

   ListPrettyPrinter &ListPrettyPrinter::operator << (int c)
   {
      os << c;
      ++cursorPos;
      return *this;
   }

   ListPrettyPrinter &ListPrettyPrinter::operator << (int64_t c)
   {
      os << c;
      ++cursorPos;
      return *this;
   }

   ListPrettyPrinter &ListPrettyPrinter::operator << (double c)
   {
      os << c;
      ++cursorPos;
      return *this;
   }

   ListPrettyPrinter &ListPrettyPrinter::operator << (const std::string &str)
   {
      os << str;
      cursorPos+=str.size();
      return *this;
   }

   ListPrettyPrinter &ListPrettyPrinter::operator << (const char *str)
   {
      os << str;
      cursorPos+=strlen(str);
      return *this;
   }

}
