/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "FlexibleSQLLayoutInternals.h"
#include "AQLException.h"
#include "AQLModel.h"
#include "AQLSupport.h"
#include "AQLToSQLTranslator.h"
#include "FlexibleSQLLayout.h"
#include "Messages.h"
#include "SQLInsertBuilder.h"
#include "FlexibleSQLLayoutDebug.h"
#include "SQLBackend.h"
#include "SQLBackendFunctions.h"

namespace {

   using namespace TypeRQ;
   using namespace TypeRQInternal;


   std::string getTripleReferenceColumn(const FlexiblePropertyLayout &fpl)
   {
      for (int j=0; j<AQLTypeSet::EXPRTYPE_SIZE; ++j)
      {
         const FlexibleExprTypeLayout &fetl=fpl.exprTypeLayouts[j];
         for (FlexibleExprTypeLayout::list_type::const_iterator i=fetl.sqlColRefs.begin();
              i!=fetl.sqlColRefs.end(); ++i)
         {
            const FlexibleSqlColumnReference &fscr=*i;
            if (fscr.referenceType==FlexibleSqlColumnReference::INDEX)
               return fpl.tripleTableColumn;
         }
      }
      return "";
   }

   ValueUpdatePlan::inner_list_type flexibleExprTypeLayoutToUpdateList(const FlexibleExprTypeLayout &fetl)
   {
      ValueUpdatePlan::inner_list_type ret;

      for (FlexibleExprTypeLayout::list_type::const_iterator i=fetl.sqlColRefs.begin();
           i!=fetl.sqlColRefs.end(); ++i)
      {
         const FlexibleSqlColumnReference &fscr=*i;
         ret.push_back(TableUpdate(fscr.referenceType==FlexibleSqlColumnReference::INDEX? fscr.table : std::string(""),
                                   fscr.referenceType==FlexibleSqlColumnReference::INDEX? fscr.indexColumn : std::string(""),
                                   fscr.valueColumn,
                                   fscr.valueColumnType,
                                   fscr.valueColumnSize));
      }
      return ret;
   }

   void findAllRequiredUpdates(AQLTypeSet::ExprType origExprType, const FlexiblePropertyLayout &fpl,
                               ValueUpdatePlan &plan)
   {
      for (AQLTypeSet::ExprType i=origExprType; i!=AQLTypeSet::UNSET; i=getNextTypePromotion(i))
      {
         const ValueUpdatePlan::inner_list_type &alternatives=
            flexibleExprTypeLayoutToUpdateList(fpl.exprTypeLayouts[i]);

         plan.tableUpdates.push_back(alternatives);
      }
   }
}

namespace TypeRQ {

   using namespace TypeRQInternal;


   ValueUpdatePlan &FlexibleSQLLayout::getLiteralUpdatePlan(AQLPropertyExpr::Property p,
                                                            AQLTypeSet::ExprType literalType)
   {
      ValueUpdatePlan *plan=valueUpdatePlans[literalType][p];
      if (plan) return *plan;

      print(OL_DEBUG, "Creating update plan for %s:%s...\n",
            getNameForAQLTripleProperty(p), getNameForExprType(literalType));

      try {
         plan=new ValueUpdatePlan;

         findAllRequiredUpdates(literalType, layoutDescriptor.propertyLayouts[p], *plan);

         // plan ready, save it
         valueUpdatePlans[literalType][p]=plan;

         if (outputLevel>=OL_DEBUG)
         {
            dumpValueUpdatePlan(*plan, p, literalType);
         }

         return *plan;
      }
      catch (...) {
         delete plan;
         throw;
      }
   }

   void FlexibleSQLLayout::generateInsertBuildersForValue(int refId,
                                                          const AQLLiteralExpr &literal,
                                                          AQLPropertyExpr::Property p,
                                                          std::map<std::string, SQLInsertBuilder *> &insertBuilders)
   {
      ValueUpdatePlan &plan=getLiteralUpdatePlan(p, literal.literalType);

      // set of tablename,columnname
      std::set<std::pair<std::string, std::string> > alreadyInserted;

      // Note: One update of every outer list must be performed. The first
      // possible of the inner list will be used, unless the inner list is skipped altogether
      for (ValueUpdatePlan::outer_list_type::const_iterator i=plan.tableUpdates.begin();
           i!=plan.tableUpdates.end(); ++i)
      {
         const ValueUpdatePlan::inner_list_type &innerList=*i;

         // First, see if we need to consider this inner list. We don't,
         // if any of the updates in inner list is already updated (inner list skip)
         for (ValueUpdatePlan::inner_list_type::const_iterator j=innerList.begin();
              j!=innerList.end(); ++j)
         {
            const TableUpdate &tabU=*j;
            if (alreadyInserted.find(std::make_pair(tabU.table, tabU.valueColumn))!=alreadyInserted.end())
            {
               goto skip_update;
            }
         }

         // Iterate until good update statement is found
         for (ValueUpdatePlan::inner_list_type::const_iterator j=innerList.begin();
              j!=innerList.end(); ++j)
         {
            const TableUpdate &tabU=*j;

            SQLInsertBuilder *insertBuilder=insertBuilders[tabU.table];
            if (!insertBuilder)
            {
               insertBuilders[tabU.table]=insertBuilder=new SQLInsertBuilder(tabU.table);
               if (!tabU.indexColumn.empty())
               {
                  insertBuilder->addColumn(tabU.indexColumn, refId);
               }
            }

            // Check if this update can be used. It cannot, if we have value column size limit
            // and the literal wouldn't fit in that
            if (tabU.valueColumnSize>0)
            {
               switch (literal.literalType)
               {
                  case AQLTypeSet::IRI:
                  case AQLTypeSet::STRING:
                  case AQLTypeSet::DATETIME:
                     if (literal.stringLiteral.size()>tabU.valueColumnSize)
                     {
                        // ok, size overflow
                        goto next_try;
                     }
                     break;

                  default:
                     // we don't consider size limits for ints, booleans and doubles
                     break;

               }
            }

            if (alreadyInserted.find(std::make_pair(tabU.table, tabU.valueColumn))==alreadyInserted.end())
            {
               AQLExpr *aqlExpr=new AQLLiteralExpr(literal);

               AQLToSQLTranslator aqlToSql(sqlBackend.getFunctionMap());
               if (literal.literalType!=tabU.columnType)
               {
                  // literal type != column type => typecast needed
                  AQLFunctionExpr *fexpr=new AQLFunctionExpr;
                  fexpr->functionName=AQLToSQLTranslator::getFunctionNameForTypecast(tabU.columnType);
                  fexpr->arguments.push_back(aqlExpr);
                  aqlExpr=fexpr;

                  // SQLFunctionMapping &mapping=
                  //    sqlBackend.getFunctionMap().
                  //    getFunctionMapping(AQLToSQLTranslator::getFunctionNameForTypecast(tabU.columnType));
                  // mapping.prepare(*fexpr, sqlBackend.getFunctionMap());
               }


               insertBuilder->addColumn(tabU.valueColumn,
                                        aqlToSql.translateExprToSql(*aqlExpr, *this));
               alreadyInserted.insert(std::make_pair(tabU.table, tabU.valueColumn));

               delete aqlExpr;
            }

            break; // one successful update is enough

           next_try:;
            // this update wasn't successful (size limit exceeded), try next
         }

        skip_update:;
      }
   }


   void FlexibleSQLLayout::generateInsertLiteralSQL(int refId,
                                                    const AQLLiteralExpr &literal,
                                                    AQLPropertyExpr::Property p,
                                                    std::list<std::string> &sqlStatements)
   {
      // tablename -> insert builder
      std::map<std::string, SQLInsertBuilder *> insertBuilders;

      generateInsertBuildersForValue(refId, literal, p, insertBuilders);

      for (std::map<std::string, SQLInsertBuilder *>::iterator i=insertBuilders.begin();
           i!=insertBuilders.end(); ++i)
      {
         const std::string &tableName=i->first;
         SQLInsertBuilder &insertBuilder=*(i->second);

         if (tableName.empty())
         {
            // inline value
         }
         else {
            // referenced value
            sqlStatements.push_back(insertBuilder.buildInsertStatement());
         }

         delete &insertBuilder;
      }
   }

   TripleUpdatePlan &FlexibleSQLLayout::getTripleUpdatePlan()
   {
      if (tripleUpdatePlan) return *tripleUpdatePlan;

      TripleUpdatePlan *plan=0;

      print(OL_DEBUG, "Building triple update plan...\n");

      try {
         plan=new TripleUpdatePlan;

         for (int i=0; i<AQLPropertyExpr::PROPERTY_SIZE; ++i)
         {
            AQLPropertyExpr::Property const p=AQLPropertyExpr::Property(i);
            FlexiblePropertyLayout &fpl=layoutDescriptor.propertyLayouts[p];
            plan->propertyRefIds[p]=getTripleReferenceColumn(fpl);
         }

         tripleUpdatePlan=plan;
         return *tripleUpdatePlan;
      }
      catch (...) {
         delete plan;
         throw;
      }
   }

   void FlexibleSQLLayout::generateInsertTripleSQL(int tripleId,
                                                   const AQLLiteralExpr &s, int sRefId,
                                                   const AQLLiteralExpr &p, int pRefId,
                                                   const AQLLiteralExpr &o, int oRefId,
                                                   std::list<std::string> &sqlStatements)
   {
      TripleUpdatePlan &plan=getTripleUpdatePlan();

      std::map<std::string, SQLInsertBuilder *> insertBuilders;

      SQLInsertBuilder *tripleTableBuilder=new SQLInsertBuilder(layoutDescriptor.tripleTable);
      insertBuilders[""]=tripleTableBuilder;

      if (!plan.propertyRefIds[AQLPropertyExpr::SUBJECT].empty())
         tripleTableBuilder->addColumn(plan.propertyRefIds[AQLPropertyExpr::SUBJECT], sRefId);
      if (!plan.propertyRefIds[AQLPropertyExpr::PREDICATE].empty())
         tripleTableBuilder->addColumn(plan.propertyRefIds[AQLPropertyExpr::PREDICATE], pRefId);
      if (!plan.propertyRefIds[AQLPropertyExpr::OBJECT].empty())
         tripleTableBuilder->addColumn(plan.propertyRefIds[AQLPropertyExpr::OBJECT], oRefId);

      generateInsertBuildersForValue(sRefId, s, AQLPropertyExpr::SUBJECT, insertBuilders);
      generateInsertBuildersForValue(pRefId, p, AQLPropertyExpr::PREDICATE, insertBuilders);
      generateInsertBuildersForValue(oRefId, o, AQLPropertyExpr::OBJECT, insertBuilders);

      const std::string stm=tripleTableBuilder->buildInsertStatement();
      if (!stm.empty())
      {
         sqlStatements.push_back(stm);
      }
      for (std::map<std::string, SQLInsertBuilder *>::iterator i=insertBuilders.begin();
           i!=insertBuilders.end(); ++i)
      {
         SQLInsertBuilder *const insertBuilder=i->second;
         delete insertBuilder;
      }

   }
}
