/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <algorithm>
#include <iostream>

#include "AQLException.h"
#include "AQLSupport.h"
#include "FlexibleSQLLayout.h"
#include "FlexibleSQLLayoutInternals.h"
#include "FlexibleSQLLayoutDebug.h"
#include "Messages.h"
#include "AQLToSQLTranslator.h"
#include "StringUtils.h"
#include "SQLJoinBuilder.h"
#include "SQLBackend.h"
#include "AQLOptimizer.h"
#include "GeneralUtils.h"

namespace {

   using namespace TypeRQInternal;
   using namespace TypeRQ;


   bool needTypecastForAccessExpr(AQLPropertyExpr &propertyExpr, AQLTypeSet accessExprTypes)
   {
      switch (propertyExpr.getExprTypeSet().singularType())
      {
         case AQLTypeSet::ANY:
         case AQLTypeSet::UNSET:
         case AQLTypeSet::REFERENCE:
            // no typecast for these sources
            return false;

         default:
            AQLTypeSet tmp=propertyExpr.getExprTypeSet();
            tmp.addSubTypes();
            if (makeIntersection(tmp, accessExprTypes)==accessExprTypes)
            {
               // access expr is of property expr type or subtype, no typecast needed
               return false;
            }
            return true;
      }
   }


   /**
    * Adds not-null criteria for every property access that has more than one accessor in accessor
    * chain or accessor that might return null. E.g., string might have string table and bigstring table,
    * integers might be typecast from strings (null-possibility).
    *
    * Note that the implementation of not-null criteria adding must be in harmony with INNER/LEFT join
    * selection in later passes.
    */
   class PropertyValueRequirerVisitor : public AQLOptionalVisitor
   {
   private:
      FlexibleLayoutDescriptor &fld;

      void addNotNulls(AQLJoinGroupLike &jgl)
      {
         for (std::list<std::string>::iterator i=jgl.names.begin(); i!=jgl.names.end(); ++i)
         {
            const std::string &joinName=*i;
            for (int j=AQLPropertyExpr::SUBJECT; j<AQLPropertyExpr::PROPERTY_SIZE; ++j)
            {
               AQLPropertyExpr::Property p=static_cast<AQLPropertyExpr::Property>(j);
               AQLTypeSet types=jgl.inferenceMap.at(TripleProperty(joinName, p));
               print(OL_DEBUG, "Considering %s:%s [%s]...\n",
                     joinName.c_str(), getNameForAQLTripleProperty(p),
                     toString(types).c_str());

               AQLJunctionCriterion *junction=new AQLJunctionCriterion;
               junction->junctionType=AQLJunctionCriterion::DISJUNCTION;

               for (AQLTypeSet::iterator k=types.begin(); k!=types.end(); ++k)
               {
                  bool requireNotNull=true;
                  AQLTypeSet::ExprType exprType=*k;
                  print(OL_DEBUG, "- %s:", getNameForExprType(exprType));

                  if (types==AQLTypeSet::ANY)
                  {
                     // ANY type does not need not-null checks, because some type alternative must match
                     // anyways
                     requireNotNull=false;
                  }

                  std::list<SQLAccessExpr *> accesses;
                  AQLPropertyExpr propertyExpr(joinName, p, exprType);
                  determineRequiredSqlAccesses(propertyExpr, fld, accesses);

                  if (requireNotNull && types.size()==1)
                  {
                     // The only alternative. We don't need not-null, if no typecast is needed

                     if (accesses.size()==1)
                     {
                        SQLAccessExpr &accessExpr=*accesses.front();
                        if (accessExpr.getExprTypeSet()==exprType)
                        {
                           requireNotNull=false;
                        }
                     }
                  }

                  if (requireNotNull)
                  {
                     AQLFunctionExpr *notNull=new AQLFunctionExpr;
                     notNull->functionName="builtin:is-not-null";
                     notNull->functionType=AQLTypeSet::BOOLEAN;
                     notNull->arguments.push_back(new AQLPropertyExpr(joinName, p, exprType));
                     junction->terms.push_back(notNull);
                  }

                  std::for_each(accesses.begin(), accesses.end(), deleteObject<SQLAccessExpr>);
                  print(OL_DEBUG, "\n");
               }

               if (junction->terms.empty())
               {
                  delete junction;
               }
               else {
                  jgl.addCriterion(junction);
               }               
            }
         }
      }


   public:
      PropertyValueRequirerVisitor(FlexibleLayoutDescriptor &_fld) : fld(_fld) {}

      void visitBeforeChildren(AQLJoinGroup &join)
      {
         addNotNulls(join);
      }

      void visitBeforeChildren(AQLQuery &aql)
      {
         addNotNulls(aql);
      }
   };

   /**
    * This converts high-level property accesses to low-level property accesses.
    * Essentially, if some values are COALESCEd from multiple columns and/or typecasted from
    * another expressions, this rewriter unwinds those accesses.
    */
   class PropertyAccessNormalizer : public AQLOptionalExpressionRewriter
   {
   private:
      FlexibleLayoutDescriptor &fld;
   public:
      PropertyAccessNormalizer(FlexibleLayoutDescriptor &_fld) : fld(_fld) {}

      virtual AQLExpr *rewrite(AQLPropertyExpr &expr)
      {
         std::list<SQLAccessExpr *> accessList;
         determineRequiredSqlAccesses(expr, fld, accessList);

         std::list<AQLExpr *> paramList;

         // convert access list to param list
         for (std::list<SQLAccessExpr *>::iterator i=accessList.begin(); i!=accessList.end(); ++i)
         {
            SQLAccessExpr *accessExpr=*i;

            paramList.push_back(accessExpr);
         }

         AQLExpr *accessExpr;

         if (paramList.size()==1)
         {
            // no coalesce required
            accessExpr=paramList.front();
         }
         else {
            // coalesce
            AQLFunctionExpr *ret=new AQLFunctionExpr();
            ret->functionName="builtin:coalesce";
            ret->arguments=paramList;
            ret->functionType.clear();

            for (AQLFunctionExpr::arg_list_type::iterator i=ret->arguments.begin();
                 i!=ret->arguments.end(); ++i)
            {
               AQLExpr *arg=*i;
               ret->functionType.setTypes(arg->getExprTypeSet());
            }

            accessExpr=ret;
         }

         // add typecast if necessary
         if (needTypecastForAccessExpr(expr, accessExpr->getExprTypeSet()))
         {
            AQLFunctionExpr *typecastExpr=new AQLFunctionExpr;
            typecastExpr->functionName=
               AQLToSQLTranslator::getFunctionNameForTypecast(expr.getExprTypeSet().singularType());
            typecastExpr->arguments.push_back(accessExpr);
            typecastExpr->functionType=expr.getExprTypeSet().singularType();
            accessExpr=typecastExpr;
         }

         delete &expr;
         return accessExpr;
      }
   };

   class SQLReferenceCollector : public AQLOptionalVisitor
   {
   private:
      FlexibleLayoutDescriptor &fld;
      JoinGroupMap &joinGroupMap;

   public:
      SQLReferenceCollector(FlexibleLayoutDescriptor &_fld, JoinGroupMap &_joinGroupMap) :
         fld(_fld), joinGroupMap(_joinGroupMap)
      {
      }

      void visitBeforeChildren(AQLJoinGroup &join)
      {
         print(OL_DEBUG, "entering join:\n");
      }

      void visitBeforeChildren(AQLCustomExpr &customExpr)
      {
         if (dynamic_cast<SQLAccessExpr *>(&customExpr))
         {
            SQLAccessExpr &expr=static_cast<SQLAccessExpr &>(customExpr);

            print(OL_DEBUG, "Found reference: %s[%s].%s.%s\n",
                  expr.joinName.c_str(), expr.tripleTableColumn.c_str(),
                  expr.colRef.table.c_str(),
                  expr.colRef.valueColumn.c_str());
            joinGroupMap.ensureAccess(expr, fld);
            std::string accessString=joinGroupMap.getAccessString(expr);
            print(OL_DEBUG, " => %s\n",
                  accessString.c_str());

         }
         else {
            throw AQLException("Unknown custom expr: %s", customExpr.toString().c_str());
         }
      }
   };

}

namespace TypeRQ {

   using namespace TypeRQInternal;

   void FlexibleSQLLayout::prepareAQLForTranslationFinal(AQLQuery &aql)
   {
      // create join groups
      joinGroupMap->newJoinGroup(&aql, 0);

      PropertyValueRequirerVisitor pvrv(layoutDescriptor);
      aql.accept(pvrv);

      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL after property value requirer visitor:\n");
         AQLPrinterVisitor printerVisitor(std::cout);
         aql.accept(printerVisitor);
      }

      normalizeOperatorsToFunctions(aql, sqlBackend.getFunctionMap());

      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL after operators-to-functions:\n");
         AQLPrinterVisitor printerVisitor(std::cout);
         aql.accept(printerVisitor);
      }

      print(OL_DEBUG, "Rewriting property accesses as per backend layout...\n");
      PropertyAccessNormalizer pan(layoutDescriptor);
      AQLExpressionRewriter::walk(aql, pan);

      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL after PropertyAccessNormalizer:\n");
         AQLPrinterVisitor printerVisitor(std::cout);
         aql.accept(printerVisitor);
      }

      print(OL_DEBUG, "Optimizing expressions...\n");
      print(OL_DEBUG, "- simplify is-null coalesces\n");
      simplifyIsNullCoalesces(aql, sqlBackend.getFunctionMap());
      print(OL_DEBUG, "- simplify junctions\n");
      simplifyJunctions(aql, sqlBackend.getFunctionMap());
      print(OL_DEBUG, "- simplify expressions\n");
      simplifyExpressions(aql, sqlBackend.getFunctionMap());
      print(OL_DEBUG, "- perform CSE\n");
      performCSE(aql, sqlBackend.getFunctionMap());

      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL after expression optimizations:\n");
         AQLPrinterVisitor printerVisitor(std::cout);
         aql.accept(printerVisitor);
      }

      print(OL_DEBUG, "Choosing function variants...\n");
      chooseVariantOfFunctions(aql, sqlBackend.getFunctionMap());
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL after chooseVariantOfFunctions:\n");
         AQLPrinterVisitor printerVisitor(std::cout);
         aql.accept(printerVisitor);
      }

      addTypecastsWhereNeeded(aql, sqlBackend.getFunctionMap());
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL after addTypecastsWhereNeeded:\n");
         AQLPrinterVisitor printerVisitor(std::cout);
         aql.accept(printerVisitor);
      }

      // Collect all property references, required in AQL-to-SQL pass. This pass
      // needs to know which properties are referenced, and how (reference, typed value access, ...)
      // so that necessary SQL joins can be constructed.
      SQLReferenceCollector referenceCollector(layoutDescriptor, *joinGroupMap);
      aql.accept(referenceCollector);
   }

   std::string FlexibleSQLLayout::translateCustomAQLExpr(AQLCustomExpr &customExpr)
   {
      SQLAccessExpr &expr=static_cast<SQLAccessExpr &>(customExpr);
      return joinGroupMap->getAccessString(expr);
   }


   std::string FlexibleSQLLayout::createSQLForAQLJoinGroupTree(AQLQuery &aql, const join_condition_map &sqlConditions)
   {
      std::stringstream ss;
      FlexibleTripleJoinGroup &ftjg=*joinGroupMap->joinMap.at(0);

      ss << "FROM ";

      JoinExpression joinExpression;
      createJoinExpressionForAqlJoin(&aql, ftjg, sqlConditions, joinExpression);
      printJoinExpression(ss, joinExpression);

      return ss.str();
   }

   void
   FlexibleSQLLayout::createJoinExpressionForAqlJoin(AQLJoinGroupLike *aqlJoin,
                                                     TypeRQInternal::FlexibleTripleJoinGroup &ftjg,
                                                     const join_condition_map &sqlConditions,
                                                     JoinExpression &joinExpression)
   {
      for (AQLJoinGroupLike::name_list::iterator i=aqlJoin->names.begin();
           i!=aqlJoin->names.end(); ++i)
      {
         JoinTerm term;
         term.table=layoutDescriptor.tripleTable;
         term.alias=*i;
         joinExpression.joinTerms.push_back(term);
      }

      // add triple value joins
      for (FlexibleTripleJoinGroup::triple_join_map::iterator i=ftjg.tripleJoins.begin();
           i!=ftjg.tripleJoins.end(); ++i)
      {
         const std::string &joinName=i->first;
         FlexibleTripleJoin &ftj=i->second;

         for (FlexibleTripleJoin::map_type::iterator j=ftj.propertyJoins.begin();
              j!=ftj.propertyJoins.end(); ++j)
         {
            FlexiblePropertyJoin &fpj=j->second;

            for (FlexiblePropertyJoin::list_type::iterator k=fpj.usedJoins.begin();
                 k!=fpj.usedJoins.end(); ++k)
            {
               SQLJoin &sqlJoin=*k;
               InnerJoinExpression *valueJoinExpression=new InnerJoinExpression;
               joinExpression.nestedJoins.push_back(valueJoinExpression);
               JoinTerm term;
               term.table=sqlJoin.tableName;
               term.alias=joinName+"_"+sqlJoin.tripleTableColumn+"_"+sqlJoin.tableName;
               valueJoinExpression->joinTerms.push_back(term);
               valueJoinExpression->isLeftOuter=sqlJoin.leftJoin;
               valueJoinExpression->joinCondition=
                  term.alias + "." + sqlJoin.indexColumn + "=" +
                  joinName+"."+sqlJoin.tripleTableColumn;
            }
         }
      }

      // add nested triple joins
      for (AQLJoinGroupLike::nested_join_list::iterator i=aqlJoin->nestedJoins.begin();
           i!=aqlJoin->nestedJoins.end(); ++i)
      {
         AQLJoinGroup *const nestedJoin=*i;

         InnerJoinExpression *innerJoinExpression=new InnerJoinExpression;
         joinExpression.nestedJoins.push_back(innerJoinExpression);
         innerJoinExpression->isLeftOuter=nestedJoin->joinType==AQLJoinGroup::LEFT_OUTER;
         innerJoinExpression->joinCondition=sqlConditions.at(nestedJoin);
         if (innerJoinExpression->joinCondition.empty())
         {
            innerJoinExpression->joinCondition="TRUE";
         }
         createJoinExpressionForAqlJoin(nestedJoin, *ftjg.nestedJoinGroups.at(nestedJoin),
                                        sqlConditions, *innerJoinExpression);
      }
   }
}
