/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <map>
#include <string>
#include <set>
#include <sstream>

#include "AQLModel.h"
#include "FlexibleSQLLayout.h"

namespace TypeRQInternal {

   using namespace TypeRQ;

   struct TableUpdate : protected AQLDebugBase
   {
      std::string table;
      std::string indexColumn; // empty for INLINEs
      std::string valueColumn;
      AQLTypeSet::ExprType columnType; // used to indicate need for typecasting, e.g., converting string to int
      size_t valueColumnSize; // 0 for no size limits (default), only used for string and IRI

      TableUpdate(const std::string &_table, const std::string &_indexColumn, const std::string &_valueColumn,
                  AQLTypeSet::ExprType _columnType, size_t _valueColumnSize) :
         table(_table), indexColumn(_indexColumn), valueColumn(_valueColumn), columnType(_columnType),
         valueColumnSize(_valueColumnSize) {}

      TableUpdate(const TableUpdate &tabU) :
         table(tabU.table), indexColumn(tabU.indexColumn), valueColumn(tabU.valueColumn), columnType(tabU.columnType),
         valueColumnSize(tabU.valueColumnSize) {}
   };

   // update plan when we're storing node values using references
   // This is property/exprtype specific
   struct ValueUpdatePlan
   {
      typedef std::list<TableUpdate> inner_list_type;
      typedef std::list<inner_list_type> outer_list_type;

      // The semantics:
      // 1) When updating a value, perform one update per outer list element
      // 2) For each outer list element (list also), perform the first possible update
      //    specified by the element (l. inner list)
      //    - It is an error, if inner list update chain contains no possible updates
      // The reasoning here is the following:
      // - When inserting a value (e.g. double), we may need to perform multiple inserts,
      //   e.g., insert double, insert string, ...
      // - Some specific values (such as strings) may be stored in multiple tables (small_strings,
      //   big_strings), and there, we need to choose the first table update that is possible.
      outer_list_type tableUpdates;

      // Conveniency function for building tableUpdates. If newOuterListElement is false,
      // the TableUpdate is appended to tableUpdates.back(). Otherwise, new outer list element
      // is created.
      void addTableColumnUpdate(const std::string &table, const std::string &indexColumn,
                                const std::string &valueColumn, AQLTypeSet::ExprType valueType,
                                size_t valueColumnSize, bool newOuterListElement);
   };

   // update plan when we're storing triples
   struct TripleUpdatePlan
   {
      // ids of property reference columns, or empty strings if not used
      std::string propertyRefIds[AQLPropertyExpr::PROPERTY_SIZE];

      // NOTE: For inline values, find value update plan and select updates that have empty indexColumn
   };

   struct SQLAccessExpr : public AQLCustomExpr
   {
      std::string joinName;
      const FlexibleSqlColumnReference &colRef;
      std::string tripleTableColumn;
      AQLPropertyExpr::Property property;
      AQLTypeSet::ExprType exprType;

      SQLAccessExpr(const std::string &_joinName, const FlexibleSqlColumnReference &_colRef,
                    const std::string &_tripleTableColumn, AQLPropertyExpr::Property _property,
                    AQLTypeSet::ExprType _exprType);

      void accept(AQLVisitor &v);
      const char *getTypeName() const;
      AQLTypeSet getExprTypeSet() const;
      void unlinkChildren();
      std::string toString() const;
      bool equals(const AQLExpr &) const;
   };

   struct SQLJoin
   {
      std::string tableName;
      std::string indexColumn; // in value table
      std::string tripleTableColumn; // index in triple table
      bool leftJoin;
      std::set<std::string> usedValueColumns; // value columns used

      SQLJoin() {}

      SQLJoin(const SQLJoin &_join) : tableName(_join.tableName), indexColumn(_join.indexColumn),
                                      tripleTableColumn(_join.tripleTableColumn),
                                      leftJoin(_join.leftJoin),
                                      usedValueColumns(_join.usedValueColumns)
      {}

   private:
      SQLJoin &operator = (const SQLJoin &_join);

   };

   struct FlexiblePropertyJoin
   {
      typedef std::list<SQLJoin> list_type;
      list_type usedJoins;

      SQLJoin *findSQLJoin(const std::string &tableName, const std::string &indexColumn);
   };

   struct FlexibleTripleJoin
   {
      // key is triple table column (e.g. subj of table Triples)
      typedef std::map<std::string, FlexiblePropertyJoin> map_type;
      map_type propertyJoins;
   };

   // FlexibleTripleJoinGroup is SQL-side of AQL join group. It tracks
   // what property joins are used for which triples.
   //
   // corresponds AQLJoinGroup
   class FlexibleTripleJoinGroup : protected AQLDebugBase
   {
   public:
      typedef std::map<std::string, FlexibleTripleJoin> triple_join_map;
      triple_join_map tripleJoins;

      AQLJoinGroupLike &jgl;

      bool leftOuter;

      typedef std::map<AQLJoinGroup *, FlexibleTripleJoinGroup *> nested_join_map;
      nested_join_map nestedJoinGroups;

      FlexibleTripleJoinGroup *const parent; // used to create access aliases

      FlexibleTripleJoinGroup(AQLJoinGroupLike &jgl, FlexibleTripleJoinGroup *parent);
      ~FlexibleTripleJoinGroup();
      void ensureLowLevelAccess(SQLAccessExpr &expr, const FlexibleLayoutDescriptor &);
   };

   struct JoinGroupMap
   {
      typedef std::map<AQLJoinGroup *, FlexibleTripleJoinGroup *> join_group_map;
      join_group_map joinMap;

      typedef std::map<std::string, FlexibleTripleJoinGroup *> join_name_map;
      join_name_map tripleJoinNameMap;

      JoinGroupMap();
      ~JoinGroupMap();

      // if aqlJoin==0, this is root join
      FlexibleTripleJoinGroup *newJoinGroup(AQLJoinGroupLike *aqlJoin, FlexibleTripleJoinGroup *parent);

      // "Touches" property access, i.e., records access. Access records are
      // used to determine which joins are really required when FROM, LEFT JOIN, and INNER JOINs
      // are written in the final SQL.
      void ensureAccess(SQLAccessExpr &, const FlexibleLayoutDescriptor &);

      // returns textual SQL access expression
      std::string getAccessString(const SQLAccessExpr &);
   };

   void determineRequiredSqlAccesses(const AQLPropertyExpr &, const FlexibleLayoutDescriptor &,
                                     std::list<SQLAccessExpr *> &);
}
