/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <exception>
#include <sstream>

#include "AQLException.h"
#include "FlexibleSQLLayoutInternals.h"
#include "FlexibleSQLLayoutDebug.h"
#include "Messages.h"
#include "AQLSupport.h"

namespace {

   using namespace TypeRQ;
   using namespace TypeRQInternal;

   bool accessAlreadyExists(const FlexibleSqlColumnReference &fscr,
                            const std::list<SQLAccessExpr *> &accesses)
   {
      // first: check that access is not already added
      for (std::list<SQLAccessExpr *>::const_iterator k=accesses.begin(); k!=accesses.end(); ++k)
      {
         const SQLAccessExpr &alreadyAdded=*(*k);
         // check if reference types are the same
         if (alreadyAdded.colRef.referenceType==fscr.referenceType)
         {
            switch (fscr.referenceType)
            {
               case FlexibleSqlColumnReference::INLINE:
               {
                  if (alreadyAdded.colRef.valueColumn==fscr.valueColumn)
                     return true;
               }

               case FlexibleSqlColumnReference::INDEX:
               {
                  if (alreadyAdded.colRef.valueColumn==fscr.valueColumn &&
                      alreadyAdded.colRef.table==fscr.table)
                     return true;
               }
            }
         }
      }
      return false; // no same access found
   }
}

namespace TypeRQInternal {

   using namespace TypeRQ;

   void determineRequiredSqlAccesses(const AQLPropertyExpr &expr, const FlexibleLayoutDescriptor &fld,
                                     std::list<SQLAccessExpr *> &accesses)
   {
      const FlexiblePropertyLayout &fpl=fld.propertyLayouts[expr.property];
      AQLTypeSet exprTypes=expr.getExprTypeSet();
      for (AQLTypeSet::iterator i=exprTypes.begin(); i!=exprTypes.end(); ++i)
      {
         AQLTypeSet::ExprType exprType=*i;
         const FlexibleExprTypeLayout &fetl=fpl.exprTypeLayouts[exprType];

         for (FlexibleExprTypeLayout::list_type::const_iterator j=fetl.sqlColRefs.begin();
              j!=fetl.sqlColRefs.end(); ++j)
         {
            const FlexibleSqlColumnReference &fscr=*j;

            if (accessAlreadyExists(fscr, accesses)) continue; // skip add update

            SQLAccessExpr *access=
               new SQLAccessExpr(expr.joinName, fscr, fpl.tripleTableColumn, expr.property, fscr.valueColumnType);
            accesses.push_back(access);
         }
      }
   }

   void ValueUpdatePlan::addTableColumnUpdate(const std::string &table, const std::string &indexColumn,
                                              const std::string &valueColumn, AQLTypeSet::ExprType valueType,
                                              size_t valueColumnSize, bool newOuterListElement)
   {
      if (newOuterListElement)
      {
         tableUpdates.push_back(inner_list_type());
      }

      tableUpdates.back().push_back(TableUpdate(table, indexColumn, valueColumn, valueType,
                                                valueColumnSize));
   }

   SQLAccessExpr::SQLAccessExpr(const std::string &_joinName, const FlexibleSqlColumnReference &_colRef,
                                const std::string &_tripleTableColumn, AQLPropertyExpr::Property _property,
                                AQLTypeSet::ExprType _exprType) :
      joinName(_joinName), colRef(_colRef), tripleTableColumn(_tripleTableColumn),
      property(_property), exprType(_exprType)
   {}

   void SQLAccessExpr::accept(AQLVisitor &v)
   {
      v.visitBeforeChildren(*this);
      v.visitAfterChildren(*this);
   }

   const char *SQLAccessExpr::getTypeName() const
   {
      return "FlexibleSQLLayout::SQLAccessExpr";
   }

   AQLTypeSet SQLAccessExpr::getExprTypeSet() const
   {
      return AQLTypeSet(exprType);
   }

   void SQLAccessExpr::unlinkChildren() {} // no children

   std::string SQLAccessExpr::toString() const
   {
      std::stringstream ss;

      ss << getTypeName();
      ss << " ";
      switch (colRef.referenceType)
      {
         case FlexibleSqlColumnReference::INLINE:
            ss << "INLINE " << colRef.valueColumn;
            break;
         case FlexibleSqlColumnReference::INDEX:
            ss << "INDEX ";
            ss << colRef.table;
            ss << ".{ix: ";
            ss << colRef.indexColumn;
            ss << ", value: ";
            ss << colRef.valueColumn;
            ss << ", triple-table-column: ";
            ss << tripleTableColumn;
            ss << '(' << getNameForAQLTripleProperty(property);
            ss << ")}";
            break;
      }
      ss << " USING JOIN ";
      ss << joinName;
      return ss.str();
   }

   bool SQLAccessExpr::equals(const AQLExpr &o) const
   {
      const SQLAccessExpr *other=dynamic_cast<const SQLAccessExpr *>(&o);
      if (!other) return false;

      if (joinName!=other->joinName) return false;
      if (property!=other->property) return false;

      if (exprType!=other->exprType) return false;

      if (colRef.referenceType!=other->colRef.referenceType) return false;
      switch (colRef.referenceType)
      {
         case FlexibleSqlColumnReference::INLINE:
            if (colRef.valueColumn!=other->colRef.valueColumn) return false;
            break;

         case FlexibleSqlColumnReference::INDEX:
            if (colRef.table!=other->colRef.table) return false;
            if (colRef.indexColumn!=other->colRef.indexColumn) return false;
            if (colRef.valueColumn!=other->colRef.valueColumn) return false;
            if (tripleTableColumn!=other->tripleTableColumn) return false;
            break;
      }

      return true;
   }

   SQLJoin *FlexiblePropertyJoin::findSQLJoin(const std::string &tableName, const std::string &indexColumn)
   {
      for (list_type::iterator i=usedJoins.begin(); i!=usedJoins.end(); ++i)
      {
         SQLJoin &sqlJoin=*i;
         if (sqlJoin.tableName==tableName && sqlJoin.indexColumn==indexColumn)
            return &sqlJoin;
      }
      return 0; // not found
   }

   FlexibleTripleJoinGroup::FlexibleTripleJoinGroup(AQLJoinGroupLike &_jgl, FlexibleTripleJoinGroup *_parent) :
      jgl(_jgl), parent(_parent) {}

   FlexibleTripleJoinGroup::~FlexibleTripleJoinGroup()
   {
      for (nested_join_map::iterator i=nestedJoinGroups.begin(); i!=nestedJoinGroups.end(); ++i)
      {
         FlexibleTripleJoinGroup *const nestedFtjg=i->second;
         delete nestedFtjg;
      }
   }

   void FlexibleTripleJoinGroup::ensureLowLevelAccess(SQLAccessExpr &expr, const FlexibleLayoutDescriptor &fld)
   {
      FlexibleTripleJoin &ftj=tripleJoins.at(expr.joinName);
      FlexiblePropertyJoin &fpj=ftj.propertyJoins[expr.tripleTableColumn];

      if (expr.colRef.referenceType==FlexibleSqlColumnReference::INLINE)
      {
         // it's enough for use to have referenced ftj.propertyJoins[...]
         return;
      }

      SQLJoin *psqlJoin=fpj.findSQLJoin(expr.colRef.table, expr.colRef.indexColumn);
      if (psqlJoin)
      {
         SQLJoin &sqlJoin=*psqlJoin;

         sqlJoin.usedValueColumns.insert(expr.colRef.valueColumn);
         return; // found
      }

      SQLJoin sqlJoin;
      sqlJoin.tableName=expr.colRef.table;
      sqlJoin.indexColumn=expr.colRef.indexColumn;
      sqlJoin.tripleTableColumn=expr.tripleTableColumn;
      sqlJoin.usedValueColumns.insert(expr.colRef.valueColumn);

      // Determine whether we need left join or if inner is sufficient.
      // Inner is sufficient, if only 1 table is joined per inferred types.

      std::set<std::string> joinTables;
      const AQLTypeSet inferredTypes=
         jgl.inferenceMap.at(TripleProperty(expr.joinName, expr.property));

      for (AQLTypeSet::iterator i=inferredTypes.begin(); i!=inferredTypes.end(); ++i)
      {
         AQLTypeSet::ExprType exprType=*i;
         const FlexibleExprTypeLayout &fetl=
            fld.propertyLayouts[expr.property].exprTypeLayouts[exprType];
         for (FlexibleExprTypeLayout::list_type::const_iterator j=fetl.sqlColRefs.begin();
              j!=fetl.sqlColRefs.end(); ++j)
         {
            const FlexibleSqlColumnReference &fscr=*j;
            if (fscr.referenceType==FlexibleSqlColumnReference::INDEX)
            {
               joinTables.insert(fscr.table);
            }
            else {
               joinTables.insert(""); // empty is as good as any for triple-table
            }
         }
      }

      sqlJoin.leftJoin=joinTables.size()>1;

      fpj.usedJoins.push_back(sqlJoin);
   }

   JoinGroupMap::JoinGroupMap() {}

   JoinGroupMap::~JoinGroupMap()
   {
      std::list<FlexibleTripleJoinGroup *> toDelete;
      for (join_group_map::iterator i=joinMap.begin(); i!=joinMap.end(); ++i)
      {
         FlexibleTripleJoinGroup *ftjg=i->second;
         if (ftjg->parent==0)
         {
            toDelete.push_back(ftjg);
         }
      }
      for (std::list<FlexibleTripleJoinGroup*>::iterator i=toDelete.begin(); i!=toDelete.end(); ++i)
      {
         FlexibleTripleJoinGroup *ftjg=*i;
         delete ftjg;         
      }
   }

   FlexibleTripleJoinGroup *JoinGroupMap::newJoinGroup(AQLJoinGroupLike *aqlJoin,
                                                       FlexibleTripleJoinGroup *parent)
   {
      FlexibleTripleJoinGroup *ftjg=new FlexibleTripleJoinGroup(*aqlJoin, parent);

      std::list<std::string> joinNames;

      if (dynamic_cast<AQLJoinGroup *>(aqlJoin))
      {
         ftjg->leftOuter=(static_cast<AQLJoinGroup *>(aqlJoin)->joinType==AQLJoinGroup::LEFT_OUTER);
      }
      else {
         // this is root join
         ftjg->leftOuter=false;
      }

      joinNames.insert(joinNames.end(), aqlJoin->names.begin(), aqlJoin->names.end());

      // root join goes as 0
      joinMap.insert(std::make_pair(dynamic_cast<AQLJoinGroup *>(aqlJoin), ftjg));

      for (std::list<std::string>::iterator i=joinNames.begin();
           i!=joinNames.end(); ++i)
      {
         const std::string &joinName=*i;
         tripleJoinNameMap.insert(std::make_pair(joinName, ftjg));

         // create entry to tripleJoins
         ftjg->tripleJoins.insert(std::make_pair(joinName, FlexibleTripleJoin()));
      }

      // add nested join groups
      for (AQLJoinGroup::nested_join_list::iterator i=aqlJoin->nestedJoins.begin();
           i!=aqlJoin->nestedJoins.end(); ++i)
      {
         AQLJoinGroup *nestedAqlJoin=*i;
         FlexibleTripleJoinGroup *nestedFtjg=newJoinGroup(nestedAqlJoin, ftjg);
         ftjg->nestedJoinGroups.insert(std::make_pair(nestedAqlJoin, nestedFtjg));
      }

      return ftjg;
   }

   void JoinGroupMap::ensureAccess(SQLAccessExpr &expr, const FlexibleLayoutDescriptor &fld)
   {
      FlexibleTripleJoinGroup *ftjg;

      try {
         ftjg=tripleJoinNameMap.at(expr.joinName);
      }
      catch (std::exception) {
         throw AQLException("Tried to use join '%s' before definition", expr.joinName.c_str());
      }

      ftjg->ensureLowLevelAccess(expr, fld);
   }

   std::string JoinGroupMap::getAccessString(const SQLAccessExpr &expr)
   {
      std::stringstream ss;
      if (expr.colRef.referenceType==FlexibleSqlColumnReference::INLINE)
      {
         ss << expr.joinName << '.' << expr.tripleTableColumn;
      }
      else {
         ss << expr.joinName << '_' << expr.tripleTableColumn << '_' << expr.colRef.table;
         ss << '.' << expr.colRef.valueColumn;
      }
      return ss.str();
   }
}
