/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <set>

#include "AQLException.h"
#include "AQLSupport.h"
#include "FlexibleSQLLayout.h"

namespace {

   using namespace TypeRQ;

   void addNativeInlineMapping(FlexibleLayoutDescriptor &fld,
                               AQLPropertyExpr::Property p,
                               AQLTypeSet::ExprType exprType,
                               const std::string &valueColumn,
                               size_t valueColumnSize=0)
   {
      FlexibleSqlColumnReference fscr;

      fscr.referenceType=FlexibleSqlColumnReference::INLINE;
      fscr.valueColumn=valueColumn;
      fscr.valueColumnSize=valueColumnSize;
      fscr.valueColumnType=exprType;

      fld.propertyLayouts[p].exprTypeLayouts[exprType].sqlColRefs.push_back(fscr);
   }

   void addNativeIndexMapping(FlexibleLayoutDescriptor &fld,
                              AQLPropertyExpr::Property p,
                              AQLTypeSet::ExprType exprType,
                              const std::string &table,
                              const std::string &indexColumn,
                              const std::string &valueColumn,
                              size_t valueColumnSize=0)
   {
      FlexibleSqlColumnReference fscr;
      fscr.referenceType=FlexibleSqlColumnReference::INDEX;
      fscr.table=table;
      fscr.indexColumn=indexColumn;
      fscr.valueColumn=valueColumn;
      fscr.valueColumnType=exprType;
      fscr.valueColumnSize=valueColumnSize;

      fld.propertyLayouts[p].exprTypeLayouts[exprType].sqlColRefs.push_back(fscr);
   }

   void addTypecastMapping(FlexibleLayoutDescriptor &fld,
                           AQLPropertyExpr::Property p,
                           AQLTypeSet::ExprType exprType,
                           AQLTypeSet::ExprType typecastFrom)
   {
      FlexibleExprTypeLayout &fetl=fld.propertyLayouts[p].exprTypeLayouts[exprType];
      FlexibleExprTypeLayout &typecastFromFetl=fld.propertyLayouts[p].exprTypeLayouts[typecastFrom];

      for (FlexibleExprTypeLayout::list_type::iterator i=typecastFromFetl.sqlColRefs.begin();
           i!=typecastFromFetl.sqlColRefs.end(); ++i)
      {
         if (i->valueColumnType!=typecastFrom) continue; // don't copy typecast values
         FlexibleSqlColumnReference fscr=*i;
         fscr.valueColumnType=typecastFrom;
         fetl.sqlColRefs.push_back(fscr);
      }
   }

   void addReferenceColumn(FlexibleLayoutDescriptor &fld,
                           AQLPropertyExpr::Property p,
                           const std::string &ref)
   {
      FlexiblePropertyLayout &fpl=fld.propertyLayouts[p];
      FlexibleSqlColumnReference fscr;
      fscr.referenceType=FlexibleSqlColumnReference::INLINE;
      fscr.valueColumn=fpl.tripleTableColumn;
      fscr.valueColumnType=AQLTypeSet::INTEGER;
      fpl.exprTypeLayouts[AQLTypeSet::REFERENCE].sqlColRefs.push_back(fscr);
   }                           

   void setTripleTableBasics(FlexibleLayoutDescriptor &fpl,
                             const std::string &table,
                             const std::string &sref,
                             const std::string &pref,
                             const std::string &oref)
   {
      fpl.tripleTable=table;
      fpl.propertyLayouts[AQLPropertyExpr::SUBJECT].tripleTableColumn=sref;
      fpl.propertyLayouts[AQLPropertyExpr::PREDICATE].tripleTableColumn=pref;
      fpl.propertyLayouts[AQLPropertyExpr::OBJECT].tripleTableColumn=oref;

      addReferenceColumn(fpl, AQLPropertyExpr::SUBJECT, sref);
      addReferenceColumn(fpl, AQLPropertyExpr::PREDICATE, pref);
      addReferenceColumn(fpl, AQLPropertyExpr::OBJECT, oref);
   }
}

namespace TypeRQ {

   void FlexibleSQLLayout::useSimpleInlineLayout(FlexibleLayoutDescriptor &fld)
   {
      fld=FlexibleLayoutDescriptor();
      setTripleTableBasics(fld, "InlinedTriples", "subj_value", "pred_value", "obj_value");

      for (int i=0; i<AQLPropertyExpr::PROPERTY_SIZE; ++i)
      {
         AQLPropertyExpr::Property p=static_cast<AQLPropertyExpr::Property>(i);

         addNativeInlineMapping(fld, p, AQLTypeSet::STRING, fld.propertyLayouts[p].tripleTableColumn);
         if (p!=AQLPropertyExpr::OBJECT)
         {
            // SUBJECT, PREDICATE
            addNativeInlineMapping(fld, p, AQLTypeSet::IRI, fld.propertyLayouts[p].tripleTableColumn);
         }
         else {
            // OBJECT
            addTypecastMapping(fld, p, AQLTypeSet::IRI, AQLTypeSet::STRING);
         }

         addTypecastMapping(fld, p, AQLTypeSet::INTEGER, AQLTypeSet::STRING);
         addTypecastMapping(fld, p, AQLTypeSet::DOUBLE, AQLTypeSet::STRING);
         addTypecastMapping(fld, p, AQLTypeSet::BOOLEAN, AQLTypeSet::STRING);
         addTypecastMapping(fld, p, AQLTypeSet::DATETIME, AQLTypeSet::STRING);
      }
  }

   void FlexibleSQLLayout::useSimpleIndexedLayout(FlexibleLayoutDescriptor &fld)
   {
      fld=FlexibleLayoutDescriptor();

      setTripleTableBasics(fld, "SI_Triples", "subj", "pred", "obj");

      for (int i=0; i<AQLPropertyExpr::PROPERTY_SIZE; ++i)
      {
         AQLPropertyExpr::Property p=static_cast<AQLPropertyExpr::Property>(i);
         addNativeIndexMapping(fld, p, AQLTypeSet::STRING, "SI_Nodes", "id", "str_value");
         if (p!=AQLPropertyExpr::OBJECT)
         {
            // SUBJECT, PREDICATE
            addNativeIndexMapping(fld, p, AQLTypeSet::IRI, "SI_Nodes", "id", "str_value");
         }
         else {
            // OBJECT
            addTypecastMapping(fld, p, AQLTypeSet::IRI, AQLTypeSet::STRING);
         }
         addTypecastMapping(fld, p, AQLTypeSet::INTEGER, AQLTypeSet::STRING);
         addTypecastMapping(fld, p, AQLTypeSet::DOUBLE, AQLTypeSet::STRING);
         addTypecastMapping(fld, p, AQLTypeSet::BOOLEAN, AQLTypeSet::STRING);
         addTypecastMapping(fld, p, AQLTypeSet::DATETIME, AQLTypeSet::STRING);
      }
   }

   void FlexibleSQLLayout::useVersatileIndexedLayout(FlexibleLayoutDescriptor &fld)
   {
      fld=FlexibleLayoutDescriptor();

      setTripleTableBasics(fld, "VI_Triples", "subj", "pred", "obj");

      for (int i=0; i<AQLPropertyExpr::PROPERTY_SIZE; ++i)
      {
         AQLPropertyExpr::Property p=static_cast<AQLPropertyExpr::Property>(i);
         addNativeIndexMapping(fld, p, AQLTypeSet::IRI, "VI_IRIs", "id", "iri_value");
         addNativeIndexMapping(fld, p, AQLTypeSet::STRING, "VI_Strings", "id", "str_value", 255);
         addNativeIndexMapping(fld, p, AQLTypeSet::STRING, "VI_BigStrings", "id", "text_value", 0);
         addNativeIndexMapping(fld, p, AQLTypeSet::INTEGER, "VI_Numbers", "id", "int_value");
         addNativeIndexMapping(fld, p, AQLTypeSet::DOUBLE, "VI_Numbers", "id", "double_value");
         addNativeIndexMapping(fld, p, AQLTypeSet::BOOLEAN, "VI_Numbers", "id", "boolean_value");
         addNativeIndexMapping(fld, p, AQLTypeSet::DATETIME, "VI_Datetimes", "id", "datetime_value");
      }
   }

   void FlexibleSQLLayout::useVersatileCompactLayout(FlexibleLayoutDescriptor &fld)
   {
      fld=FlexibleLayoutDescriptor();

      setTripleTableBasics(fld, "VC_Triples", "subj", "pred", "obj");

      for (int i=0; i<AQLPropertyExpr::PROPERTY_SIZE; ++i)
      {
         AQLPropertyExpr::Property p=static_cast<AQLPropertyExpr::Property>(i);
         addNativeIndexMapping(fld, p, AQLTypeSet::IRI, "VC_IRIs", "id", "iri_value");
         addNativeIndexMapping(fld, p, AQLTypeSet::STRING, "VC_Strings", "id", "str_value", 255);
         addNativeIndexMapping(fld, p, AQLTypeSet::STRING, "VC_BigStrings", "id", "text_value", 0);
         addNativeIndexMapping(fld, p, AQLTypeSet::INTEGER, "VC_Integers", "id", "int_value");
         addNativeIndexMapping(fld, p, AQLTypeSet::DOUBLE, "VC_Doubles", "id", "double_value");
         addNativeIndexMapping(fld, p, AQLTypeSet::BOOLEAN, "VC_Booleans", "id", "boolean_value");
         addNativeIndexMapping(fld, p, AQLTypeSet::DATETIME, "VC_Datetimes", "id", "datetime_value");

         addTypecastMapping(fld, p, AQLTypeSet::DOUBLE, AQLTypeSet::INTEGER);
      }
   }
}
