/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <list>
#include <map>
#include <set>
#include <string>
#include <sstream>

#include <stdio.h>
#include <stddef.h>

#include "AQLException.h"
#include "AQLModel.h"
#include "FlexibleSQLLayout.h"
#include "FlexibleSQLLayoutDebug.h"
#include "SQLBackend.h"
#include "Messages.h"

namespace {
   using namespace TypeRQ;

   struct SQLColumn
   {
      AQLTypeSet::ExprType type;
      size_t size;
      bool isIndexColumn;
      bool allowNull;

      SQLColumn() : type(), size() {}
      SQLColumn(AQLTypeSet::ExprType _type, size_t _size, bool _isIndexColumn) :
         type(_type), size(_size), isIndexColumn(_isIndexColumn), allowNull(false) {}
   };

   struct SQLTable
   {
      typedef std::map<std::string, SQLColumn> columns_type;
      columns_type columns;

      typedef std::list<std::string> column_order_list;
      column_order_list columnOrders;

      std::string primaryKeyColumn;

      // returns true if column was actually added
      bool ensureColumn(const std::string &col, const SQLColumn &sqlColumn, bool isPrimaryKey)
      {
         bool ret;
         AQLTypeSet::ExprType exprType=sqlColumn.type;

         if (exprType==AQLTypeSet::REFERENCE) exprType=AQLTypeSet::INTEGER;

         if (exprType==AQLTypeSet::STRING) exprType=AQLTypeSet::ANY;
         if (exprType==AQLTypeSet::IRI) exprType=AQLTypeSet::ANY;

         if (columns.find(col)==columns.end())
         {
            columns[col]=SQLColumn(exprType, sqlColumn.size, sqlColumn.isIndexColumn);
            columnOrders.push_back(col);
            ret=true; // column added
         }
         else {
            ret=false; // column not added (already found)
            if (columns.at(col).type!=exprType)
            {
               columns[col]=SQLColumn(AQLTypeSet::ANY, 0, sqlColumn.isIndexColumn);
            }
            else {
               // use bigger size
               size_t &size=columns.at(col).size;
               if (size>0)
               {
                  if (sqlColumn.size==0 || sqlColumn.size>size)
                  {
                     size=sqlColumn.size;
                  }
               }
            }
         }

         if (isPrimaryKey)
         {
            if (primaryKeyColumn.empty())
               primaryKeyColumn=col;
            else if (primaryKeyColumn!=col)
               primaryKeyColumn="##AMBIGUOUS##";
         }
         return ret;
      }
   };

   struct SQLSchema
   {
      typedef std::map<std::string, SQLTable> map_type;
      map_type tables;
   };

}


namespace TypeRQ {
   using namespace TypeRQInternal;


   std::string FlexibleSQLLayout::generateCreateSQL()
   {
      if (outputLevel>=OL_DEBUG)
      {
         dumpFlexibleLayoutDescriptor(layoutDescriptor);
      }

      SQLSchema schema;

      // populate all table entries

      // - first, ensure we have triple table
      schema.tables[layoutDescriptor.tripleTable];

      for (int i=0; i<AQLPropertyExpr::PROPERTY_SIZE; ++i)
      {
         FlexiblePropertyLayout &fpl=layoutDescriptor.propertyLayouts[i];

         for (int j=0; j<AQLTypeSet::EXPRTYPE_SIZE; ++j)
         {
            FlexibleExprTypeLayout &fetl=fpl.exprTypeLayouts[j];

            for (FlexibleExprTypeLayout::list_type::iterator k=fetl.sqlColRefs.begin();
                 k!=fetl.sqlColRefs.end(); ++k)
            {
               FlexibleSqlColumnReference &fscr=*k;
               if (fscr.valueColumnType!=static_cast<AQLTypeSet::ExprType>(j)) continue; // this is reused
               switch (fscr.referenceType)
               {
                  case FlexibleSqlColumnReference::INLINE:
                     schema.tables[layoutDescriptor.tripleTable].
                        ensureColumn(fscr.valueColumn, SQLColumn(AQLTypeSet::ExprType(j), fscr.valueColumnSize, true),
                                     false);
                     break;

                  case FlexibleSqlColumnReference::INDEX:
                     // ensure triple table has index column
                     schema.tables[layoutDescriptor.tripleTable].
                        ensureColumn(fpl.tripleTableColumn,
                                     SQLColumn(AQLTypeSet::INTEGER, 0, true), false);

                     // ensure value table has index column
                     schema.tables[fscr.table].
                        ensureColumn(fscr.indexColumn,
                                     SQLColumn(AQLTypeSet::INTEGER, 0, true), true);

                     // ensure value table has value column
                     schema.tables[fscr.table].
                        ensureColumn(fscr.valueColumn,
                                     SQLColumn(AQLTypeSet::ExprType(j), fscr.valueColumnSize, false),
                                     false);
                     break;

               }
            }
         }
      }

      // now, check which columns in tables might be NULL
      for (SQLSchema::map_type::iterator i=schema.tables.begin();
           i!=schema.tables.end();
           ++i)
      {
         SQLTable &table=i->second;
         int valueCols=0;
         for (SQLTable::column_order_list::iterator j=table.columnOrders.begin();
              j!=table.columnOrders.end();
              ++j)
         {
            const std::string &col=*j;
            const SQLColumn &columnDef=table.columns.at(col);
            if (!columnDef.isIndexColumn) ++valueCols;
         }

         if (valueCols>=2)
         {
            for (SQLTable::column_order_list::iterator j=table.columnOrders.begin();
                 j!=table.columnOrders.end();
                 ++j)
            {
               const std::string &col=*j;
               SQLColumn &columnDef=table.columns.at(col);
               if (!columnDef.isIndexColumn) columnDef.allowNull=true;
            }
         }
      }

      std::stringstream ss;

      // output drop statements for tables
      for (SQLSchema::map_type::iterator i=schema.tables.begin();
           i!=schema.tables.end();
           ++i)
      {
         const std::string &tableName=i->first;
         ss << "DROP TABLE " << tableName << ';' << std::endl;
      }
      ss << std::endl << std::endl;

      // finally, output all tables and columns
      for (SQLSchema::map_type::iterator i=schema.tables.begin();
           i!=schema.tables.end();
           ++i)
      {
         const std::string &tableName=i->first;
         SQLTable &table=i->second;

         ss << "CREATE TABLE " << tableName << " (\n";

         bool firstColumn=true;
         for (SQLTable::column_order_list::iterator j=table.columnOrders.begin();
              j!=table.columnOrders.end();
              ++j)
         {
            if (!firstColumn)
            {
               ss << ",\n";
            }

            const std::string &col=*j;
            const SQLColumn &columnDef=table.columns.at(col);
            ss << "  " << col << " ";

            ss << sqlBackend.getSQLNameForType(columnDef.type, columnDef.size);

            if (!columnDef.allowNull)
            {
               ss << " NOT NULL";
            }

            firstColumn=false;
         }

         // put primary keys and indexes
         if (tableName==layoutDescriptor.tripleTable)
         {
            bool usePrimaryKey=true;

            for (int k=AQLPropertyExpr::SUBJECT; k<=AQLPropertyExpr::OBJECT; ++k)
            {
               FlexibleExprTypeLayout &fetl=
                  layoutDescriptor.propertyLayouts[k].exprTypeLayouts[AQLTypeSet::ANY];

               for (FlexibleExprTypeLayout::list_type::iterator l=fetl.sqlColRefs.begin();
                    usePrimaryKey && l!=fetl.sqlColRefs.end(); ++l)
               {
                  FlexibleSqlColumnReference &fscr=*l;

                  // triple table: check if we can apply primary key
                  usePrimaryKey=fscr.referenceType==FlexibleSqlColumnReference::INDEX;
               }
            }

            if (usePrimaryKey)
            {
               ss << ",\n  PRIMARY KEY(";
               ss << layoutDescriptor.propertyLayouts[AQLPropertyExpr::SUBJECT].tripleTableColumn;
               ss << ", ";
               ss << layoutDescriptor.propertyLayouts[AQLPropertyExpr::PREDICATE].tripleTableColumn;
               ss << ", ";
               ss << layoutDescriptor.propertyLayouts[AQLPropertyExpr::OBJECT].tripleTableColumn;
               ss << ")";
            }
         }
         else {
            if (!table.primaryKeyColumn.empty())
            {
               ss << ",\n  PRIMARY KEY(";
               ss << table.primaryKeyColumn;
               ss << ")";
            }
         }

         ss << ')' << sqlBackend.getCapabilities().createTableSuffix << ";\n\n";
      }
      return ss.str();
   }


}
