/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <string>
#include <list>
#include <map>

#include <stddef.h>

#include "AQLDebug.h"
#include "AQLModel.h"
#include "SQLLayout.h"

namespace TypeRQInternal {
   struct ReferenceAccessMap;
   struct FlexibleTripleJoinGroup;
   struct ValueUpdatePlan;
   struct TripleUpdatePlan;
   struct JoinGroupMap;
}

namespace TypeRQ {

   struct SQLBackend;
   struct SQLInsertBuilder;
   struct JoinExpression;

   struct FlexibleSqlColumnReference
   {
      enum ReferenceType
      {
         INLINE, // inline in triple table
         INDEX, // simple index
      };

      ReferenceType referenceType;


      std::string table; // sql table where the values are found -- ignored if reference type == INLINE
      std::string indexColumn; // column for indices -- ignored if reference type == INLINE
      std::string valueColumn; // column for values
      size_t valueColumnSize; // 0 for no size limits (default)

      // column data type
      AQLTypeSet::ExprType valueColumnType;

      FlexibleSqlColumnReference() : valueColumnSize(0), valueColumnType(AQLTypeSet::UNSET) {}
   };

   struct FlexibleExprTypeLayout
   {
      typedef std::list<FlexibleSqlColumnReference> list_type;


      // If more than 1 element, columns are always LEFT JOINed and COALESCE is used
      // to select first column that has non-null value.
      //
      // Must contain at least 1 element
      list_type sqlColRefs;
   };

   struct FlexiblePropertyLayout
   {
      // used to reference any column with INDEX ref type. So, must not be empty
      // if any expr type layout has INDEX sql col ref
      std::string tripleTableColumn;

      // Type-specific layouts for storing property (i.e. node) data
      //
      // Simplest is: specify table and column for ANY
      // and use typecast (source ANY) for everything else
      FlexibleExprTypeLayout exprTypeLayouts[AQLTypeSet::EXPRTYPE_SIZE];
   };

   struct FlexibleLayoutDescriptor
   {
      std::string tripleTable; // SQL table for triples

      FlexiblePropertyLayout propertyLayouts[AQLPropertyExpr::PROPERTY_SIZE];
   };


   class FlexibleSQLLayout : public SQLLayout
   {
   protected:
      FlexibleLayoutDescriptor &layoutDescriptor;
      SQLBackend &sqlBackend;
      int maxPropertyAccessDepth;

      TypeRQInternal::JoinGroupMap *const joinGroupMap;

      TypeRQInternal::ValueUpdatePlan *
      valueUpdatePlans[AQLTypeSet::EXPRTYPE_SIZE][AQLPropertyExpr::PROPERTY_SIZE];

      TypeRQInternal::TripleUpdatePlan *tripleUpdatePlan;


      TypeRQInternal::ValueUpdatePlan &getLiteralUpdatePlan(AQLPropertyExpr::Property, AQLTypeSet::ExprType);
      TypeRQInternal::TripleUpdatePlan &getTripleUpdatePlan();

   public:
      FlexibleSQLLayout(FlexibleLayoutDescriptor &, SQLBackend &);
      virtual ~FlexibleSQLLayout();

      virtual void prepareAQLForTranslationFinal(AQLQuery &);

      virtual std::string translateCustomAQLExpr(AQLCustomExpr &);

      // if join==0, this is the root join
      virtual std::string createSQLForAQLJoinGroupTree(AQLQuery &aql, const join_condition_map &);

      virtual std::string getPropertyAccess(const std::string &joinName, AQLPropertyExpr::Property, AQLTypeSet);

      virtual std::string generateCreateSQL();

      virtual void getReferenceForValueSQL(const AQLLiteralExpr &, AQLPropertyExpr::Property,
                                           std::list<std::string> &queryStatements);

      virtual void generateInsertLiteralSQL(int refId, const AQLLiteralExpr &s,
                                            AQLPropertyExpr::Property,
                                            std::list<std::string> &sqlStatements);
      virtual void generateInsertTripleSQL(int tripleId,
                                           const AQLLiteralExpr &s, int sRefId,
                                           const AQLLiteralExpr &p, int pRefId,
                                           const AQLLiteralExpr &o, int oRefId,
                                           std::list<std::string> &sqlStatements);


      virtual std::string escapeString(const std::string &stringLiteral);

      virtual const SQLBackendCapabilities &getBackendCapabilities();
      virtual int getMaxPropertyAccessDepth() const;
      virtual void overrideMaxPropertyAccessDepth(int depth);

      static void useSimpleInlineLayout(FlexibleLayoutDescriptor &);
      static void useSimpleIndexedLayout(FlexibleLayoutDescriptor &);
      static void useVersatileIndexedLayout(FlexibleLayoutDescriptor &);
      static void useVersatileCompactLayout(FlexibleLayoutDescriptor &);

   protected:
      // creates reference, adds joins when necessary
      // returns expression for the reference
      std::string createSQLPropertyReference(TypeRQInternal::FlexibleTripleJoinGroup &join,
                                             AQLPropertyExpr::Property property,
                                             FlexibleSqlColumnReference &sqlColRef,
                                             AQLTypeSet::ExprType exprType,
                                             TypeRQInternal::ReferenceAccessMap &accessMap,
                                             int ordinal,
                                             bool leftJoinOnly);

      void getReferenceForValueSQLInternal(const AQLLiteralExpr &literal,
                                           AQLPropertyExpr::Property property,
                                           std::list<std::string> &sqlQueries,
                                           AQLTypeSet::ExprType typecastFrom);

      void createJoinExpressionForAqlJoin(AQLJoinGroupLike *aqlJoin,
                                          TypeRQInternal::FlexibleTripleJoinGroup &ftjg,
                                          const join_condition_map &sqlConditions,
                                          JoinExpression &toPopulate);

      void generateInsertBuildersForValue(int refId,
                                          const AQLLiteralExpr &literal,
                                          AQLPropertyExpr::Property p,
                                          std::map<std::string, SQLInsertBuilder *> &insertBuilders);

   };
}
