/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <map>
#include <set>
#include <vector>
#include <stdio.h>
#include <iostream>
#include <sstream>

#include "FlexibleSQLLayoutInternals.h"
#include "AQLException.h"
#include "AQLToSQLTranslator.h"
#include "FlexibleSQLLayout.h"
#include "AQLSupport.h"
#include "SQLBackend.h"
#include "Messages.h"
#include "SQLBackendFunctions.h"

namespace TypeRQInternal {

   using namespace TypeRQ;

}

namespace {
   using namespace TypeRQ;
   using namespace TypeRQInternal;


}

namespace TypeRQ {
   using namespace TypeRQInternal;

   FlexibleSQLLayout::FlexibleSQLLayout(FlexibleLayoutDescriptor &_layoutDescriptor, SQLBackend &_backend) :
      layoutDescriptor(_layoutDescriptor), sqlBackend(_backend),
      maxPropertyAccessDepth(1), // inherent to SQL engines and the layouts used
      joinGroupMap(new JoinGroupMap),
      valueUpdatePlans(), tripleUpdatePlan(0)
   {
   }

   FlexibleSQLLayout::~FlexibleSQLLayout()
   {
      delete joinGroupMap;

      for (int i=0; i<AQLPropertyExpr::PROPERTY_SIZE; ++i)
      {
         for (int j=0; j<AQLTypeSet::EXPRTYPE_SIZE; ++j)
         {
            delete valueUpdatePlans[j][i];
         }
      }
      delete tripleUpdatePlan;
   }

   std::string FlexibleSQLLayout::escapeString(const std::string &stringLiteral)
   {
      return sqlBackend.escapeString(stringLiteral);
   }

   std::string FlexibleSQLLayout::getPropertyAccess(const std::string &, AQLPropertyExpr::Property p,
                                                    AQLTypeSet t)
   {
      throw AQLException("FlexibleSQLLayout::getPropertyAccess(): All property accesses should have been already translated!");
   }

   void FlexibleSQLLayout::getReferenceForValueSQL(const AQLLiteralExpr &literal,
                                                   AQLPropertyExpr::Property property,
                                                   std::list<std::string> &sqlQueries)
   {
      FlexiblePropertyLayout &fpl=layoutDescriptor.propertyLayouts[property];
      FlexibleExprTypeLayout &fetl=fpl.exprTypeLayouts[literal.literalType];

      for (FlexibleExprTypeLayout::list_type::iterator i=fetl.sqlColRefs.begin();
           i!=fetl.sqlColRefs.end(); ++i)
      {
         FlexibleSqlColumnReference &fscr=*i;

         std::string sql;

         switch (fscr.referenceType)
         {
            case FlexibleSqlColumnReference::INLINE:
               sql="SELECT ";
               sql+=fpl.tripleTableColumn;
               sql+=" FROM ";
               sql+=layoutDescriptor.tripleTable;
               break;

            case FlexibleSqlColumnReference::INDEX:
               sql="SELECT ";
               sql+=fscr.indexColumn;
               sql+=" FROM ";
               sql+=fscr.table;
               break;

            default:
               throw AQLException("FlexibleSQLLayout::getReferenceForValueSQL: unhandled reference type %d",
                                  (int)fscr.referenceType);
         }

         sql+=" WHERE ";
         sql+=fscr.valueColumn;
         sql+="=";

         // append value, but handle case of type promotions
         if (literal.literalType!=fscr.valueColumnType)
         {
            // literal and value column do not share type => typecast needed
            AQLToSQLTranslator aqlToSql(sqlBackend.getFunctionMap());

            // literal type != column type => typecast needed
            AQLFunctionExpr *fexpr=new AQLFunctionExpr;
            fexpr->functionName=AQLToSQLTranslator::getFunctionNameForTypecast(fscr.valueColumnType);
            fexpr->arguments.push_back(new AQLLiteralExpr(literal));

            // SQLFunctionMapping &mapping=
            //    sqlBackend.getFunctionMap().
            //    getFunctionMapping(AQLToSQLTranslator::getFunctionNameForTypecast(fscr.valueColumnType));
            // mapping.prepare(*fexpr, sqlBackend.getFunctionMap());
            try {
               sql+=aqlToSql.translateExprToSql(*fexpr, *this);
            }
            catch (...) {
               delete fexpr;
               throw;
            }

            delete fexpr;
         }
         else {
            // same type
            AQLToSQLTranslator aqlToSql(sqlBackend.getFunctionMap());
            sql+=aqlToSql.aqlLiteralExprToSQL(literal, *this);
         }

         // be sure that returned column is not null
         sql+=" AND ";
         sql+=fscr.valueColumn;
         sql+=" IS NOT NULL";

         sqlQueries.push_back(sql);
      }
   }


   const SQLBackendCapabilities &FlexibleSQLLayout::getBackendCapabilities()
   {
      return sqlBackend.getCapabilities();
   }

   int FlexibleSQLLayout::getMaxPropertyAccessDepth() const
   {
      return maxPropertyAccessDepth;
   }

   void FlexibleSQLLayout::overrideMaxPropertyAccessDepth(int depth)
   {
      maxPropertyAccessDepth=depth;
   }
}
