/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <vector>
#include <limits>

#include "AQLDebug.h"
#include "AQLException.h"
#include "ExpressionWriter.h"

namespace {

   struct Context
   {
      std::string op;
      int precedence;
      bool isBinOp;
      bool isLeftAssociative;
      bool isCommutative;
      bool parenAfterOp;
      int terms;
   };

}

namespace TypeRQInternal {
   using namespace std;
   using namespace TypeRQ;

   struct ExpressionWriterContext : protected AQLDebugBase
   {
      std::vector<Context> contextStack;

      ExpressionWriterContext()
      {
         newContext("EXPRESSION WRITER STACK UNDERFLOW!", std::numeric_limits<int>::max(),
                    false, true, true, false);
      }

      ~ExpressionWriterContext()
      {
         popContext();
      }

      void newContext(const char *op, int precedence, bool isBinOp, bool isLeftAssociative, bool isCommutative,
                      bool parenAfterOp)
      {
         Context ctx;
         ctx.op=op;
         ctx.precedence=precedence;
         ctx.isBinOp=isBinOp;
         ctx.isLeftAssociative=isLeftAssociative;
         ctx.isCommutative=isCommutative;
         ctx.parenAfterOp=parenAfterOp;
         ctx.terms=1;
         contextStack.push_back(ctx);
      }

      void popContext()
      {
         contextStack.pop_back();
      }

      void nextTerm()
      {
         ++(contextStack.back().terms);
      }

      const Context &operator() ()
      {
         return contextStack.back();
      }
   };

}

namespace TypeRQ {
   using namespace TypeRQInternal;

   ExpressionWriter::ExpressionWriter(std::ostream &_os) :
      ctx(*new ExpressionWriterContext), os(_os)
   {}

   ExpressionWriter::~ExpressionWriter()
   {
      delete &ctx;
   }

   void ExpressionWriter::startUnOp(const char *op, int precedence, bool isLeftAssociative)
   {
      bool parens;
      if (ctx().precedence <= precedence)
      {
         // precedence is same or lower, parens always needed - for clarity, at least
         parens=true;
      }
      else // (ctx().precedence > precedence)
      {
         // precedence is higher, parens never needed
         parens=false;
      }

      if (parens) os << '(';

      if (!isLeftAssociative) os << op;

      ctx.newContext(op, precedence, false, isLeftAssociative, false, parens);
   }

   void ExpressionWriter::startBinOp(const char *op, int precedence, bool isLeftAssociative, bool isCommutative)
   {
      bool parens;
      if (ctx().precedence < precedence)
      {
         // precedence is lower, parens always needed
         parens=true;
      }
      else if (ctx().precedence > precedence)
      {
         // precedence is higher, parens never needed
         parens=false;
      }
      // below this point: precedence is equal
      else if (ctx().op==op)
      {
         parens=!isCommutative;
      }
      // non-commutativity below this point
      else if (isLeftAssociative && ctx().terms==1)
      {
         parens=false;
      }
      else {
         // no reason detected why parens could be omitted
         parens=true;
      }

      if (parens) os << '(';

      ctx.newContext(op, precedence, true, isLeftAssociative, isCommutative, parens);
   }

   void ExpressionWriter::startFunction(const std::string &fn)
   {
      os << fn << '(';
      ctx.newContext(",", std::numeric_limits<int>::max(), true, true, false, true);
   }

   void ExpressionWriter::endOp()
   {
      if (!ctx().isBinOp && ctx().isLeftAssociative)
      {
         os << ctx().op;
      }

      if (ctx().parenAfterOp) os << ')';
      ctx.popContext();
   }

   void ExpressionWriter::nextTerm()
   {
      os << ctx().op;
      ctx.nextTerm();
   }

   SQLExpressionWriter::SQLExpressionWriter(std::ostream &_os)
      : ExpressionWriter(_os)
   {
   }

   void SQLExpressionWriter::startDot()
   {
      startBinOp(".", 0, true, false);
   }

   void SQLExpressionWriter::startUnaryMinus()
   {
      startUnOp("-", 1, false);
   }

   void SQLExpressionWriter::startExp()
   {
      startBinOp("^", 2, true, false);
   }

   void SQLExpressionWriter::startMult()
   {
      startBinOp("*", 3, true, true);
   }

   void SQLExpressionWriter::startDiv()
   {
      startBinOp("/", 3, true, true);
   }

   void SQLExpressionWriter::startMod()
   {
      startBinOp("%", 3, true, false);
   }

   void SQLExpressionWriter::startPlus()
   {
      startBinOp("+", 4, true, true);
   }

   void SQLExpressionWriter::startMinus()
   {
      startBinOp("-", 4, true, false);
   }

   void SQLExpressionWriter::startUnaryIs(const char *op)
   {
      startUnOp(op, 5, true);
   }

   void SQLExpressionWriter::startConcat()
   {
      startBinOp("||", 6, true, true);
   }

   void SQLExpressionWriter::startIn()
   {
      startBinOp(" IN ", 7, false, false);
   }

   void SQLExpressionWriter::startBetween()
   {
      startBinOp(" BETWEEN ", 8, false, false);
   }

   void SQLExpressionWriter::startLike(const char *op)
   {
      startBinOp(op, 9, false, false);
   }

   void SQLExpressionWriter::startComparison(const char *op)
   {
      startBinOp(op, 10, false, false);
   }

   void SQLExpressionWriter::startEq()
   {
      startBinOp("=", 11, false, false);
   }

   void SQLExpressionWriter::startNot()
   {
      startUnOp("NOT ", 12, false);
   }

   void SQLExpressionWriter::startAnd()
   {
      startBinOp(" AND ", 13, true, true);
   }

   void SQLExpressionWriter::startOr()
   {
      startBinOp(" OR ", 14, true, true);
   }
}
