/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <algorithm>
#include <cstdlib>
#include <limits>
#include <map>
#include <set>
#include <string>
#include <sstream>
#include <iostream>

#include <cstring>

#include <stdio.h>

#include "AQLToSQLTranslator.h"
#include "AQLSupport.h"
#include "AQLException.h"
#include "SQLLayout.h"
#include "SQLBackend.h"
#include "SQLBackendFunctions.h"
#include "Messages.h"
#include "AQLOptimizer.h"
#include "ExpressionWriter.h"
#include "AQLTypeInference.h"

namespace {
   using namespace TypeRQ;

   // this is used for function -> SQL syntax transformations, such as concatenate(a,b) => a || b
   struct FunctionContext
   {
      SQLFunctionMapping &functionMapping;
      FunctionContext(SQLFunctionMapping &_functionMapping) : functionMapping(_functionMapping) {}
   };

   struct TranslatorContext
   {
      SQLLayout &sqlLayout;

      SQLFunctionMap &functionMap;

      std::list<std::stringstream *> queryStreamStack;
      std::list<SQLExpressionWriter *> ewStack;
      SQLExpressionWriter *query;
      std::stringstream *querySS;

      std::list<FunctionContext> functionContextStack;

      TranslatorContext(SQLLayout &_layout, SQLFunctionMap &_functionMap)
         : sqlLayout(_layout), functionMap(_functionMap), query(0), querySS(0)
      {
         pushNewQueryContext();
      }

      ~TranslatorContext()
      {
         popQueryContext();
      }

      void pushNewQueryContext()
      {
         if (query)
         {
            ewStack.push_back(query);
            queryStreamStack.push_back(querySS);
         }
         querySS=new std::stringstream;
         query=new SQLExpressionWriter(*querySS);
      }

      void popQueryContext()
      {
         if (!query) throw AQLException("Translator context: cannot pop query context from empty stack");
         delete query;
         delete querySS;

         if (!queryStreamStack.empty())
         {
            query=ewStack.back();
            ewStack.pop_back();
            querySS=queryStreamStack.back();
            queryStreamStack.pop_back();
         }
         else {
            query=0;
            querySS=0;
         }
      }
   };

   std::string integerToString(int i)
   {
      char buffer[16];
      snprintf(buffer, 16, "%d", i);
      buffer[15]='\0';
      return std::string(buffer);
   }

   class AQLToSQLVisitor : public AQLVisitor {
   private:
      TranslatorContext &context;
      std::vector<std::string> sqlColumnNames;
      int selects;

      SQLLayout::join_condition_map joinConditionMap;

   public:
      AQLToSQLVisitor(TranslatorContext &_context) : context(_context) {}

      void visitBetweenChildren(AQLSelect &, int) {}
      void visitBeforeChildren(AQLSort &) {}
      void visitBeforeChildren(AQLQuery &) {}
      void visitBeforeJoins(AQLQuery &) {}
      void visitAfterSelects(AQLQuery &) {}
      void visitAfterCriterion(AQLQuery &) {}
      void visitAfterSorts(AQLQuery &) {}
      void visitAfterChildren(AQLNotExpression &) {}

      void visitBeforeChildren(AQLTypecastExpression &)
      {
         throw AQLException("AQLToSQLVisitor::visitBeforeChildren(AQLToSQLVisitor &): typecasts should have been normalized into conversion function calls!");
      }

      void visitAfterChildren(AQLTypecastExpression &)
      {
         throw AQLException("AQLToSQLVisitor::visitAfterChildren(AQLToSQLVisitor &): typecasts should have been normalized into conversion function calls!");
      }


      void visitBeforeSelects(AQLQuery &q)
      {
         sqlColumnNames.resize(q.selects.size());
         for (size_t i=0; i<sqlColumnNames.size(); ++i)
         {
            std::stringstream ss;
            ss << 'c' << i;
            sqlColumnNames.at(i)=ss.str();
         }

         (*context.query) << "SELECT";
         if (q.distinct)
         {
            (*context.query) << " DISTINCT";
         }
         selects=0;
      }

      void visitAfterJoins(AQLQuery &aql)
      {
         if (selects==0) (*context.query) << " 1";
         (*context.query) << "\n  ";

         (*context.query) << 
            context.sqlLayout.createSQLForAQLJoinGroupTree(aql, joinConditionMap);
      }

      void visitBeforeChildren(AQLJoinGroup &join)
      {
         context.pushNewQueryContext();
      }

      void visitBeforeNestedJoins(AQLJoinGroup &join)
      {
      }

      void visitAfterChildren(AQLJoinGroup &join)
      {
         joinConditionMap[&join]=context.querySS->str();
         context.popQueryContext();
      }

      void visitBeforeChildren(AQLSelect &)
      {
         if (selects==0)
            (*context.query) << ' ';
         else
            (*context.query) << ",\n       ";
      }
      void visitAfterChildren(AQLSelect &)
      {
         (*context.query) <<  " AS " << sqlColumnNames.at(selects);
         ++selects;
      }


      void visitBeforeCriterion(AQLQuery &q)
      {
         if (q.criterion) (*context.query) << "\n WHERE ";
      }

      void visit(AQLPropertyExpr &expr)
      {
         (*context.query)
            << context.sqlLayout.getPropertyAccess(expr.joinName, expr.property, expr.getExprTypeSet());
      }
      void visit(AQLLiteralExpr &expr)
      {
         switch (expr.literalType)
         {
            case AQLTypeSet::IRI:
            case AQLTypeSet::STRING:
               (*context.query) << '\'' << context.sqlLayout.escapeString(expr.stringLiteral) << '\'';
               break;

            case AQLTypeSet::DATETIME:
            {
               // Datetime literals are converted from string
               AQLLiteralExpr lexpr(expr.stringLiteral, AQLTypeSet::STRING);
               AQLFunctionExpr fexpr;
               fexpr.functionName="builtin:to-datetime";
               fexpr.arguments.push_back(&lexpr);
               try {
                  fexpr.accept(*this);
               }
               catch (...) {
                  fexpr.unlinkChildren();
                  throw;
               }
               fexpr.unlinkChildren();
               break;
            }

            case AQLTypeSet::BOOLEAN:
               if (expr.booleanLiteral)
                  (*context.query) << "TRUE";
               else
                  (*context.query) << "FALSE";
               break;

            case AQLTypeSet::INTEGER: {
               std::ostringstream stm;
               stm << expr.intLiteral;
               (*context.query) << stm.str();
               break;
            }

            case AQLTypeSet::DOUBLE: {
               std::ostringstream stm;
               stm << expr.doubleLiteral;
               (*context.query) << stm.str();
               break;
            }

            default:
               throw AQLException("AQLToSQLTVisitor::visit(AQLLiteralExpr &): unhandled literal type %d",
                                  (int)expr.literalType);
         }
      }

      void visit(AQLNullExpr &)
      {
         (*context.query) << "NULL";
      }

      void visitBeforeChildren(AQLFunctionExpr &fexpr)
      {
         // push out function context
         SQLFunctionMapping &functionMapping=context.functionMap.getFunctionMapping(fexpr.functionName);
         FunctionContext functionContext(functionMapping);
         context.functionContextStack.push_back(functionContext);
         functionContext.functionMapping.startSQLExp(fexpr, *context.query);

         //context.pushNewQueryContext();
      }
      void visitBetweenChildren(AQLFunctionExpr &, int)
      {
         /*
         context.functionContextStack.back().arguments.push_back(context.query->str());
         context.popQueryContext();
         context.pushNewQueryContext();
         */
         context.query->nextTerm();
      }
      void visitAfterChildren(AQLFunctionExpr &fexpr)
      {
         FunctionContext &functionContext=context.functionContextStack.back();

         //if (!fexpr.arguments.empty())
         //   functionContext.arguments.push_back(context.query->str());

         //context.popQueryContext();

         // translate function
         functionContext.functionMapping.endSQLExp(fexpr, *context.query);

         // pop out function context
         context.functionContextStack.pop_back();
      }
      void visitBeforeChildren(AQLComparisonCriterion &)
      {
         throw AQLException("Internal error: Comparisons should have been normalized");
      }
      void visitBetweenChildren(AQLComparisonCriterion &comparison)
      {
      }
      void visitAfterChildren(AQLComparisonCriterion &)
      {
      }

      void visitBeforeChildren(AQLNotExpression &)
      {
         throw AQLException("Internal error: Logical negations (not) should have been normalized");
      }

      void visitBeforeChildren(AQLJunctionCriterion &junction)
      {
         throw AQLException("Internal error: Junctions should have been normalized");
      }
      void visitBetweenChildren(AQLJunctionCriterion &junction, int pos)
      {
      }
      void visitAfterChildren(AQLJunctionCriterion &)
      {
      }
      void visitAfterChildren(AQLSort &sort)
      {
         if (sort.ascending)
            (*context.query) << " ASC";
         else
            (*context.query) << " DESC";
      }
      void visitBeforeSorts(AQLQuery &query)
      {
         if (!query.sorts.empty())
            (*context.query) << "\nORDER BY ";
      }
      void visitBetweenSorts(AQLQuery &, int)
      {
         (*context.query) << ", ";
      }
      void visitAfterChildren(AQLQuery &query)
      {
         if (query.maxRows>=0)
         {
            (*context.query) << "\nLIMIT ";
            (*context.query) << integerToString(query.maxRows);
         }
         if (query.rowOffset >= 0)
         {
            (*context.query) << "\nOFFSET ";
            (*context.query) << integerToString(query.rowOffset);
         }
      }

      void visitBeforeChildren(AQLCustomExpr &expr)
      {
         //throw AQLException("TODO: CustomExpr");
         (*context.query) << context.sqlLayout.translateCustomAQLExpr(expr);
      }
      void visitBetweenChildren(AQLCustomExpr &, int pos)
      {

      }
      void visitAfterChildren(AQLCustomExpr &)
      {

      }
   };

}

namespace TypeRQ {

   AQLToSQLTranslator::AQLToSQLTranslator(SQLFunctionMap &_functionMap) :
      functionMap(_functionMap)
   {

   }

   AQLToSQLTranslator::~AQLToSQLTranslator()
   {

   }

   const char *AQLToSQLTranslator::getFunctionNameForTypecast(AQLTypeSet::ExprType exprType)
   {
      switch (exprType)
      {
         case AQLTypeSet::IRI:
            return "builtin:to-iri";
            break;

         case AQLTypeSet::STRING:
            return "builtin:to-string";
            break;

         case AQLTypeSet::INTEGER:
            return "builtin:to-integer";
            break;

         case AQLTypeSet::DOUBLE:
            return "builtin:to-double";
            break;

         case AQLTypeSet::DATETIME:
            return "builtin:to-datetime";
            break;

         case AQLTypeSet::BOOLEAN:
            return "builtin:to-boolean";
            break;

         default:
            throw AQLException("AQLToSQLTranslator::getFunctionNameForTypecast: Unknown type %d", (int)exprType);
      }
   }

   std::string AQLToSQLTranslator::aqlLiteralExprToSQL(const AQLLiteralExpr &_expr, SQLLayout &sqlLayout)
   {
      TranslatorContext t(sqlLayout, functionMap);
      AQLToSQLVisitor v(t);
      AQLLiteralExpr expr(_expr);
      expr.accept(v);
      return t.querySS->str();
   }

   std::string AQLToSQLTranslator::translateExprToSql(AQLExpr &expr, SQLLayout &layout)
   {
      TranslatorContext context(layout, functionMap);

      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "Expr before SQL transformation:\n");
         AQLPrinterVisitor aqlPrinterVisitor(std::cout);
         expr.accept(aqlPrinterVisitor);
      }

      AQLToSQLVisitor visitor(context);
      expr.accept(visitor);
      print(OL_DEBUG, "SQL form: %s\n", context.querySS->str().c_str());

      return context.querySS->str();
   }

   std::string AQLToSQLTranslator::translateToSql(AQLQuery &aql, SQLLayout &layout)
   {
      print(OL_DEBUG, "Checking triple name sanity...\n");
      checkTripleNameSanity(aql);

      print(OL_DEBUG, "Merging inner joins...\n");
      mergeInnerJoins(aql);
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "Done\n");
         print(OL_DEBUG, "AQL currently:\n");
         AQLPrinterVisitor aqlPrinterVisitor(std::cout);
         aql.accept(aqlPrinterVisitor);
      }

      print(OL_DEBUG, "Checking property access sanity...\n");
      checkPropertyAccesses(aql);

      print(OL_DEBUG, "Normalizing logical expressions...\n");
      normalizeLogicalExpressions(aql);
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL after logical expression normalization:\n");
         AQLPrinterVisitor aqlPrinterVisitor(std::cout);
         aql.accept(aqlPrinterVisitor);
      }

      print(OL_DEBUG, "Rewriting operators as functions...\n");
      normalizeOperatorsToFunctions(aql, functionMap);
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL currently:\n");
         AQLPrinterVisitor aqlPrinterVisitor(std::cout);
         aql.accept(aqlPrinterVisitor);
      }

      print(OL_DEBUG, "Performing type inference...\n");
      performTypeInference(aql, functionMap);
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL currently:\n");
         AQLPrinterVisitor aqlPrinterVisitor(std::cout);
         aql.accept(aqlPrinterVisitor);
      }

      print(OL_DEBUG, "Replacing empty type sets with nulls...\n");
      replaceEmptyTypeSetsWithNulls(aql);
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL currently:\n");
         AQLPrinterVisitor aqlPrinterVisitor(std::cout);
         aql.accept(aqlPrinterVisitor);
      }

      print(OL_DEBUG, "Possibly flattening nested joins...\n");
      const int mpad=layout.getMaxPropertyAccessDepth();
      if (mpad < std::numeric_limits<int>::max())
      {
         flattenNestedOptionalJoins(aql, mpad);
         if (outputLevel>=OL_DEBUG)
         {
            print(OL_DEBUG, "AQL after flattening (level %d):\n", mpad);
            AQLPrinterVisitor aqlPrinterVisitor(std::cout);
            aql.accept(aqlPrinterVisitor);
         }
      }
      else {
         print(OL_DEBUG, "Flattening not needed (level %d)\n", mpad);         
      }

      print(OL_DEBUG, "Optimizing comparisons...\n");
      optimizeComparisons(aql);
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL after comparison optimizations:\n");
         AQLPrinterVisitor aqlPrinterVisitor(std::cout);
         aql.accept(aqlPrinterVisitor);
      }

      print(OL_DEBUG, "Selecting function variants...\n");
      chooseVariantOfFunctions(aql, functionMap);
      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "AQL after function variant selection:\n");
         AQLPrinterVisitor aqlPrinterVisitor(std::cout);
         aql.accept(aqlPrinterVisitor);
      }

      print(OL_DEBUG, "Giving chance to SQL layout manager to prepare AQL\n");
      layout.prepareAQLForTranslationFinal(aql);

      if (outputLevel>=OL_DEBUG)
      {
         print(OL_DEBUG, "Final AQL before SQL transformation:\n");
         AQLPrinterVisitor aqlPrinterVisitor(std::cout);
         aql.accept(aqlPrinterVisitor);
      }

      print(OL_DEBUG, "Producing SQL...\n");
      TranslatorContext context(layout, functionMap);
      AQLToSQLVisitor visitor(context);
      aql.accept(visitor);
      print(OL_DEBUG, "SQL produced\n");

      return context.querySS->str();
   }

}



