/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <ostream>
#include <string>

#include "AQLModel.h"

namespace TypeRQ {

   struct ListPrettyPrinter;

   const char *getNameForAQLTripleProperty(AQLPropertyExpr::Property p);

   const char *getNameForExprType(AQLTypeSet::ExprType);

   std::ostream &operator << (std::ostream &os, AQLTypeSet typeset);
   ListPrettyPrinter &operator << (ListPrettyPrinter &os, AQLTypeSet typeset);
   std::string toString(AQLTypeSet);


   // returns simplified version of literal. Simplification is as:
   // - if double literal is integer-valued => integer literal
   // probably more to come
   AQLLiteralExpr simplifyLiteral(const AQLLiteralExpr &);

   /**
    * This is a base for visitors which only want to implement some of the
    * visit methods, but not all
    */
   class AQLOptionalVisitor : public AQLVisitor {

   public:
      virtual void visitBeforeChildren(AQLJunctionCriterion &);
      virtual void visitBetweenChildren(AQLJunctionCriterion &, int);
      virtual void visitAfterChildren(AQLJunctionCriterion &);

      virtual void visit(AQLPropertyExpr &);
      virtual void visit(AQLLiteralExpr &);
      virtual void visit(AQLNullExpr &);

      virtual void visitBeforeChildren(AQLComparisonCriterion &);
      virtual void visitBetweenChildren(AQLComparisonCriterion &);
      virtual void visitAfterChildren(AQLComparisonCriterion &);

      virtual void visitBeforeChildren(AQLNotExpression &);
      virtual void visitAfterChildren(AQLNotExpression &);

      virtual void visitBeforeChildren(AQLFunctionExpr &);
      virtual void visitBetweenChildren(AQLFunctionExpr &, int pos);
      virtual void visitAfterChildren(AQLFunctionExpr &);

      virtual void visitBeforeChildren(AQLTypecastExpression &);
      virtual void visitAfterChildren(AQLTypecastExpression &);

      virtual void visitBeforeChildren(AQLJoinGroup &);
      virtual void visitBeforeNestedJoins(AQLJoinGroup &);
      virtual void visitAfterChildren(AQLJoinGroup &);

      virtual void visitBeforeChildren(AQLSelect &);
      virtual void visitBetweenChildren(AQLSelect &, int);
      virtual void visitAfterChildren(AQLSelect &);

      virtual void visitBeforeChildren(AQLSort &);
      virtual void visitAfterChildren(AQLSort &);

      virtual void visitBeforeChildren(AQLQuery &);
      virtual void visitAfterChildren(AQLQuery &);
      virtual void visitBeforeSelects(AQLQuery &);
      virtual void visitAfterSelects(AQLQuery &);
      virtual void visitBeforeJoins(AQLQuery &);
      virtual void visitAfterJoins(AQLQuery &);
      virtual void visitBeforeCriterion(AQLQuery &);
      virtual void visitAfterCriterion(AQLQuery &);
      virtual void visitBeforeSorts(AQLQuery &);
      virtual void visitBetweenSorts(AQLQuery &, int pos);
      virtual void visitAfterSorts(AQLQuery &);

      virtual void visitBeforeChildren(AQLCustomExpr &);
      virtual void visitBetweenChildren(AQLCustomExpr &, int pos);
      virtual void visitAfterChildren(AQLCustomExpr &);
   };

   /**
    * Visitor that prints AQL into stream, mostly for debugging purposes
    */
   class AQLPrinterVisitor : public AQLVisitor {

   private:
      ListPrettyPrinter &os;

   protected:
      void printString(const std::string &) const;
      void printTypeInferenceMap(const AQLJoinGroupLike &) const;

   public:
      AQLPrinterVisitor(std::ostream &);
      ~AQLPrinterVisitor();

      virtual void visitBeforeChildren(AQLJunctionCriterion &);
      virtual void visitBetweenChildren(AQLJunctionCriterion &, int);
      virtual void visitAfterChildren(AQLJunctionCriterion &);

      virtual void visit(AQLPropertyExpr &);
      virtual void visit(AQLLiteralExpr &);
      virtual void visit(AQLNullExpr &);
      virtual void visitBeforeChildren(AQLFunctionExpr &);
      virtual void visitBetweenChildren(AQLFunctionExpr &, int pos);
      virtual void visitAfterChildren(AQLFunctionExpr &);

      virtual void visitBeforeChildren(AQLTypecastExpression &);
      virtual void visitAfterChildren(AQLTypecastExpression &);

      virtual void visitBeforeChildren(AQLComparisonCriterion &);
      virtual void visitBetweenChildren(AQLComparisonCriterion &);
      virtual void visitAfterChildren(AQLComparisonCriterion &);

      virtual void visitBeforeChildren(AQLNotExpression &);
      virtual void visitAfterChildren(AQLNotExpression &);

      virtual void visitBeforeChildren(AQLJoinGroup &);
      virtual void visitBeforeNestedJoins(AQLJoinGroup &);
      virtual void visitAfterChildren(AQLJoinGroup &);

      virtual void visitBeforeChildren(AQLSelect &);
      virtual void visitBetweenChildren(AQLSelect &, int);
      virtual void visitAfterChildren(AQLSelect &);

      virtual void visitBeforeChildren(AQLSort &);
      virtual void visitAfterChildren(AQLSort &);

      virtual void visitBeforeChildren(AQLQuery &);
      virtual void visitAfterChildren(AQLQuery &);
      virtual void visitBeforeSelects(AQLQuery &);
      virtual void visitAfterSelects(AQLQuery &);
      virtual void visitBeforeJoins(AQLQuery &);
      virtual void visitAfterJoins(AQLQuery &);
      virtual void visitBeforeCriterion(AQLQuery &);
      virtual void visitAfterCriterion(AQLQuery &);
      virtual void visitBeforeSorts(AQLQuery &);
      virtual void visitBetweenSorts(AQLQuery &, int pos);
      virtual void visitAfterSorts(AQLQuery &);

      virtual void visitBeforeChildren(AQLCustomExpr &);
      virtual void visitBetweenChildren(AQLCustomExpr &, int pos);
      virtual void visitAfterChildren(AQLCustomExpr &);

   };

   class AQLExpressionRewriter
   {
   public:
      // return new if typecast expression should be rewritten
      // otherwise, return 0
      virtual AQLExpr *rewrite(AQLJunctionCriterion &) = 0;
      virtual AQLExpr *rewrite(AQLTypecastExpression &) = 0;
      virtual AQLExpr *rewrite(AQLPropertyExpr &) = 0;
      virtual AQLExpr *rewrite(AQLLiteralExpr &) = 0;
      virtual AQLExpr *rewrite(AQLNullExpr &) = 0;
      virtual AQLExpr *rewrite(AQLFunctionExpr &) = 0;
      virtual AQLExpr *rewrite(AQLNotExpression &) = 0;
      virtual AQLExpr *rewrite(AQLComparisonCriterion &) = 0;
      virtual AQLExpr *rewrite(AQLCustomExpr &) = 0;

      static void walk(AQLQuery &, AQLExpressionRewriter &, bool childrenFirst=true);
   };

   class AQLOptionalExpressionRewriter : public AQLExpressionRewriter
   {
   public:
      virtual AQLExpr *rewrite(AQLJunctionCriterion &);
      virtual AQLExpr *rewrite(AQLTypecastExpression &);
      virtual AQLExpr *rewrite(AQLPropertyExpr &);
      virtual AQLExpr *rewrite(AQLLiteralExpr &);
      virtual AQLExpr *rewrite(AQLNullExpr &);
      virtual AQLExpr *rewrite(AQLFunctionExpr &);
      virtual AQLExpr *rewrite(AQLNotExpression &);
      virtual AQLExpr *rewrite(AQLComparisonCriterion &);
      virtual AQLExpr *rewrite(AQLCustomExpr &);
   };

   /**
    * Returns next supertype in promotion chain. E.g., int -> double.
    * If the type cannot be promoted, AQLExpr::UNSET is returned.
    */
   AQLTypeSet::ExprType getNextTypePromotion(AQLTypeSet::ExprType);

   /**
    * Returns closest common supertype or AQLExpr::UNSET, if no such thing exists.
    */
   AQLTypeSet::ExprType getClosestCommonSupertype(AQLTypeSet::ExprType, AQLTypeSet::ExprType);
}
