/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <algorithm>

#include "AQLListParser.h"
#include "AQLModel.h"
#include "AQLQueryExecutor.h"
#include "AQLSupport.h"
#include "AQLToSQLTranslator.h"
#include "Messages.h"
#include "SPARQLParser.h"
#include "SPARQLToAQL.h"
#include "SQLBackend.h"


namespace {
   using namespace TypeRQ;

   struct TextGrid
   {
      const int cols;
      std::vector<size_t> columnSizes;

      typedef std::vector<std::string> row_type;
      typedef std::list<row_type> grid_type;
      grid_type cells;

      TextGrid(int _cols) : cols(_cols), columnSizes(_cols) {}

      void newRow()
      {
         cells.push_back(std::vector<std::string>(cols));
      }

      void put(int col, const std::string &text)
      {
         cells.back().at(col)=text;
         columnSizes.at(col)=std::max(columnSizes.at(col), text.size());
      }

      size_t rowLengthWithPadding()
      {
         size_t ret=1+cols*3;
         for (int i=0; i<cols; ++i)
         {
            ret+=columnSizes.at(i);
         }
         return ret;
      }
   };

   void putRepeatedChar(std::ostream &os, char c, size_t count)
   {
      for (size_t i=0; i<count; ++i)
      {
         os << c;
      }
   }

   void putPadding(std::ostream &os, size_t fieldSize, size_t textSize)
   {
      if (textSize>=fieldSize) return;
      putRepeatedChar(os, ' ', fieldSize-textSize);
   }

   void printAQLResult(std::ostream &os, AQLQuery &aql, SQLResultIterator i, bool gridFormat)
   {
      const int cols=std::min(static_cast<size_t>(i.getColCount()), aql.selects.size());
      int rows;
      if (!gridFormat)
      {
         int row=1;
         while (i.nextRow())
         {
            os << "Row " << row << std::endl;
            AQLQuery::select_list_type::iterator k=aql.selects.begin();
            for (int j=0; j<cols; ++k, ++j)
            {
               os << "- " << (*k)->label << ": ";
               if (!i.isNull(j))
               {
                  os << i.getString(j);
               }
               else {
                  os << "(null)";
               }
               os << std::endl;
            }
            os << std::endl;
            ++row;
         }
         rows=row;
      }
      else {
         TextGrid textGrid(cols);

         AQLQuery::select_list_type::iterator k=aql.selects.begin();
         textGrid.newRow();
         for (int j=0; j<cols; ++k, ++j)
         {
            textGrid.put(j, (*k)->label);
         }

         while (i.nextRow())
         {
            textGrid.newRow();
            for (int j=0; j<cols; ++j)
            {
               if (!i.isNull(j))
               {
                  textGrid.put(j, i.getString(j));
               }
               else {
                  textGrid.put(j, "(null)");
               }
            }
         }
         rows=textGrid.cells.size()-1;
         bool headerRow=true;
         for (TextGrid::grid_type::iterator j=textGrid.cells.begin(); j!=textGrid.cells.end(); ++j)
         {
            const TextGrid::row_type &row=*j;
            for (int l=0; l<cols; ++l)
            {
               os << "| ";
               const std::string &text=row.at(l);
               os << text;
               putPadding(os, textGrid.columnSizes.at(l), text.size());
               os << ' ';
            }
            os << '|' << std::endl;
            if (headerRow)
            {
               putRepeatedChar(os, '=', textGrid.rowLengthWithPadding());
               os << std::endl;
               headerRow=false;
            }
         }
      }

      os << std::endl;
      os << rows << " rows" << std::endl;
   }
}

namespace TypeRQ
{
   QueryExecutor::QueryExecutor(int _finalStage)
      : currentStage(0), finalStage(_finalStage), is(0), deleteIsAfterUse(false)
   {}

   QueryExecutor::~QueryExecutor()
   {
      releaseIs();
   }

   std::istream &QueryExecutor::getIs()
   {
      return *is;
   }

   void QueryExecutor::releaseIs()
   {
      if (deleteIsAfterUse)
      {
         delete is;
      }
      is=0;
      deleteIsAfterUse=false;
   }

   void QueryExecutor::newQuery(std::istream *_is, bool _deleteIsAfterUse)
   {
      releaseResources();
      releaseIs();

      is=_is;
      deleteIsAfterUse=_deleteIsAfterUse;

      currentStage=0;
   }

   int QueryExecutor::getCurrentStage() const
   {
      return currentStage;
   }

   int QueryExecutor::getFinalStage() const
   {
      return finalStage;
   }

   void QueryExecutor::advanceToStage(int toStage)
   {
      while (currentStage<toStage && currentStage<finalStage)
      {
         ++currentStage;
         print(OL_VERBOSE, "Query execution stage %d (%s)\n", currentStage, getStageName(currentStage));
         processStage(currentStage);
      }
   }

   AQLQueryExecutor::AQLQueryExecutor(SQLLayout &_sqlLayout, SQLBackend &_sqlBackend) :
      QueryExecutor(3),
      sqlLayout(_sqlLayout), sqlBackend(_sqlBackend), aqlQuery(0), sqlResultIterator(0)
   {}

   AQLQueryExecutor::~AQLQueryExecutor()
   {
      releaseResources();
   }

   void AQLQueryExecutor::releaseResources()
   {
      delete aqlQuery;
      aqlQuery=0;

      sqlQuery.erase();

      delete sqlResultIterator;
      sqlResultIterator=0;
   }

   const char *AQLQueryExecutor::getStageName(int stage) const
   {
      switch (stage)
      {
         case 0: return "initial";
         case 1: return "aql";
         case 2: return "sql";
         case 3: return "result";
         default: return "unknown";
      }
   }

   void AQLQueryExecutor::processStage(int stage)
   {
      switch (stage)
      {
         case 1:
         {
            AQLListParser parser;
            aqlQuery=parser.parseQuery(getIs());
            break;
         }

         case 2:
         {
            AQLToSQLTranslator translator(sqlBackend.getFunctionMap());
/*
            if (odc.forceMaxPropertyAccessOrder>=0)
            {
               translator.setMaxPropertyAccessDepth(odc.forceMaxPropertyAccessOrder);
            }
*/
            sqlQuery=translator.translateToSql(*aqlQuery, sqlLayout);
            break;
         }

         case 3:
         {
            sqlResultIterator=new SQLResultIterator(sqlBackend.executeQuery(sqlQuery));
            break;
         }

         default: // nothing here
            break;
      }
   }

   void AQLQueryExecutor::printCurrentStageResult(std::ostream &os)
   {
      switch (getCurrentStage())
      {
         case 1:
         {
            AQLPrinterVisitor aqlPrinterVisitor(os);
            aqlQuery->accept(aqlPrinterVisitor);
            break;
         }

         case 2:
         {
            os << "SQL query:" << std::endl << sqlQuery << std::endl;
            break;
         }

         case 3:
         {
            printAQLResult(os, *aqlQuery, *sqlResultIterator, true);
            break;
         }
      }
   }

   SPARQLQueryExecutor::SPARQLQueryExecutor(SQLLayout &_sqlLayout, SQLBackend &_sqlBackend,
                                            SPARQLParser::VariableBindingMode vbm)
      : QueryExecutor(4),
        sqlLayout(_sqlLayout), sqlBackend(_sqlBackend), variableBindingMode(vbm),
        sparqlParser(), aqlQuery(),
        sqlResultIterator()
   {
   }

   SPARQLQueryExecutor::~SPARQLQueryExecutor()
   {
      releaseResources();
   }

   void SPARQLQueryExecutor::releaseResources()
   {
      delete sparqlParser;
      sparqlParser=0;

      delete aqlQuery;
      aqlQuery=0;

      sqlQuery.erase();

      delete sqlResultIterator;
      sqlResultIterator=0;
   }

   const char *SPARQLQueryExecutor::getStageName(int stage) const
   {
      switch (stage)
      {
         case 0: return "initial";
         case 1: return "sparql";
         case 2: return "aql";
         case 3: return "sql";
         case 4: return "result";
         default: return "unknown";
      }
   }

   void SPARQLQueryExecutor::printCurrentStageResult(std::ostream &os)
   {
      switch (getCurrentStage())
      {
         case 1:
         {
            sparqlParser->printIntermediateQuery(os);
            break;
         }

         case 2:
         {
            AQLPrinterVisitor aqlPrinterVisitor(os);
            aqlQuery->accept(aqlPrinterVisitor);
            break;
         }

         case 3:
         {
            os << sqlQuery << std::endl;
            break;
         }

         case 4:
         {
            printAQLResult(os, *aqlQuery, *sqlResultIterator, true);
            break;
         }
      }
   }

   void SPARQLQueryExecutor::processStage(int stage)
   {
      switch (stage)
      {
         case 1:
         {
            sparqlParser=new SPARQLParser;
            sparqlParser->parseQuery(getIs(), sqlBackend.getFunctionMap(), variableBindingMode);
            break;
         }
         case 2:
         {
            aqlQuery=sparqlParser->translateToAQL();
            break;
         }
         case 3:
         {
            AQLToSQLTranslator translator(sqlBackend.getFunctionMap());
            sqlQuery=translator.translateToSql(*aqlQuery, sqlLayout);
            break;
         }
         case 4:
         {
            sqlResultIterator=new SQLResultIterator(sqlBackend.executeQuery(sqlQuery));
            break;
         }
      }
   }

}
