/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

namespace TypeRQ {

   struct AQLQuery;
   struct SQLFunctionMap;

   // various passes

   void checkTripleNameSanity(AQLQuery &);
   void mergeInnerJoins(AQLQuery &);
   void checkPropertyAccesses(AQLQuery &);

   /**
    * Flattens nested optional joins when necessary. Flattening is considered necessary,
    * if parent properties are used in join conditions which have distance greater than
    * maxPropertyAccessDepth. E.g., direct parent property has access depth 1, and grand-parents
    * property has access depth 2. This transformation does not consider sibling property accesses
    * at all.
    *
    * @param maxPropertyAccessDepth The maximum property access depth or 0 to force flattening regardless
    * of property accesses. For typical SQL database backend
    * the correct parameter value is 1.
    */
   void flattenNestedOptionalJoins(AQLQuery &, int maxPropertyAccessDepth);

   void replaceEmptyTypeSetsWithNulls(AQLQuery &);

   void normalizeLogicalExpressions(AQLQuery &);

   void chooseVariantOfFunctions(AQLQuery &, SQLFunctionMap &);
   void optimizeComparisons(AQLQuery &);

   void normalizeOperatorsToFunctions(AQLQuery &, SQLFunctionMap &);

   // expression optimization passes
   void simplifyIsNullCoalesces(AQLQuery &, SQLFunctionMap &);
   void simplifyJunctions(AQLQuery &, SQLFunctionMap &);
   void simplifyExpressions(AQLQuery &, SQLFunctionMap &);
   void performCSE(AQLQuery &, SQLFunctionMap &);

   void addTypecastsWhereNeeded(AQLQuery &, SQLFunctionMap &);

}
