/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <map>
#include <set>
#include <string>
#include <algorithm>
#include <iostream>

#include "AQLException.h"
#include "AQLOptimizer.h"
#include "AQLModel.h"
#include "AQLSupport.h"
#include "Messages.h"
#include "SQLBackendFunctions.h"
#include "AQLToSQLTranslator.h"

namespace
{
   using namespace TypeRQ;

   class TripleNameCheckerVisitor : public AQLOptionalVisitor
   {
   protected:
      std::set<std::string> knownTriples;

      void newJoinGroup(AQLJoinGroupLike &jg)
      {
         for (AQLJoinGroupLike::name_list::const_iterator i=jg.names.begin();
              i!=jg.names.end(); ++i)
         {
            const std::string &name=*i;
            if (!knownTriples.insert(name).second)
            {
               throw AQLException("More than one triple defined with name %s", name.c_str());
            }
         }
      }

      void visitBeforeChildren(AQLJoinGroup &join)
      {
         newJoinGroup(join);
      }

      void visitBeforeChildren(AQLQuery &aql)
      {
         newJoinGroup(aql);
      }
   };

   class PropertyAccessCheckerVisitor : public AQLOptionalVisitor
   {
   protected:
      std::set<std::string> knownTriples;
      std::set<std::string> usedTriples; // triples used in selects, sorts, or criteria

      void newJoinGroup(AQLJoinGroupLike &jg)
      {
         for (AQLJoinGroupLike::name_list::const_iterator i=jg.names.begin();
              i!=jg.names.end(); ++i)
         {
            const std::string &name=*i;
            knownTriples.insert(name);
         }
      }

   public:
      void visit(AQLPropertyExpr &pexpr)
      {
         usedTriples.insert(pexpr.joinName);
      }

      void visitBeforeChildren(AQLJoinGroup &join)
      {
         newJoinGroup(join);
      }

      void visitBeforeChildren(AQLQuery &aql)
      {
         newJoinGroup(aql);
      }
      void visitAfterChildren(AQLQuery &aql)
      {
         for (std::set<std::string>::const_iterator i=usedTriples.begin();
              i!=usedTriples.end(); ++i)
         {
            const std::string &joinName=*i;
            if (knownTriples.find(joinName)==knownTriples.end())
            {
               throw AQLException("Triple %s used in a property expression but not found "
                                  "in any join group", joinName.c_str());
            }
         }
      }
   };

   class TripleOfPropertyAccessCollectorVisitor : public AQLOptionalVisitor
   {
   protected:
      std::set<std::string> joinNames;
   public:
      void visit(AQLPropertyExpr &pexpr)
      {
         joinNames.insert(pexpr.joinName);
      }
      const std::set<std::string> &getJoinNames() const
      {
         return joinNames;
      }
   };

   int calculatePropertyAccessOrder(const std::string &joinName, AQLJoinGroupLike &initial,
                                    const std::map<AQLJoinGroupLike *, AQLJoinGroupLike *> &parentMap)
   {
      AQLJoinGroupLike *i=&initial;
      int level=0;
      while (true)
      {
         if (std::find(i->names.begin(), i->names.end(), joinName)!=i->names.end())
         {
            // access joinName target found
            return level;
         }

         // not found here, try parent
         ++level;
         std::map<AQLJoinGroupLike *, AQLJoinGroupLike *>::const_iterator parentI=parentMap.find(i);
         if (parentI==parentMap.end()) return 0;
         i=parentI->second;
      }
   }

   int calculateMaxPropertyAccessOrder(AQLJoinGroupLike &jgl,
                                       const std::map<AQLJoinGroupLike *, AQLJoinGroupLike *> &parentMap)
   {
      // sanity check: must not be root join
      if (parentMap.find(&jgl)==parentMap.end())
      {
         throw AQLException("calculateMaxPropertyAccessOrder(): Internal error: jgl is parent!");
      }

      TripleOfPropertyAccessCollectorVisitor topacv;
      jgl.accept(topacv);

      const std::set<std::string> &joinNames=topacv.getJoinNames();
      int level=0;
      for (std::set<std::string>::const_iterator i=joinNames.begin(); i!=joinNames.end(); ++i)
      {
         const std::string joinName=*i;
         const int propertyLevel=calculatePropertyAccessOrder(joinName, jgl, parentMap);
         print(OL_DEBUG, "- difference %s --> %s is %d\n",
               jgl.names.front().c_str(), joinName.c_str(), propertyLevel);
         level=std::max(level, propertyLevel);
      }
      return level;
   }

   void flattenNestedOptionalJoinsRecursive(AQLJoinGroupLike &jgl, int mpad,
                                            std::map<AQLJoinGroupLike *, AQLJoinGroupLike *> &parentMap)
   {
      // first, call flatten to all children
      for (AQLJoinGroupLike::nested_join_list::iterator i=jgl.nestedJoins.begin();
           i!=jgl.nestedJoins.end(); ++i)
      {
         AQLJoinGroup &childJg=*(*i);
         parentMap[&childJg]=&jgl;
         flattenNestedOptionalJoinsRecursive(childJg, mpad, parentMap);
         parentMap.erase(&childJg);
      }

      // ok, children-level flattened. Flatten this level
      for (AQLJoinGroupLike::nested_join_list::iterator i=jgl.nestedJoins.begin();
           i!=jgl.nestedJoins.end(); )
      {
         AQLJoinGroup &childJg=*(*i);
         ++i;

         parentMap[&childJg]=&jgl;
         for (AQLJoinGroupLike::nested_join_list::iterator j=childJg.nestedJoins.begin();
              j!=childJg.nestedJoins.end(); )
         {
            AQLJoinGroup &grandchildJg=*(*j);
            AQLJoinGroupLike::nested_join_list::iterator erase_cand_j=j;
            ++j;

            parentMap[&grandchildJg]=&childJg;
            if (mpad==0 || mpad < calculateMaxPropertyAccessOrder(grandchildJg, parentMap))
            {
               // transformation needed
               print(OL_DEBUG, "- flattening %s by a level\n", grandchildJg.names.front().c_str());
               childJg.nestedJoins.erase(erase_cand_j);
               jgl.nestedJoins.insert(i, &grandchildJg);
               AQLFunctionExpr *fexpr=new AQLFunctionExpr;
               fexpr->functionName="builtin:is-not-null";
               fexpr->arguments.push_back(new AQLPropertyExpr(childJg.names.front(),
                                                              AQLPropertyExpr::SUBJECT,
                                                              AQLTypeSet::REFERENCE));
               grandchildJg.addCriterion(fexpr);
            }
            parentMap.erase(&grandchildJg);
         }
         parentMap.erase(&childJg);
      }
   }

   class EmptyTypeSetToNullReplacor : public AQLOptionalExpressionRewriter
   {
   protected:
      AQLExpr *possiblyRewriteWithNull(AQLExpr &expr)
      {
         if (expr.getExprTypeSet()==AQLTypeSet::UNSET)
         {
            delete &expr;
            return new AQLNullExpr;
         }
         return 0;
      }

   public:
      AQLExpr *rewrite(AQLPropertyExpr &pexpr)
      {
         return possiblyRewriteWithNull(pexpr);
      }
   };

   class AQLLogicalExpressionNormalizingRewriter : public AQLOptionalExpressionRewriter
   {
   public:
      AQLExpr *rewrite(AQLNotExpression &nexpr)
      {
         AQLNotExpression *notParam=dynamic_cast<AQLNotExpression *>(nexpr.expr);
         if (notParam)
         {
            AQLExpr *ret=notParam->expr;
            notParam->expr=0;
            delete &nexpr;
            return ret;
         }

         AQLComparisonCriterion *cc=dynamic_cast<AQLComparisonCriterion *>(nexpr.expr);
         if (cc)
         {
            // negate comparison
            switch (cc->comparisonType)
            {
               case AQLComparisonCriterion::LESS:
                  cc->comparisonType=AQLComparisonCriterion::GREATER_OR_EQUAL;
                  break;
               case AQLComparisonCriterion::LESS_OR_EQUAL:
                  cc->comparisonType=AQLComparisonCriterion::GREATER;
                  break;
               case AQLComparisonCriterion::EQUAL:
                  cc->comparisonType=AQLComparisonCriterion::NOT_EQUAL;
                  break;
               case AQLComparisonCriterion::NOT_EQUAL:
                  cc->comparisonType=AQLComparisonCriterion::EQUAL;
                  break;
               case AQLComparisonCriterion::GREATER_OR_EQUAL:
                  cc->comparisonType=AQLComparisonCriterion::LESS;
                  break;
               case AQLComparisonCriterion::GREATER:
                  cc->comparisonType=AQLComparisonCriterion::LESS_OR_EQUAL;
                  break;
            }

            nexpr.expr=0;
            delete &nexpr;
            return cc;
         }

         AQLJunctionCriterion *jc=dynamic_cast<AQLJunctionCriterion *>(nexpr.expr);
         if (jc)
         {
            // negate junction
            if (jc->junctionType==AQLJunctionCriterion::CONJUNCTION)
               jc->junctionType=AQLJunctionCriterion::DISJUNCTION;
            else
               jc->junctionType=AQLJunctionCriterion::CONJUNCTION;

            for (AQLJunctionCriterion::term_list::iterator i=jc->terms.begin(); i!=jc->terms.end(); ++i)
            {
               AQLExpr *param=*i;
               AQLNotExpression *notExpr=new AQLNotExpression;
               notExpr->expr=param;
               *i=notExpr;
            }

            nexpr.expr=0;
            delete &nexpr;
            return jc;
         }

         AQLLiteralExpr *le=dynamic_cast<AQLLiteralExpr *>(nexpr.expr);
         if (le && le->literalType==AQLTypeSet::BOOLEAN)
         {
            AQLLiteralExpr *ret=new AQLLiteralExpr(!le->booleanLiteral);
            delete &nexpr;
            return ret;
         }

         return 0;
      }
   };

   class ChooseFunctionVariantVisitor : public AQLOptionalVisitor
   {
   protected:
      SQLFunctionMap &functionMap;
   public:
      ChooseFunctionVariantVisitor(SQLFunctionMap &_functionMap) : functionMap(_functionMap) {}

      static void chooseVariant(AQLFunctionExpr &fexpr, SQLFunctionMap &functionMap)
      {
         if (!fexpr.chosenVariant)
         {
            fexpr.chosenVariant=new SQLFunctionType(AQLTypeSet::UNSET);
            functionMap.getFunctionMapping(fexpr.functionName).determineVariant(fexpr, *fexpr.chosenVariant);
            fexpr.functionType=fexpr.chosenVariant->returnType;
         }
      }

      void visitAfterChildren(AQLFunctionExpr &fexpr)
      {
         chooseVariant(fexpr, functionMap);
      }
   };

   class AQLInnerJoinMergerVisitor : public AQLOptionalVisitor
   {
   protected:
      void mergeInnerJoinGroupLikes(AQLJoinGroupLike &jg)
      {
         for (AQLJoinGroupLike::nested_join_list::iterator i=jg.nestedJoins.begin();
              i!=jg.nestedJoins.end(); )
         {
            AQLJoinGroup *const nestedJoin=*i;

            if (nestedJoin->joinType==AQLJoinGroup::INNER)
            {
               // Inner joins should be merged with parent join, i.e. this. So, why not merge?

               // merge join names
               jg.names.insert(jg.names.end(), nestedJoin->names.begin(), nestedJoin->names.end());

               // move criterion to parent
               if (nestedJoin->criterion)
               {
                  jg.addCriterion(nestedJoin->criterion);
                  nestedJoin->criterion=0;
               }

               // remove the nested join
               AQLJoinGroupLike::nested_join_list::iterator to_remove=i;
               ++i;
               jg.nestedJoins.erase(to_remove);
               delete nestedJoin;
            }
            else {
               ++i;
            }
         }
      }

   public:
      void visitAfterChildren(AQLJoinGroup &joinGroup)
      {
         mergeInnerJoinGroupLikes(joinGroup);
      }

      void visitAfterChildren(AQLQuery &aqlQuery)
      {
         mergeInnerJoinGroupLikes(aqlQuery);
      }
   };

   class AQLComparisonOptimizerVisitor : public AQLOptionalVisitor {
   protected:

      void optimizeEquality(AQLFunctionExpr &c)
      {
         // we assume that function has 2 parameters and is either comp-eq or comp-ne
         AQLFunctionExpr::arg_list_type::iterator i=c.arguments.begin();
         AQLExpr *const left=*i;
         ++i;
         AQLExpr *const right=*i;         

         if (dynamic_cast<AQLPropertyExpr *>(left) &&
             dynamic_cast<AQLPropertyExpr *>(right)) {
            // left and right sides are both properties, so streamline this to use references instead of real values
            AQLPropertyExpr *pleft=static_cast<AQLPropertyExpr *>(left);
            AQLPropertyExpr *pright=static_cast<AQLPropertyExpr *>(right);

            pleft->propertyType=AQLTypeSet::REFERENCE;
            pright->propertyType=AQLTypeSet::REFERENCE;

            return;
         }
      }

   public:

      void visitBeforeChildren(AQLFunctionExpr &fexpr)
      {
         if (fexpr.arguments.size()==2)
         {
            if (fexpr.functionName=="builtin:comp-eq" || fexpr.functionName=="builtin:comp-ne")
            {
               optimizeEquality(fexpr);
            }
         }
      }
   };

   class AQLNullCriterionToTrueVisitor : public AQLOptionalVisitor
   {
   public:
      void visitAfterChildren(AQLQuery &jgl)
      {
         if (!jgl.criterion) jgl.criterion=new AQLLiteralExpr(true);
      }
      void visitAfterChildren(AQLJoinGroup &jgl)
      {
         if (!jgl.criterion) jgl.criterion=new AQLLiteralExpr(true);      
      }
   };

   class AQLOperatorsToFunctionsRewriter : public AQLOptionalExpressionRewriter
   {
   protected:
      SQLFunctionMap &functionMap;

   public:
      AQLOperatorsToFunctionsRewriter(SQLFunctionMap &_functionMap) : functionMap(_functionMap) {}

      virtual AQLExpr *rewrite(AQLTypecastExpression &expr)
      {
         const char *fnName=AQLToSQLTranslator::getFunctionNameForTypecast(expr.toType);

         //SQLFunctionMapping &mapping=functionMap.getFunctionMapping(fnName);

         AQLFunctionExpr *ret=new AQLFunctionExpr;
         ret->functionName=fnName;
         ret->functionType=expr.toType;
         ret->arguments.push_back(expr.expr);

         //mapping.prepare(*ret, functionMap);

         expr.unlinkChildren();
         delete &expr;

         return ret;
      }

      virtual AQLExpr *rewrite(AQLNotExpression &expr)
      {
         const char *fnName="builtin:not";
         //SQLFunctionMapping &mapping=functionMap.getFunctionMapping(fnName);

         AQLFunctionExpr *ret=new AQLFunctionExpr;
         ret->functionName=fnName;
         ret->arguments.push_back(expr.expr);

         //mapping.prepare(*ret, functionMap);

         expr.unlinkChildren();
         delete &expr;

         return ret;
      }


      virtual AQLExpr *rewrite(AQLJunctionCriterion &expr)
      {
         const char *fnName;
         switch (expr.junctionType)
         {
            case AQLJunctionCriterion::DISJUNCTION:
               fnName="builtin:or";
               break;
            case AQLJunctionCriterion::CONJUNCTION:
               fnName="builtin:and";
               break;
            default:
               throw AQLException("Bad junction type: %d", int(expr.junctionType));
         }

         //SQLFunctionMapping &mapping=functionMap.getFunctionMapping(fnName);

         AQLFunctionExpr *ret=new AQLFunctionExpr;
         ret->functionName=fnName;

         for (AQLJunctionCriterion::term_list::iterator i=expr.terms.begin(); i!=expr.terms.end(); ++i)
         {
            AQLExpr *term=*i;
            ret->arguments.push_back(term);
         }

         //mapping.prepare(*ret, functionMap);

         expr.unlinkChildren();
         delete &expr;

         return ret;
      }

      virtual AQLExpr *rewrite(AQLComparisonCriterion &expr)
      {
         const char *fnName;
         switch (expr.comparisonType)
         {
            case AQLComparisonCriterion::LESS:
               fnName="builtin:comp-lt";
               break;
            case AQLComparisonCriterion::LESS_OR_EQUAL:
               fnName="builtin:comp-le";
               break;
            case AQLComparisonCriterion::EQUAL:
               fnName="builtin:comp-eq";
               break;
            case AQLComparisonCriterion::NOT_EQUAL:
               fnName="builtin:comp-ne";
               break;
            case AQLComparisonCriterion::GREATER_OR_EQUAL:
               fnName="builtin:comp-ge";
               break;
            case AQLComparisonCriterion::GREATER:
               fnName="builtin:comp-gt";
               break;
            default:
               throw AQLException("Bad junction type: %d", int(expr.comparisonType));
         }

         //SQLFunctionMapping &mapping=functionMap.getFunctionMapping(fnName);

         AQLFunctionExpr *ret=new AQLFunctionExpr;
         ret->functionName=fnName;

         ret->arguments.push_back(expr.left);
         ret->arguments.push_back(expr.right);

         //mapping.prepare(*ret, functionMap);

         expr.unlinkChildren();
         delete &expr;

         return ret;
      }
   };

   class AQLTypecastAdderVisitor : public AQLOptionalVisitor
   {
   private:
      SQLFunctionMap &functionMap;

      AQLExpr *wrapWithTypecastIfNeeded(AQLExpr &expr, AQLTypeSet targetType)
      {
         AQLTypeSet sourceType=expr.getExprTypeSet();

         if (makeIntersection(sourceType, targetType)==sourceType)
         {
            // source type is included in target type, return as-is
            return &expr;
         }

         print(OL_DEBUG, "%s\n", toString(expr.getExprTypeSet()).c_str());
         if (targetType==AQLTypeSet::REFERENCE)
         {
            // expression type is reference: return as-is
            return &expr;
         }

         if (targetType==AQLTypeSet::UNSET)
         {
            throw AQLException("Cannot cast to empty typeset");
         }

         // cast to first type of target typeset
         AQLTypeSet::ExprType toType=*targetType.begin();
         const char *castFnName=AQLToSQLTranslator::getFunctionNameForTypecast(toType);
         AQLFunctionExpr *castFn=new AQLFunctionExpr;
         castFn->functionName=castFnName;
         castFn->arguments.push_back(&expr);
         return castFn;
      }

   public:
      AQLTypecastAdderVisitor(SQLFunctionMap &_functionMap) : functionMap(_functionMap) {}

      void visitAfterChildren(AQLFunctionExpr &fexpr)
      {
         if (fexpr.chosenVariant)
         {
            size_t j=0;
            for (AQLFunctionExpr::arg_list_type::iterator i=fexpr.arguments.begin();
                 i!=fexpr.arguments.end();
                 ++i, ++j)
            {
               AQLTypeSet targetType=fexpr.chosenVariant->paramTypes.at(j);

               AQLExpr *expr=wrapWithTypecastIfNeeded(**i, targetType);
               *i=expr;
            }
         }
         else {
            throw AQLException("Variant not chosen for function %s", fexpr.functionName.c_str());
         }
      }
   };

   class IsNullCoalesceSimplifierRewriter : public AQLOptionalExpressionRewriter
   {
   public:
      AQLExpr *rewrite(AQLFunctionExpr &fexpr)
      {
         if (fexpr.functionName=="builtin:is-not-null" || fexpr.functionName=="builtin:is-null")
         {
            AQLExpr *param=fexpr.arguments.front();
            AQLFunctionExpr *fparam=dynamic_cast<AQLFunctionExpr *>(param);
            if (fparam && fparam->functionName=="builtin:coalesce")
            {
               AQLFunctionExpr *junction=new AQLFunctionExpr;
               junction->functionType=AQLTypeSet::BOOLEAN;
               junction->functionName=(fexpr.functionName=="builtin:is-not-null"?
                                       "builtin:or" :
                                       "builtin:and");

               for (AQLFunctionExpr::arg_list_type::iterator i=fparam->arguments.begin();
                    i!=fparam->arguments.end(); ++i)
               {
                  AQLFunctionExpr *isNullTerm=new AQLFunctionExpr;
                  isNullTerm->functionType=AQLTypeSet::BOOLEAN;
                  isNullTerm->functionName=fexpr.functionName;
                  isNullTerm->arguments.push_back(*i);
                  junction->arguments.push_back(isNullTerm);
               }

               fparam->arguments.clear();
               delete &fexpr;

               return junction;
            }
         }
         return 0;
      }
   };

   class JunctionSimplifierVisitor : public AQLOptionalVisitor
   {
   public:
      void visitAfterChildren(AQLFunctionExpr &fexpr)
      {
         if (fexpr.functionName=="builtin:or" || fexpr.functionName=="builtin:and")
         {
            for (AQLFunctionExpr::arg_list_type::iterator i=fexpr.arguments.begin();
                 i!=fexpr.arguments.end(); )
            {
               AQLFunctionExpr::arg_list_type::iterator i_next=i;
               ++i_next;

               AQLFunctionExpr *fparam=dynamic_cast<AQLFunctionExpr *>(*i);
               if (fparam && fparam->functionName==fexpr.functionName)
               {
                  fexpr.arguments.insert(i_next, fparam->arguments.begin(), fparam->arguments.end());
                  fparam->arguments.clear();
                  delete fparam;
                  fexpr.arguments.erase(i);
                  delete fexpr.chosenVariant;
                  fexpr.chosenVariant=0;
               }

               i=i_next;
            }
         }
      }
   };

   class ExpressionSimplifierRewriter : public AQLOptionalExpressionRewriter
   {
   public:
      AQLExpr *rewrite(AQLFunctionExpr &fexpr)
      {
         if (fexpr.functionName=="builtin:not")
         {
            AQLFunctionExpr *fparam=dynamic_cast<AQLFunctionExpr *>(fexpr.arguments.front());
            if (fparam)
            {
               std::string replacement;
               if (fparam->functionName=="builtin:is-not-null")
               {
                  replacement="builtin:is-null";
               }
               else if (fparam->functionName=="builtin:is-null")
               {
                  replacement="builtin:is-not-null";
               }

               if (!replacement.empty())
               {
                  fparam->functionName=replacement;
                  delete fparam->chosenVariant;
                  fparam->chosenVariant=0;
                  fexpr.arguments.pop_front();
                  delete &fexpr;
                  return fparam;
               }
            }

            AQLLiteralExpr *lparam=dynamic_cast<AQLLiteralExpr *>(fexpr.arguments.front());
            if (lparam)
            {
               if (lparam->literalType==AQLTypeSet::BOOLEAN)
               {
                  AQLLiteralExpr *ret=new AQLLiteralExpr(!lparam->booleanLiteral);
                  delete &fexpr;
                  return ret;
               }
            }
         }
         else if (fexpr.functionName=="builtin:is-not-null")
         {
            if (dynamic_cast<AQLNullExpr *>(fexpr.arguments.front()))
            {
               delete &fexpr;
               return new AQLLiteralExpr(false);
            }
         }
         else if (fexpr.functionName=="builtin:is-null")
         {
            if (dynamic_cast<AQLNullExpr *>(fexpr.arguments.front()))
            {
               delete &fexpr;
               return new AQLLiteralExpr(true);
            }
         }
         else if (fexpr.functionName=="builtin:and" || fexpr.functionName=="builtin:or")
         {
            // if and:
            // - every TRUE term is eliminated, any FALSE term reduces expr to false
            // if or:
            // - every FALSE term is eliminated, any TRUE term reduces expr to true

            const bool isAnd=fexpr.functionName=="builtin:and";
            for (AQLFunctionExpr::arg_list_type::iterator i=fexpr.arguments.begin();
                 i!=fexpr.arguments.end();)
            {
               AQLLiteralExpr *larg=dynamic_cast<AQLLiteralExpr *>(*i);
               if (larg && larg->literalType==AQLTypeSet::BOOLEAN)
               {
                  if (larg->booleanLiteral==isAnd)
                  {
                     delete larg;
                     fexpr.arguments.erase(i++);
                  }
                  else {
                     delete &fexpr;
                     return new AQLLiteralExpr(!isAnd);
                  }
               }
               else {
                  ++i;
               }
            }
         }


         return 0;
      }
   };

   class CSEEliminatorVisitor : public AQLOptionalVisitor
   {
   public:
      void visitAfterChildren(AQLFunctionExpr &fexpr)
      {
         if (fexpr.functionName=="builtin:or" || fexpr.functionName=="builtin:and")
         {
            for (AQLFunctionExpr::arg_list_type::iterator i=fexpr.arguments.begin();
                 i!=fexpr.arguments.end(); ++i)
            {
               AQLExpr &arg1=**i;

               AQLFunctionExpr::arg_list_type::iterator j=i;
               ++j;

               for (; j!=fexpr.arguments.end(); )
               {
                  AQLExpr &arg2=**j;

                  if (arg1.equals(arg2))
                  {
                     // common subexpression detected!
                     if (outputLevel>=OL_DEBUG)
                     {
                        print(OL_DEBUG, "-- eliminated common sub-expression from %s\n",
                              fexpr.functionName.c_str());
                        AQLPrinterVisitor pv(std::cout);
                        print(OL_DEBUG, "--- expression 1:\n",
                              fexpr.functionName.c_str());
                        arg1.accept(pv);
                        print(OL_DEBUG, "--- expression 2:\n",
                              fexpr.functionName.c_str());
                        arg2.accept(pv);
                     }

                     fexpr.arguments.erase(j++);
                     delete fexpr.chosenVariant;
                     fexpr.chosenVariant=0;
                     delete &arg2;
                  }
                  else {
                     ++j;
                  }
               }
            }         
         }
      }
   };
}

namespace TypeRQ {

   void checkTripleNameSanity(AQLQuery &aql)
   {
      TripleNameCheckerVisitor tncv;
      aql.accept(tncv);
   }

   void checkPropertyAccesses(AQLQuery &aql)
   {
      PropertyAccessCheckerVisitor pacv;
      aql.accept(pacv);
   }

   void flattenNestedOptionalJoins(AQLQuery &aql, int maxPropertyAccessDepth)
   {
      std::map<AQLJoinGroupLike *, AQLJoinGroupLike *> parentMap;
      flattenNestedOptionalJoinsRecursive(aql, maxPropertyAccessDepth, parentMap);
   }

   void replaceEmptyTypeSetsWithNulls(AQLQuery &aql)
   {
      EmptyTypeSetToNullReplacor etstnr;
      AQLExpressionRewriter::walk(aql, etstnr);
   }

   void normalizeLogicalExpressions(AQLQuery &aql)
   {
      AQLLogicalExpressionNormalizingRewriter lenr;
      AQLExpressionRewriter::walk(aql, lenr, false);
   }

   void chooseVariantOfFunctions(AQLQuery &aql, SQLFunctionMap &functionMap)
   {
      ChooseFunctionVariantVisitor pfv(functionMap);
      aql.accept(pfv);
   }

   void optimizeComparisons(AQLQuery &aql)
   {
      AQLComparisonOptimizerVisitor v;
      aql.accept(v);
   }

   void mergeInnerJoins(AQLQuery &aql)
   {
      AQLInnerJoinMergerVisitor ijmv;
      aql.accept(ijmv);
   }

   void normalizeOperatorsToFunctions(AQLQuery &aql, SQLFunctionMap &functionMap)
   {
      AQLNullCriterionToTrueVisitor ncttv;
      aql.accept(ncttv);
      AQLOperatorsToFunctionsRewriter otfr(functionMap);
      AQLExpressionRewriter::walk(aql, otfr);
   }

   void addTypecastsWhereNeeded(AQLQuery &aql, SQLFunctionMap &functionMap)
   {
      AQLTypecastAdderVisitor tav(functionMap);
      aql.accept(tav);
   }

   void simplifyIsNullCoalesces(AQLQuery &aql, SQLFunctionMap &functionMap)
   {
      IsNullCoalesceSimplifierRewriter incsr;
      AQLExpressionRewriter::walk(aql, incsr);
   }

   void simplifyJunctions(AQLQuery &aql, SQLFunctionMap &functionMap)
   {
      JunctionSimplifierVisitor jsv;
      aql.accept(jsv);
   }

   void simplifyExpressions(AQLQuery &aql, SQLFunctionMap &)
   {
      ExpressionSimplifierRewriter esw;
      AQLExpressionRewriter::walk(aql, esw);
   }

   void performCSE(AQLQuery &aql, SQLFunctionMap &functionMap)
   {
      CSEEliminatorVisitor cev;
      aql.accept(cev);
   }

}
