/*
 *  Type-ARQuE - the experimental SPARQL to SQL translator.
 *  Copyright (C) 2010  Sami Kiminki / Aalto University
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <list>
#include <string>
#include <vector>
#include <map>

#include <stdint.h>

#include "AQLDebug.h"

namespace TypeRQ {

   struct AQLVisitor;
   struct SQLFunctionType;

   struct AQLVisitable : protected AQLDebugBase
   {
      virtual void accept(AQLVisitor &) = 0;
   };

   struct AQLTypeSet
   {
   public:
      typedef uint8_t bitfield_type;

   private:
      bitfield_type bitField;

   public:
      enum ExprType {
         UNSET=-1,
         STRING=0, // 0
         IRI,      // 1
         DOUBLE,   // 2
         INTEGER,  // 3
         BOOLEAN,  // 4
         DATETIME, // 5

         REFERENCE,// 6 - used to equi-compare nodes, like ref(t1.o) == ref(t2.s)
         ANY,      // 7 - used when we have not yet fixed the type, and we can use whatever we like
                   //     this is mostly used in variables, where the expression itself determines the actual
                   //     variable type we'd like -- e.g. ?a == 5, use integer for ?a

         // keep this last!
         EXPRTYPE_SIZE,
      };

      AQLTypeSet();
      AQLTypeSet(const AQLTypeSet &);
      AQLTypeSet(bitfield_type);
      AQLTypeSet(ExprType);
      AQLTypeSet &operator =(AQLTypeSet);

      bitfield_type getBitField() const;
      void setBitField(bitfield_type);

      bool isType(ExprType exprType) const;
      void clear();
      void setAny();
      void removeType(ExprType exprType);
      void removeTypes(AQLTypeSet typeSet);
      void setType(ExprType exprType);
      void setTypes(AQLTypeSet typeset);

      void addSuperTypes();
      void addSubTypes();

      bool operator ==(AQLTypeSet) const;
      bool operator !=(AQLTypeSet) const;

      // determines a singular type, or UNSET if no type bits set, or ANY if ambiguous
      ExprType singularType() const;

      size_t size() const;

      static bitfield_type getBitMaskForExprType(ExprType exprType);

      class iterator
      {
      private:
         iterator &operator=(const iterator &);

      protected:
         const bitfield_type bits;
         int shift; // -1 is end

         void skipZeroBits(); // advance over zero bits
      public:
         iterator(bitfield_type, bool endIterator);
         iterator(const iterator &);
         iterator &operator ++();
         iterator &operator ++(int);
         bool operator == (const iterator &) const;
         bool operator != (const iterator &) const;

         ExprType operator *() const;
      };

      iterator begin() const;
      iterator end() const;
   };

   AQLTypeSet makeUnion(AQLTypeSet a, AQLTypeSet b);
   AQLTypeSet makeIntersection(AQLTypeSet a, AQLTypeSet b);



   struct AQLExpr : public AQLVisitable
   {

      virtual const char *getTypeName() const = 0;

      virtual AQLTypeSet getExprTypeSet() const = 0;

      // unwire children so that destructor will not also delete children
      virtual void unlinkChildren() = 0;

      virtual bool equals(const AQLExpr &) const = 0;
   };

   // custom expression. These are used if backends need some special expr types, such as
   // sql column references.
   struct AQLCustomExpr : public AQLExpr
   {
      virtual std::string toString() const = 0;
      virtual bool equals(const AQLExpr &) const = 0;
   };

   struct AQLLogicalExpr : public AQLExpr
   {
      virtual AQLTypeSet getExprTypeSet() const;
   };

   struct AQLJunctionCriterion : public AQLLogicalExpr
   {
      enum JunctionType
      {
         DISJUNCTION, // or
         CONJUNCTION, // and
      };

      JunctionType junctionType;

      typedef std::list<AQLExpr *> term_list;
      term_list terms;

      virtual const char *getTypeName() const;

      virtual ~AQLJunctionCriterion();
      virtual void accept(AQLVisitor &);

      virtual void unlinkChildren();
      virtual bool equals(const AQLExpr &) const;
   };

   struct AQLPropertyExpr : public AQLExpr
   {
      enum Property
      {
         SUBJECT=0,
         PREDICATE,
         OBJECT,

         // keep this last!
         PROPERTY_SIZE,
      };

      std::string joinName;
      Property property;
      AQLTypeSet propertyType;

      AQLPropertyExpr();
      AQLPropertyExpr(const AQLPropertyExpr &);
      AQLPropertyExpr(const std::string &joinName, Property property, AQLTypeSet propertyType);

      virtual const char *getTypeName() const;
      virtual AQLTypeSet getExprTypeSet() const;

      virtual void accept(AQLVisitor &);
      virtual void unlinkChildren();
      virtual bool equals(const AQLExpr &) const;
   };

   struct AQLLiteralExpr : public AQLExpr
   {
      const std::string stringLiteral;
      const bool booleanLiteral;
      const int64_t intLiteral;
      const double doubleLiteral;

      const AQLTypeSet::ExprType literalType;

      // use this for STRING, IRI, and DATETIME
      // DATETIME format is as in XML: YYYY-MM-DD'T'HH:MI:SS.ssss'Z'
      AQLLiteralExpr(const std::string &stringLiteral, AQLTypeSet::ExprType exprType);

      AQLLiteralExpr(int64_t intLiteral);
      AQLLiteralExpr(double doubleLiteral);
      AQLLiteralExpr(bool booleanLiteral);

      AQLLiteralExpr(const AQLLiteralExpr &);


      virtual const char *getTypeName() const;
      virtual AQLTypeSet getExprTypeSet() const;

      virtual void accept(AQLVisitor &);
      virtual void unlinkChildren();
      virtual bool equals(const AQLExpr &) const;
   };

   struct AQLNullExpr : public AQLExpr
   {
      AQLNullExpr();
      virtual const char *getTypeName() const;
      virtual AQLTypeSet getExprTypeSet() const;
      virtual void accept(AQLVisitor &);
      virtual void unlinkChildren();
      virtual bool equals(const AQLExpr &) const;
   };

   struct AQLFunctionExpr : public AQLExpr
   {
      std::string functionName;

      typedef std::list<AQLExpr *> arg_list_type;
      arg_list_type arguments;

      AQLTypeSet functionType;

      // The variant chosen by ChooseFunctionVariantVisitor, when there is more than 1 alternative.
      SQLFunctionType *chosenVariant;

      AQLFunctionExpr();
      virtual ~AQLFunctionExpr();
      virtual const char *getTypeName() const;
      virtual AQLTypeSet getExprTypeSet() const;
      virtual void accept(AQLVisitor &);
      virtual void unlinkChildren();
      virtual bool equals(const AQLExpr &) const;
   };

   struct AQLNotExpression : public AQLLogicalExpr
   {
      AQLExpr *expr;

      AQLNotExpression();
      virtual ~AQLNotExpression();
      virtual const char *getTypeName() const;
      virtual void accept(AQLVisitor &);
      virtual void unlinkChildren();
      virtual bool equals(const AQLExpr &) const;
   };

   struct AQLTypecastExpression : public AQLExpr
   {
      AQLExpr *expr;
      AQLTypeSet::ExprType toType;

      AQLTypecastExpression();
      AQLTypecastExpression(AQLExpr *expr, AQLTypeSet::ExprType toType);
      virtual ~AQLTypecastExpression();
      virtual const char *getTypeName() const;
      virtual void accept(AQLVisitor &);
      virtual AQLTypeSet getExprTypeSet() const;
      virtual void unlinkChildren();
      virtual bool equals(const AQLExpr &) const;
   };

   struct AQLComparisonCriterion : public AQLLogicalExpr
   {
      enum ComparisonType
      {
         LESS, LESS_OR_EQUAL, EQUAL, NOT_EQUAL, GREATER_OR_EQUAL, GREATER,
      };

      AQLExpr *left;
      AQLExpr *right;

      ComparisonType comparisonType;

      virtual const char *getTypeName() const;
      AQLComparisonCriterion();
      AQLComparisonCriterion(AQLExpr *left, AQLExpr *right, ComparisonType);
      virtual ~AQLComparisonCriterion();
      virtual void accept(AQLVisitor &);
      virtual void unlinkChildren();
      virtual bool equals(const AQLExpr &) const;
   };

   struct AQLJoinGroup;

   struct TripleProperty
   {
      std::string joinName;
      AQLPropertyExpr::Property property;
      TripleProperty() : joinName(), property() {}
      TripleProperty(const std::string &_j, AQLPropertyExpr::Property _p) : joinName(_j), property(_p) {}

      struct less
      {
         bool operator () (const TripleProperty &tp1, const TripleProperty &tp2) const
         {
            if (tp1.joinName<tp2.joinName) return true;
            if (tp1.joinName>tp2.joinName) return false;
            if (tp1.property<tp2.property) return true;
            return false;
         }
      };
   };

   struct AQLJoinGroupLike : public AQLVisitable
   {
      // join names, can be used in property expressions
      // must be non-empty
      //
      // These joins are inserted as a group: either all are joined or none.
      // The criterion is used to decide the whether the group is joined or not.
      typedef std::list<std::string> name_list;
      name_list names;

      // join criterion
      AQLExpr *criterion;

      typedef std::list<AQLJoinGroup *> nested_join_list;
      nested_join_list nestedJoins;

      // the following map is filled by type inference pass
      typedef std::map<TripleProperty, AQLTypeSet, TripleProperty::less> property_type_map;
      property_type_map inferenceMap;

      AQLJoinGroupLike();
      virtual ~AQLJoinGroupLike();
      void addCriterion(AQLExpr *);
   };

   struct AQLJoinGroup : public AQLJoinGroupLike
   {
      enum JoinType
      {
         INNER, // mandatory join
         LEFT_OUTER,
         // optional join
      };

      JoinType joinType;


      AQLJoinGroup();
      virtual ~AQLJoinGroup();
      virtual void accept(AQLVisitor &);

   };

   bool operator == (const AQLPropertyExpr &left, const AQLPropertyExpr &right);
   bool operator != (const AQLPropertyExpr &left, const AQLPropertyExpr &right);

   struct AQLSelect : public AQLVisitable
   {
      // name of the select, used in result sets
      std::string label;

      // expression to be selected
      AQLExpr *expr;

      AQLSelect();
      virtual ~AQLSelect();
      virtual void accept(AQLVisitor &);
   };

   struct AQLSort : public AQLVisitable
   {
      bool ascending;

      // the expression used for sorting
      AQLExpr *expr;

      AQLSort();
      virtual ~AQLSort();
      virtual void accept(AQLVisitor &);
   };

   struct AQLQuery : public AQLJoinGroupLike
   {
      // joins
      //typedef std::list<AQLJoinGroup *> join_list_type;
      //std::list<AQLJoinGroup *> joins;

      // selects
      typedef std::list<AQLSelect *> select_list_type;
      select_list_type selects;

      // criterion for search results, analogous to SQL WHERE
      //AQLExpr *criterion;

      // the sort expressions
      typedef std::list<AQLSort *> sort_list_type;
      sort_list_type sorts;

      // result limiting
      bool distinct;
      int maxRows;
      int rowOffset;

      virtual void accept(AQLVisitor &);
      AQLQuery();
      virtual ~AQLQuery();
   };

   struct AQLResultItem : protected AQLDebugBase
   {
      std::string value;
   };

   struct AQLResult : protected AQLDebugBase
   {
      virtual ~AQLResult();

      virtual const std::vector<std::string> &getHeader() = 0;

      virtual bool hasNextRow() = 0;
      virtual const std::vector<AQLResultItem *> &nextRow() = 0;
   };

   class AQLVisitor : protected AQLDebugBase
   {
   public:

      // junction
      virtual void visitBeforeChildren(AQLJunctionCriterion &) = 0;
      virtual void visitBetweenChildren(AQLJunctionCriterion &, int pos) = 0;
      virtual void visitAfterChildren(AQLJunctionCriterion &) = 0;

      // expressions
      virtual void visit(AQLPropertyExpr &) = 0;
      virtual void visit(AQLLiteralExpr &) = 0;
      virtual void visit(AQLNullExpr &) = 0;

      virtual void visitBeforeChildren(AQLFunctionExpr &) = 0;
      virtual void visitBetweenChildren(AQLFunctionExpr &, int pos) = 0;
      virtual void visitAfterChildren(AQLFunctionExpr &) = 0;

      virtual void visitBeforeChildren(AQLTypecastExpression &) = 0;
      virtual void visitAfterChildren(AQLTypecastExpression &) = 0;

      // boolean expressions
      virtual void visitBeforeChildren(AQLComparisonCriterion &) = 0;
      virtual void visitBetweenChildren(AQLComparisonCriterion &) = 0;
      virtual void visitAfterChildren(AQLComparisonCriterion &) = 0;

      virtual void visitBeforeChildren(AQLNotExpression &) = 0;
      virtual void visitAfterChildren(AQLNotExpression &) = 0;

      // joins
      virtual void visitBeforeChildren(AQLJoinGroup &) = 0;
      virtual void visitBeforeNestedJoins(AQLJoinGroup &) = 0;
      virtual void visitAfterChildren(AQLJoinGroup &) = 0;

      // selects
      virtual void visitBeforeChildren(AQLSelect &) = 0;
      virtual void visitBetweenChildren(AQLSelect &, int pos) = 0;
      virtual void visitAfterChildren(AQLSelect &) = 0;

      // sorts
      virtual void visitBeforeChildren(AQLSort &) = 0;
      virtual void visitAfterChildren(AQLSort &) = 0;

      // query
      virtual void visitBeforeChildren(AQLQuery &) = 0;
      virtual void visitAfterChildren(AQLQuery &) = 0;
      virtual void visitBeforeSelects(AQLQuery &) = 0;
      virtual void visitAfterSelects(AQLQuery &) = 0;
      virtual void visitBeforeJoins(AQLQuery &) = 0;
      virtual void visitAfterJoins(AQLQuery &) = 0;
      virtual void visitBeforeCriterion(AQLQuery &) = 0;
      virtual void visitAfterCriterion(AQLQuery &) = 0;
      virtual void visitBeforeSorts(AQLQuery &) = 0;
      virtual void visitBetweenSorts(AQLQuery &, int pos) = 0;
      virtual void visitAfterSorts(AQLQuery &) = 0;

      // custom extensions
      virtual void visitBeforeChildren(AQLCustomExpr &) = 0;
      virtual void visitBetweenChildren(AQLCustomExpr &, int pos) = 0;
      virtual void visitAfterChildren(AQLCustomExpr &) = 0;
   };


}
