#ifndef VALUATIONFORMULA_H
#define VALUATIONFORMULA_H

#include "element.h"
#include "valuation.h"
#include <set>
#include <list>
#include "elementschema.h"

class ValuationFormula
{
protected:
	std::set<TypeVariable> types;
	std::set<RelationVariable> relvars;
	std::set<AtomVariable> atomvars;
	
	ValuationFormula();
	ValuationFormula(const std::set<TypeVariable>&, const std::set<RelationVariable>&, const std::set<AtomVariable>&);
	ValuationFormula(const ValuationFormula&);
public:
	virtual ~ValuationFormula();
	virtual bool instance(const Valuation&) const = 0;
	const std::set<TypeVariable>& getTypeVariables() const;
	const std::set<RelationVariable>& getRelationVariables() const;
	const std::set<AtomVariable>& getAtomVariables() const;
	virtual std::list<ValuationFormula*> conjuncts() const = 0;
	virtual bool isExistentialFree(int = 0) const = 0;
	virtual bool isMembershipNegated(int = 0) const = 0;
	virtual bool isUnquantified() const = 0;
	virtual ValuationFormula* clone() const = 0;
	virtual ValuationFormula* createPrimedVersion(const AtomVariable&) const = 0;
	virtual void setNumericValue(VarType, int, int);
	virtual void setNumericValuesToBoundVariables(int) = 0;
	virtual std::string toString(const Context& c) const = 0;
};

class ConjunctiveValuationFormula : public ValuationFormula
{
	ValuationFormula* vf1;
	ValuationFormula* vf2;
public:
	ConjunctiveValuationFormula(const ValuationFormula&, const ValuationFormula&);
	ConjunctiveValuationFormula(ValuationFormula*, ValuationFormula*);
	ConjunctiveValuationFormula(const ConjunctiveValuationFormula&);
	ConjunctiveValuationFormula& operator=(const ConjunctiveValuationFormula&);
	~ConjunctiveValuationFormula();
	const ValuationFormula& oneValuationFormula() const;
	const ValuationFormula& theOtherValuationFormula() const;
	bool instance(const Valuation&) const;
	std::list<ValuationFormula*> conjuncts() const;
	bool isExistentialFree(int = 0) const;
	bool isMembershipNegated(int = 0) const;
	bool isUnquantified() const;
	ValuationFormula* clone() const;
	ValuationFormula* createPrimedVersion(const AtomVariable&) const;
	void setNumericValue(VarType, int, int);
	void setNumericValuesToBoundVariables(int);
	std::string toString(const Context& c) const;
};

class NegatedValuationFormula : public ValuationFormula
{
	ValuationFormula* vf;
public:
	// the second parameter is used only to differentiate the constructor from the copy constructor
	NegatedValuationFormula(const ValuationFormula&, int);
	NegatedValuationFormula(ValuationFormula*);
	NegatedValuationFormula(const NegatedValuationFormula&);
	NegatedValuationFormula& operator=(const NegatedValuationFormula&);
	~NegatedValuationFormula();
	const ValuationFormula& valuationFormula() const;
	bool instance(const Valuation&) const;
	std::list<ValuationFormula*> conjuncts() const;
	bool isExistentialFree(int = 0) const;
	bool isMembershipNegated(int = 0) const;
	bool isUnquantified() const;
	ValuationFormula* clone() const;
	ValuationFormula* createPrimedVersion(const AtomVariable&) const;
	void setNumericValue(VarType, int, int);
	void setNumericValuesToBoundVariables(int);
	std::string toString(const Context& c) const;
};

class UniversalValuationFormula : public ValuationFormula
{
	AtomVariable* av;
	ValuationFormula* vf;
public:
	UniversalValuationFormula(const AtomVariable&, const ValuationFormula&);
	UniversalValuationFormula(AtomVariable*, ValuationFormula*);
	UniversalValuationFormula(const UniversalValuationFormula&);
	UniversalValuationFormula& operator=(const UniversalValuationFormula&);
	~UniversalValuationFormula();
	const ValuationFormula& valuationFormula() const;
	const AtomVariable& getAtomVariable() const;
	bool instance(const Valuation&) const;
	std::list<ValuationFormula*> conjuncts() const;
	bool isExistentialFree(int = 0) const;
	bool isMembershipNegated(int = 0) const;
	bool isUnquantified() const;
	ValuationFormula* clone() const;
	ValuationFormula* createPrimedVersion(const AtomVariable&) const;
	void setNumericValue(VarType, int, int);
	void setNumericValuesToBoundVariables(int);
	std::string toString(const Context& c) const;
};

class ElementaryValuationFormula : public ValuationFormula
{
	RelationVariable* rel;
	AtomVariable* atom1;
	AtomVariable* atom2;
public:
	ElementaryValuationFormula(const RelationVariable&, const AtomVariable&, const AtomVariable&);
	ElementaryValuationFormula(RelationVariable*, AtomVariable*, AtomVariable*);
	ElementaryValuationFormula(const ElementaryValuationFormula&);
	ElementaryValuationFormula& operator=(const ElementaryValuationFormula&);
	~ElementaryValuationFormula();
	const RelationVariable& getRelation() const;
	const AtomVariable& getOneAtomVariable() const;
	const AtomVariable& getTheOtherAtomVariable() const;
	bool instance(const Valuation&) const;
	std::list<ValuationFormula*> conjuncts() const;
	bool isExistentialFree(int = 0) const;
	bool isMembershipNegated(int = 0) const;
	bool isUnquantified() const;
	ValuationFormula* clone() const;
	ValuationFormula* createPrimedVersion(const AtomVariable&) const;
	void setNumericValue(VarType, int, int);
	void setNumericValuesToBoundVariables(int);
	std::string toString(const Context& c) const;
};

/*
class RelationValuationFormula : public ValuationFormula
{
	ActionSet* rel;
	AtomSchema* atom1;
	AtomSchema* atom2;
public:
	RelationValuationFormula(const ActionSet&, const AtomSchema&, const AtomSchema&);
	RelationValuationFormula(ActionSet*, AtomSchema*, AtomSchema*);
	RelationValuationFormula(const RelationValuationFormula&);
	RelationValuationFormula& operator=(const RelationValuationFormula&);
	~RelationValuationFormula();
	const ActionSet& getRelation() const;
	const AtomSchema& getOneAtomSchema() const;
	const AtomSchema& getTheOtherAtomSchema() const;
	bool instance(const Valuation&) const;
	std::list<ValuationFormula*> conjuncts() const;
	bool isExistentialFree(int = 0) const;
	bool isMembershipNegated(int = 0) const;
	bool isUnquantified() const;
	ValuationFormula* clone() const;
	void setNumericValue(VarType, int, int);
	void setNumericValuesToBoundVariables(int);
	std::string toString(const Context& c) const;
};
*/

class EquivalenceValuationFormula : public ValuationFormula
{
	AtomSchema* atom1;
	AtomSchema* atom2;
public:
	EquivalenceValuationFormula(const AtomSchema&, const AtomSchema&);
	EquivalenceValuationFormula(AtomSchema*, AtomSchema*);
	EquivalenceValuationFormula(const EquivalenceValuationFormula&);
	EquivalenceValuationFormula& operator=(const EquivalenceValuationFormula&);
	~EquivalenceValuationFormula();
	const AtomSchema& getOneAtomSchema() const;
	const AtomSchema& getTheOtherAtomSchema() const;
	bool instance(const Valuation&) const;
	std::list<ValuationFormula*> conjuncts() const;
	bool isExistentialFree(int = 0) const;
	bool isMembershipNegated(int = 0) const;
	bool isUnquantified() const;
	ValuationFormula* clone() const;
	ValuationFormula* createPrimedVersion(const AtomVariable&) const;
	void setNumericValue(VarType, int, int);
	void setNumericValuesToBoundVariables(int);
	std::string toString(const Context& c) const;
};

class InclusionValuationFormula : public ValuationFormula
{
	AtomSchemaTuple* atom;
	AtomRelation* set;
public:
	InclusionValuationFormula(const AtomSchemaTuple&, const AtomRelation&);
	InclusionValuationFormula(AtomSchemaTuple*, AtomRelation*);
	InclusionValuationFormula(const InclusionValuationFormula&);
	InclusionValuationFormula& operator=(const InclusionValuationFormula&);
	~InclusionValuationFormula();
	const AtomSchemaTuple& getAtomSchema() const;
	const AtomRelation& getDomain() const;
	bool instance(const Valuation&) const;
	std::list<ValuationFormula*> conjuncts() const;
	bool isExistentialFree(int = 0) const;
	bool isMembershipNegated(int = 0) const;
	bool isUnquantified() const;
	ValuationFormula* clone() const;
	ValuationFormula* createPrimedVersion(const AtomVariable&) const;
	void setNumericValue(VarType, int, int);
	void setNumericValuesToBoundVariables(int);
	std::string toString(const Context& c) const;
};

class TopValuationFormula : public ValuationFormula
{
public:
	TopValuationFormula();
	TopValuationFormula(const TopValuationFormula&);
	TopValuationFormula& operator=(const TopValuationFormula&);
	~TopValuationFormula();
	bool instance(const Valuation&) const;
	std::list<ValuationFormula*> conjuncts() const;
	bool isExistentialFree(int = 0) const;
	bool isMembershipNegated(int = 0) const;
	bool isUnquantified() const;
	ValuationFormula* clone() const;
	ValuationFormula* createPrimedVersion(const AtomVariable&) const;
	void setNumericValue(VarType, int, int);
	void setNumericValuesToBoundVariables(int);
	std::string toString(const Context& c) const;
};

class NamedValuationFormula : public ValuationFormula
{
	const std::string* name;
	ValuationFormula* vf;
public:
	NamedValuationFormula(const std::string&, const ValuationFormula&);
	NamedValuationFormula(const std::string*, ValuationFormula*);
	NamedValuationFormula(const NamedValuationFormula&);
	NamedValuationFormula& operator=(const NamedValuationFormula&);
	~NamedValuationFormula();
	bool instance(const Valuation&) const;
	const std::string& getName() const;
	const ValuationFormula& getValuationFormula() const;
	std::list<ValuationFormula*> conjuncts() const;
	bool isExistentialFree(int = 0) const;
	bool isMembershipNegated(int = 0) const;
	bool isUnquantified() const;
	ValuationFormula* clone() const;
	ValuationFormula* createPrimedVersion(const AtomVariable&) const;
	void setNumericValue(VarType, int, int);
	void setNumericValuesToBoundVariables(int);
	std::string toString(const Context& c) const;
};

#endif
