#include "valuationformula.h"
#include <iostream>
#include <assert.h>
#include "context.h"

ValuationFormula::ValuationFormula() : types(std::set<TypeVariable>()), relvars(std::set<RelationVariable>()), atomvars(std::set<AtomVariable>())
{
}

ValuationFormula::ValuationFormula(const std::set<TypeVariable>& tvs, const std::set<RelationVariable>& rvs, const std::set<AtomVariable>& avs) : types(tvs), relvars(rvs), atomvars(avs)
{
}

ValuationFormula::ValuationFormula(const ValuationFormula& vf) : types(vf.types), relvars(vf.relvars), atomvars(vf.atomvars)
{
}

ValuationFormula::~ValuationFormula()
{
}

const std::set<TypeVariable>& ValuationFormula::getTypeVariables() const
{
	return types;
}

const std::set<RelationVariable>& ValuationFormula::getRelationVariables() const
{
	return relvars;
}

const std::set<AtomVariable>& ValuationFormula::getAtomVariables() const
{
	return atomvars;
}

void ValuationFormula::setNumericValue(VarType vt, int id, int var_id)
{
	for(std::set<TypeVariable>::iterator sit = types.begin(); sit != types.end(); sit++)
	{
		const_cast<TypeVariable&>(*sit).setNumericValue(vt,id,var_id);
	}
	for(std::set<RelationVariable>::iterator sit = relvars.begin(); sit != relvars.end(); sit++)
	{
		const_cast<RelationVariable&>(*sit).setNumericValue(vt,id,var_id);
	}
	for(std::set<AtomVariable>::iterator sit = atomvars.begin(); sit != atomvars.end(); sit++)
	{
		const_cast<AtomVariable&>(*sit).setNumericValue(vt,id,var_id);
	}
}

ConjunctiveValuationFormula::ConjunctiveValuationFormula(const ValuationFormula& f1, const ValuationFormula& f2) : ValuationFormula(f1), vf1(f1.clone()), vf2(f2.clone())
{
	types.insert(vf2->getTypeVariables().begin(), vf2->getTypeVariables().end());
	relvars.insert(vf2->getRelationVariables().begin(), vf2->getRelationVariables().end());
	atomvars.insert(vf2->getAtomVariables().begin(), vf2->getAtomVariables().end());
}

ConjunctiveValuationFormula::ConjunctiveValuationFormula(ValuationFormula* f1, ValuationFormula* f2) : ValuationFormula(), vf1(f1), vf2(f2)
{
	assert(vf1 != 0 && vf2 != 0);

	types.insert(vf1->getTypeVariables().begin(), vf1->getTypeVariables().end());
	relvars.insert(vf1->getRelationVariables().begin(), vf1->getRelationVariables().end());
	atomvars.insert(vf1->getAtomVariables().begin(), vf1->getAtomVariables().end());
	types.insert(vf2->getTypeVariables().begin(), vf2->getTypeVariables().end());
	relvars.insert(vf2->getRelationVariables().begin(), vf2->getRelationVariables().end());
	atomvars.insert(vf2->getAtomVariables().begin(), vf2->getAtomVariables().end());
}

ConjunctiveValuationFormula::ConjunctiveValuationFormula(const ConjunctiveValuationFormula& f) : ValuationFormula(f), vf1(f.vf1->clone()), vf2(f.vf2->clone())
{
}

ConjunctiveValuationFormula& ConjunctiveValuationFormula::operator=(const ConjunctiveValuationFormula& f)
{
	if(&f == this) return *this;
	ConjunctiveValuationFormula cvf(f);
	std::swap(types, cvf.types);
	std::swap(relvars, cvf.relvars);
	std::swap(atomvars, cvf.atomvars);
	std::swap(vf1, cvf.vf1);
	std::swap(vf2, cvf.vf2);
	return *this;
}

ValuationFormula* ConjunctiveValuationFormula::clone() const
{
	assert(vf1 != 0 && vf2 != 0);

	return new ConjunctiveValuationFormula(*this);
}

ValuationFormula* ConjunctiveValuationFormula::createPrimedVersion(const AtomVariable& av) const
{
	assert(vf1 != 0 && vf2 != 0);

	return new ConjunctiveValuationFormula(vf1->createPrimedVersion(av), vf2->createPrimedVersion(av));
}

ConjunctiveValuationFormula::~ConjunctiveValuationFormula()
{
	delete vf1;
	delete vf2;
}

const ValuationFormula& ConjunctiveValuationFormula::oneValuationFormula() const
{
	assert(vf1 != 0);
	
	return *vf1;
}

const ValuationFormula& ConjunctiveValuationFormula::theOtherValuationFormula() const
{
	assert(vf2 != 0);
	
	return *vf2;
}

bool ConjunctiveValuationFormula::instance(const Valuation& val) const
{
	assert(vf1 != 0 && vf2 != 0);

	return (vf1->instance(val) && vf2->instance(val));
}

std::list<ValuationFormula*> ConjunctiveValuationFormula::conjuncts() const
{
	assert(vf1 != 0 && vf2 != 0);

	std::list<ValuationFormula*> cons(vf1->conjuncts());
	std::list<ValuationFormula*> cons2(vf2->conjuncts());
	cons.splice(cons.end(), cons2);
	return cons;
}

bool ConjunctiveValuationFormula::isExistentialFree(int numOfNegations) const
{
	assert(vf1 != 0 && vf2 != 0);

	return (vf1->isExistentialFree(numOfNegations) && vf2->isExistentialFree(numOfNegations));
}

bool ConjunctiveValuationFormula::isMembershipNegated(int numOfNegations) const
{
	assert(vf1 != 0 && vf2 != 0);

	return (vf1->isMembershipNegated(numOfNegations) && vf2->isMembershipNegated(numOfNegations));
}

bool ConjunctiveValuationFormula::isUnquantified() const
{
	assert(vf1 != 0 && vf2 != 0);

	return (vf1->isUnquantified() && vf2->isUnquantified());
}

void ConjunctiveValuationFormula::setNumericValue(VarType vt, int id, int var_id)
{
	assert(vf1 != 0 && vf2 != 0);

	ValuationFormula::setNumericValue(vt,id,var_id);
	vf1->setNumericValue(vt,id,var_id);
	vf2->setNumericValue(vt,id,var_id);
}

void ConjunctiveValuationFormula::setNumericValuesToBoundVariables(int val)
{
	assert(vf1 != 0 && vf2 != 0);

	vf1->setNumericValuesToBoundVariables(val);
	vf2->setNumericValuesToBoundVariables(val);
}

std::string ConjunctiveValuationFormula::toString(const Context& c) const
{
	assert(vf1 != 0 && vf2 != 0);

	std::string str(1, '(');
	str.append(vf1->toString(c));
	str.push_back('&');
	str.append(vf2->toString(c));
	str.push_back(')');
	return str;
}


NegatedValuationFormula::NegatedValuationFormula(const ValuationFormula& f, int) : ValuationFormula(f), vf(f.clone())
{
}

NegatedValuationFormula::NegatedValuationFormula(ValuationFormula* f) : ValuationFormula(), vf(f)
{
	assert(vf != 0);

	types.insert(vf->getTypeVariables().begin(), vf->getTypeVariables().end());
	relvars.insert(vf->getRelationVariables().begin(), vf->getRelationVariables().end());
	atomvars.insert(vf->getAtomVariables().begin(), vf->getAtomVariables().end());
}

NegatedValuationFormula::NegatedValuationFormula(const NegatedValuationFormula& f) : ValuationFormula(f), vf(f.vf->clone())
{
}

NegatedValuationFormula& NegatedValuationFormula::operator=(const NegatedValuationFormula& f)
{
	if(&f == this) return *this;
	NegatedValuationFormula nvf(f);
	std::swap(types, nvf.types);
	std::swap(relvars, nvf.relvars);
	std::swap(atomvars, nvf.atomvars);
	std::swap(vf, nvf.vf);
	return *this;
}

ValuationFormula* NegatedValuationFormula::clone() const
{
	assert(vf != 0);

	return new NegatedValuationFormula(*this);
}

ValuationFormula* NegatedValuationFormula::createPrimedVersion(const AtomVariable& av) const
{
	assert(vf != 0);

	return new NegatedValuationFormula(vf->createPrimedVersion(av));
}

NegatedValuationFormula::~NegatedValuationFormula()
{
	delete vf;
}

const ValuationFormula& NegatedValuationFormula::valuationFormula() const
{
	assert(vf != 0);

	return *vf;
}

bool NegatedValuationFormula::instance(const Valuation& val) const
{
	assert(vf != 0);

	return (!vf->instance(val));
}

std::list<ValuationFormula*> NegatedValuationFormula::conjuncts() const
{
	std::list<ValuationFormula*> cons;
	cons.push_back(clone());
	return cons;
}

bool NegatedValuationFormula::isExistentialFree(int numOfNegations) const
{
	assert(vf != 0);

	return (vf->isExistentialFree(numOfNegations+1));
}

bool NegatedValuationFormula::isMembershipNegated(int numOfNegations) const
{
	assert(vf != 0);

	return (vf->isMembershipNegated(numOfNegations+1));
}

bool NegatedValuationFormula::isUnquantified() const
{
	assert(vf != 0);

	return vf->isUnquantified();
}

void NegatedValuationFormula::setNumericValue(VarType vt, int id, int var_id)
{
	assert(vf != 0);

	ValuationFormula::setNumericValue(vt,id,var_id);
	vf->setNumericValue(vt,id,var_id);
}

void NegatedValuationFormula::setNumericValuesToBoundVariables(int val)
{
	assert(vf != 0);

	vf->setNumericValuesToBoundVariables(val);
}

std::string NegatedValuationFormula::toString(const Context& c) const
{
	assert(vf != 0);

	std::string str(1, '!');
	str.push_back('(');
	str.append(vf->toString(c));
	str.push_back(')');
	return str;
}

UniversalValuationFormula::UniversalValuationFormula(const AtomVariable& a, const ValuationFormula& f) : ValuationFormula(f), av(new AtomVariable(a)), vf(f.clone())
{
	if(av->second.second == TYPEVAR)
	{
		types.insert(av->second);
	}
	std::set<AtomVariable>::iterator it = atomvars.find(*av);
	if(it != atomvars.end()) atomvars.erase(it);
}

UniversalValuationFormula::UniversalValuationFormula(AtomVariable* a, ValuationFormula* f) : ValuationFormula(), av(a), vf(f)
{
	assert(vf != 0 && av != 0);
	
	types.insert(vf->getTypeVariables().begin(), vf->getTypeVariables().end());
	relvars.insert(vf->getRelationVariables().begin(), vf->getRelationVariables().end());
	atomvars.insert(vf->getAtomVariables().begin(), vf->getAtomVariables().end());
	if(av->second.second == TYPEVAR)
	{
		types.insert(av->second);
	}
	std::set<AtomVariable>::iterator it = atomvars.find(*av);
	if(it != atomvars.end()) atomvars.erase(it);
}

UniversalValuationFormula::UniversalValuationFormula(const UniversalValuationFormula& f) : ValuationFormula(f), av(new AtomVariable(*(f.av))), vf(f.vf->clone())
{
}

ValuationFormula* UniversalValuationFormula::clone() const
{
	assert(vf != 0 && av != 0);
	
	return new UniversalValuationFormula(*this);
}

ValuationFormula* UniversalValuationFormula::createPrimedVersion(const AtomVariable& pv) const
{
	assert(vf != 0 && av != 0);
	
	if(pv == *av)
	{
		return new UniversalValuationFormula(*this);
	}
	else
	{
		return new UniversalValuationFormula(new AtomVariable(*av), vf->createPrimedVersion(pv));
	}
}

UniversalValuationFormula& UniversalValuationFormula::operator=(const UniversalValuationFormula& f)
{
	if(&f == this) return *this;
	UniversalValuationFormula uvf(f);
	std::swap(types, uvf.types);
	std::swap(relvars, uvf.relvars);
	std::swap(atomvars, uvf.atomvars);
	std::swap(av, uvf.av);
	std::swap(vf, uvf.vf);
	return *this;
}

UniversalValuationFormula::~UniversalValuationFormula()
{
	delete vf;
	delete av;
}

const ValuationFormula& UniversalValuationFormula::valuationFormula() const
{
	assert(vf != 0);
	
	return *vf;
}

const AtomVariable& UniversalValuationFormula::getAtomVariable() const
{
	assert(av != 0);
	
	return *av;
}

bool UniversalValuationFormula::instance(const Valuation& val) const
{
	assert(vf != 0 && av != 0);
	
	bool inst = true;
	std::list<Valuation*> vals = val.addAtomVariable(av->second);
	for(std::list<Valuation*>::const_iterator it = vals.begin(); it != vals.end(); ++it)
	{
		inst = inst && vf->instance(**it);
		delete *it;
	}
	return inst;
}

std::list<ValuationFormula*> UniversalValuationFormula::conjuncts() const
{
	std::list<ValuationFormula*> cons;
	cons.push_back(clone());
	return cons;
}

bool UniversalValuationFormula::isExistentialFree(int numOfNegations) const
{
	assert(vf != 0);
	
	if(numOfNegations%2 == 0)
	{
		return (vf->isExistentialFree(numOfNegations));
	}
	else
	{
		return false;
	}
}

bool UniversalValuationFormula::isMembershipNegated(int numOfNegations) const
{
	assert(vf != 0);
	
	return (vf->isMembershipNegated(numOfNegations));
}

bool UniversalValuationFormula::isUnquantified() const
{
	return false;
}

void UniversalValuationFormula::setNumericValue(VarType vt, int av_id, int var_id)
{
	assert(av != 0 && vf != 0);
	
	ValuationFormula::setNumericValue(vt,av_id,var_id);
	if(vt != ATOMVAR || av->first != av_id)
	{
		av->setNumericValue(vt,av_id,var_id);
		vf->setNumericValue(vt,av_id,var_id);
	}
}

void UniversalValuationFormula::setNumericValuesToBoundVariables(int val)
{
	assert(av != 0 && vf != 0);

	av->id = val;
	vf->setNumericValue(ATOMVAR, av->first, val);
	vf->setNumericValuesToBoundVariables(val + 1);
}

std::string UniversalValuationFormula::toString(const Context& c) const
{
	assert(av != 0 && vf != 0);
	
	std::string str(1, '(');
	str.push_back('?');
	str.append(av->toString(c));
	str.push_back(':');
	str.append(vf->toString(c));
	str.push_back(')');
	return str;
}

ElementaryValuationFormula::ElementaryValuationFormula(const RelationVariable& r, const AtomVariable& a1, const AtomVariable& a2) : ValuationFormula(), rel(new RelationVariable(r)), atom1(new AtomVariable(a1)), atom2(new AtomVariable(a2))
{
	relvars.insert(*rel);
	types.insert(rel->second);
	types.insert(rel->third);
	atomvars.insert(*atom1);
	atomvars.insert(*atom2);
	if(atom1->second.second == TYPEVAR) types.insert(atom1->second);
	if(atom2->second.second == TYPEVAR) types.insert(atom2->second);
}

ElementaryValuationFormula::ElementaryValuationFormula(RelationVariable* r, AtomVariable* a1, AtomVariable* a2) : ValuationFormula(), rel(r), atom1(a1), atom2(a2)
{
	assert(rel != 0 && atom1 != 0 && atom2 != 0);
			
	relvars.insert(*rel);
	types.insert(rel->second);
	types.insert(rel->third);
	atomvars.insert(*atom1);
	atomvars.insert(*atom2);
	if(atom1->second.second == TYPEVAR) types.insert(atom1->second);
	if(atom2->second.second == TYPEVAR) types.insert(atom2->second);
}

ElementaryValuationFormula::ElementaryValuationFormula(const ElementaryValuationFormula& evf) : ValuationFormula(evf), rel(new RelationVariable(*(evf.rel))), atom1(new AtomVariable(*(evf.atom1))), atom2(new AtomVariable(*(evf.atom2)))
{
}

ElementaryValuationFormula& ElementaryValuationFormula::operator=(const ElementaryValuationFormula& evf)
{
	if(this == &evf) return *this;
	ElementaryValuationFormula f(evf);
	std::swap(types, f.types);
	std::swap(relvars, f.relvars);
	std::swap(atomvars, f.atomvars);
	std::swap(rel, f.rel);
	std::swap(atom1, f.atom1);
	std::swap(atom2, f.atom2);
	return *this;
}

ElementaryValuationFormula::~ElementaryValuationFormula()
{
	delete atom1;
	delete atom2;
	delete rel;
}

const RelationVariable& ElementaryValuationFormula::getRelation() const
{
	assert(rel != 0);
	
	return *rel;
}

const AtomVariable& ElementaryValuationFormula::getOneAtomVariable() const
{
	assert(atom1 != 0);
	
	return *atom1;
}

const AtomVariable& ElementaryValuationFormula::getTheOtherAtomVariable() const
{
	assert(atom2 != 0);
	
	return *atom2;
}

ValuationFormula* ElementaryValuationFormula::clone() const
{
	assert(rel != 0 && atom1 != 0 && atom2 != 0);

	return new ElementaryValuationFormula(*this);
}

ValuationFormula* ElementaryValuationFormula::createPrimedVersion(const AtomVariable& pv) const
{
	assert(rel != 0 && atom1 != 0 && atom2 != 0);

	AtomVariable* av1;
	AtomVariable* av2;
	if(*atom1 == pv)
	{
		av1 = new AtomVariable(pv);
		av1->first = -(av1->first)-1;
	}
	else
	{
		av1 = new AtomVariable(*atom1);
	}
	if(*atom2 == pv)
	{
		av2 = new AtomVariable(pv);
		av2->first = -(av2->first)-1;
	}
	else
	{
		av2 = new AtomVariable(*atom2);
	}
	return new ElementaryValuationFormula(new RelationVariable(*rel), av1, av2);
}

bool ElementaryValuationFormula::instance(const Valuation& val) const
{
	bool inst;
	if(atom1->second.second == TYPEVAR && atom2->second.second == TYPEVAR)
	{
		inst = val.isMember(rel->id, atom1->id, atom2->id);
	}
	else
	{
		inst = false;
	}
	return inst;
}

std::list<ValuationFormula*> ElementaryValuationFormula::conjuncts() const
{
	std::list<ValuationFormula*> cons;
	cons.push_back(clone());
	return cons;
}

bool ElementaryValuationFormula::isExistentialFree(int) const
{
	return true;
}

bool ElementaryValuationFormula::isMembershipNegated(int numOfNegations) const
{
	return (numOfNegations%2 == 1);
}

bool ElementaryValuationFormula::isUnquantified() const
{
	return true;
}

void ElementaryValuationFormula::setNumericValue(VarType vt, int id, int var_id)
{
	assert(rel != 0 && atom1 != 0 && atom2 != 0);

	ValuationFormula::setNumericValue(vt,id,var_id);
	rel->setNumericValue(vt,id,var_id);
	atom1->setNumericValue(vt,id,var_id);
	atom2->setNumericValue(vt,id,var_id);
}

void ElementaryValuationFormula::setNumericValuesToBoundVariables(int val)
{
}

std::string ElementaryValuationFormula::toString(const Context& c) const
{
	assert(atom1 != 0 && rel != 0 && atom2 != 0);
	
	std::string str;
	str.append(atom1->toString(c));
	str.push_back(' ');
	str.append(rel->toString(c));
	str.push_back(' ');
	str.append(atom2->toString(c));
	return str;
}

/*
RelationValuationFormula::RelationValuationFormula(const ActionSet& r, const AtomSchema& a1, const AtomSchema& a2) : ValuationFormula(), rel(new ActionSet(r)), atom1(a1.clone()), atom2(a2.clone())
{
	atomvars.insert(atom1->getAtomVariables().begin(), atom1->getAtomVariables().end());
	atomvars.insert(atom2->getAtomVariables().begin(), atom2->getAtomVariables().end());
	types.insert(atom1->getTypeVariables().begin(), atom1->getTypeVariables().end());
	types.insert(atom2->getTypeVariables().begin(), atom2->getTypeVariables().end());
}

RelationValuationFormula::RelationValuationFormula(ActionSet* r, AtomSchema* a1, AtomSchema* a2) : ValuationFormula(), rel(r), atom1(a1), atom2(a2)
{
	assert(rel != 0 && atom1 != 0 && atom2 != 0);
			
	atomvars.insert(atom1->getAtomVariables().begin(), atom1->getAtomVariables().end());
	atomvars.insert(atom2->getAtomVariables().begin(), atom2->getAtomVariables().end());
	types.insert(atom1->getTypeVariables().begin(), atom1->getTypeVariables().end());
	types.insert(atom2->getTypeVariables().begin(), atom2->getTypeVariables().end());
}

RelationValuationFormula::RelationValuationFormula(const RelationValuationFormula& evf) : ValuationFormula(evf), rel(new ActionSet(*(evf.rel))), atom1(evf.atom1->clone()), atom2(evf.atom2->clone())
{
}

RelationValuationFormula& RelationValuationFormula::operator=(const RelationValuationFormula& evf)
{
	if(this == &evf) return *this;
	RelationValuationFormula f(evf);
	std::swap(types, f.types);
	std::swap(relvars, f.relvars);
	std::swap(atomvars, f.atomvars);
	std::swap(rel, f.rel);
	std::swap(atom1, f.atom1);
	std::swap(atom2, f.atom2);
	return *this;
}

RelationValuationFormula::~RelationValuationFormula()
{
	delete atom1;
	delete atom2;
	delete rel;
}

const ActionSet& RelationValuationFormula::getRelation() const
{
	assert(rel != 0);
	
	return *rel;
}

const AtomSchema& RelationValuationFormula::getOneAtomSchema() const
{
	assert(atom1 != 0);
	
	return *atom1;
}

const AtomSchema& RelationValuationFormula::getTheOtherAtomSchema() const
{
	assert(atom2 != 0);
	
	return *atom2;
}

ValuationFormula* RelationValuationFormula::clone() const
{
	assert(rel != 0 && atom1 != 0 && atom2 != 0);

	return new RelationValuationFormula(*this);
}

bool RelationValuationFormula::instance(const Valuation& val) const
{
	Atom a1 = atom1->instance(val);
	Atom a2 = atom2->instance(val);
	Action act;
	act.push_back(a1);
	return rel->operator()(act,a2);
}

std::list<ValuationFormula*> RelationValuationFormula::conjuncts() const
{
	std::list<ValuationFormula*> cons;
	cons.push_back(clone());
	return cons;
}

bool RelationValuationFormula::isExistentialFree(int) const
{
	return true;
}

bool RelationValuationFormula::isMembershipNegated(int numOfNegations) const
{
	return (numOfNegations%2 == 1);
}

bool RelationValuationFormula::isUnquantified() const
{
	return true;
}

void RelationValuationFormula::setNumericValue(VarType vt, int id, int var_id)
{
	assert(atom1 != 0 && atom2 != 0);

	ValuationFormula::setNumericValue(vt,id,var_id);
	atom1->setNumericValue(vt,id,var_id);
	atom2->setNumericValue(vt,id,var_id);
}

void RelationValuationFormula::setNumericValuesToBoundVariables(int val)
{
}

std::string RelationValuationFormula::toString(const Context& c) const
{
	assert(atom1 != 0 && rel != 0 && atom2 != 0);
	
	std::string str;
	str.append(atom1->toString(c));
	str.push_back(' ');
	str.append(c.relationToString(*rel));
	str.push_back(' ');
	str.append(atom2->toString(c));
	return str;
}
*/

EquivalenceValuationFormula::EquivalenceValuationFormula(const AtomSchema& a1, const AtomSchema& a2) : ValuationFormula(), atom1(a1.clone()), atom2(a2.clone())
{
	atomvars.insert(atom1->getAtomVariables().begin(), atom1->getAtomVariables().end());
	atomvars.insert(atom2->getAtomVariables().begin(), atom2->getAtomVariables().end());
	types.insert(atom1->getTypeVariables().begin(), atom1->getTypeVariables().end());
	types.insert(atom2->getTypeVariables().begin(), atom2->getTypeVariables().end());
}

EquivalenceValuationFormula::EquivalenceValuationFormula(AtomSchema* a1, AtomSchema* a2) : ValuationFormula(), atom1(a1), atom2(a2)
{
	assert(atom1 != 0 && atom2 != 0);
	
	atomvars.insert(atom1->getAtomVariables().begin(), atom1->getAtomVariables().end());
	atomvars.insert(atom2->getAtomVariables().begin(), atom2->getAtomVariables().end());
	types.insert(atom1->getTypeVariables().begin(), atom1->getTypeVariables().end());
	types.insert(atom2->getTypeVariables().begin(), atom2->getTypeVariables().end());
}

EquivalenceValuationFormula::EquivalenceValuationFormula(const EquivalenceValuationFormula& evf) : ValuationFormula(evf), atom1(evf.atom1->clone()), atom2(evf.atom2->clone())
{
}

EquivalenceValuationFormula& EquivalenceValuationFormula::operator=(const EquivalenceValuationFormula& evf)
{
	if(&evf == this) return *this;
	EquivalenceValuationFormula f(evf);
	std::swap(types, f.types);
	std::swap(relvars, f.relvars);
	std::swap(atomvars, f.atomvars);
	std::swap(atom1, f.atom1);
	std::swap(atom2, f.atom2);
	return *this;
}

EquivalenceValuationFormula::~EquivalenceValuationFormula()
{
	delete atom1;
	delete atom2;
}

const AtomSchema& EquivalenceValuationFormula::getOneAtomSchema() const
{
	assert(atom1 != 0);
	
	return *atom1;
}

const AtomSchema& EquivalenceValuationFormula::getTheOtherAtomSchema() const
{
	assert(atom2 != 0);

	return *atom2;
}

ValuationFormula* EquivalenceValuationFormula::clone() const
{
	assert(atom1 != 0 && atom2 != 0);

	return new EquivalenceValuationFormula(*this);
}

ValuationFormula* EquivalenceValuationFormula::createPrimedVersion(const AtomVariable& av) const
{
	assert(atom1 != 0 && atom2 != 0);

	return new EquivalenceValuationFormula(atom1->createPrimedVersion(av), atom2->createPrimedVersion(av));
}

bool EquivalenceValuationFormula::instance(const Valuation& val) const
{
	assert(atom1 != 0 && atom2 != 0);

	bool inst;
	Atom a1 = atom1->instance(val);
	Atom a2 = atom2->instance(val);
	inst = (a1 == a2);
	return inst;
}

std::list<ValuationFormula*> EquivalenceValuationFormula::conjuncts() const
{
	std::list<ValuationFormula*> cons;
	cons.push_back(clone());
	return cons;
}

bool EquivalenceValuationFormula::isExistentialFree(int) const
{
	return true;
}

bool EquivalenceValuationFormula::isMembershipNegated(int numOfNegations) const
{
	return true;
}

bool EquivalenceValuationFormula::isUnquantified() const
{
	return true;
}

void EquivalenceValuationFormula::setNumericValue(VarType vt, int id, int var_id)
{
	assert(atom1 != 0 && atom2 != 0);

	ValuationFormula::setNumericValue(vt,id,var_id);
	atom1->setNumericValue(vt,id,var_id);
	atom2->setNumericValue(vt,id,var_id);
}

void EquivalenceValuationFormula::setNumericValuesToBoundVariables(int val)
{
}

std::string EquivalenceValuationFormula::toString(const Context& c) const
{
	assert(atom1 != 0 && atom2 != 0);
	
	std::string str;
	str.append(atom1->toString(c));
	str.push_back(' ');
	str.push_back('=');
	str.push_back(' ');
	str.append(atom2->toString(c));
	return str;
}

InclusionValuationFormula::InclusionValuationFormula(const AtomSchemaTuple& a, const AtomRelation& d) : ValuationFormula(), atom(new AtomSchemaTuple(a)), set(new AtomRelation(d))
{
	assert(atom != 0 && set != 0);
	
	atomvars.insert(atom->getAtomVariables().begin(), atom->getAtomVariables().end());
	types.insert(atom->getTypeVariables().begin(), atom->getTypeVariables().end());
}

InclusionValuationFormula::InclusionValuationFormula(AtomSchemaTuple* a, AtomRelation* d) : ValuationFormula(), atom(a), set(d)
{
	assert(atom != 0 && set != 0);
	
	atomvars.insert(atom->getAtomVariables().begin(), atom->getAtomVariables().end());
	types.insert(atom->getTypeVariables().begin(), atom->getTypeVariables().end());
}

InclusionValuationFormula::InclusionValuationFormula(const InclusionValuationFormula& evf) : ValuationFormula(evf), atom(new AtomSchemaTuple(*(evf.atom))), set(new AtomRelation(*(evf.set)))
{
}

InclusionValuationFormula& InclusionValuationFormula::operator=(const InclusionValuationFormula& evf)
{
	if(&evf == this) return *this;
	InclusionValuationFormula f(evf);
	std::swap(types, f.types);
	std::swap(relvars, f.relvars);
	std::swap(atomvars, f.atomvars);
	std::swap(atom, f.atom);
	std::swap(set, f.set);
	return *this;
}

InclusionValuationFormula::~InclusionValuationFormula()
{
	delete atom;
	delete set;
}

const AtomSchemaTuple& InclusionValuationFormula::getAtomSchema() const
{
	assert(atom != 0);
	
	return *atom;
}

const AtomRelation& InclusionValuationFormula::getDomain() const
{
	assert(set != 0);

	return *set;
}

ValuationFormula* InclusionValuationFormula::clone() const
{
	assert(atom != 0 && set != 0);

	return new InclusionValuationFormula(*this);
}

ValuationFormula* InclusionValuationFormula::createPrimedVersion(const AtomVariable& av) const
{
	assert(atom != 0 && set != 0);

	return new InclusionValuationFormula(atom->createPrimedVersion(av), new AtomRelation(*set));
}

bool InclusionValuationFormula::instance(const Valuation& val) const
{
	assert(atom != 0 && set != 0);

	bool inst;
	AtomTuple a = atom->instance(val);
	return set->isElement(a);
}

std::list<ValuationFormula*> InclusionValuationFormula::conjuncts() const
{
	std::list<ValuationFormula*> cons;
	cons.push_back(clone());
	return cons;
}

bool InclusionValuationFormula::isExistentialFree(int) const
{
	return true;
}

bool InclusionValuationFormula::isMembershipNegated(int numOfNegations) const
{
	return true;
}

bool InclusionValuationFormula::isUnquantified() const
{
	return true;
}

void InclusionValuationFormula::setNumericValue(VarType vt, int id, int var_id)
{
	assert(atom != 0 && set != 0);

	ValuationFormula::setNumericValue(vt,id,var_id);
	atom->setNumericValue(vt,id,var_id);
}

void InclusionValuationFormula::setNumericValuesToBoundVariables(int val)
{
}

std::string InclusionValuationFormula::toString(const Context& c) const
{
	assert(atom != 0 && set != 0);
	
	std::string str;
	str.append(atom->toString(c));
	str.push_back(' ');
	str.append("in");
	str.push_back(' ');
	str.append(set->toString(c));
	return str;
}

TopValuationFormula::TopValuationFormula() : ValuationFormula()
{
}

TopValuationFormula::TopValuationFormula(const TopValuationFormula& tvf) : ValuationFormula(tvf)
{
}

TopValuationFormula& TopValuationFormula::operator=(const TopValuationFormula& evf)
{
	return *this;
}

TopValuationFormula::~TopValuationFormula()
{
}

ValuationFormula* TopValuationFormula::clone() const
{
	return new TopValuationFormula();
}

ValuationFormula* TopValuationFormula::createPrimedVersion(const AtomVariable&) const
{
	return new TopValuationFormula();
}

bool TopValuationFormula::instance(const Valuation& val) const
{
	return true;
}

std::list<ValuationFormula*> TopValuationFormula::conjuncts() const
{
	std::list<ValuationFormula*> cons;
	cons.push_back(clone());
	return cons;
}

bool TopValuationFormula::isExistentialFree(int) const
{
	return true;
}

bool TopValuationFormula::isMembershipNegated(int numOfNegations) const
{
	return true;
}

bool TopValuationFormula::isUnquantified() const
{
	return true;
}

void TopValuationFormula::setNumericValue(VarType,int,int)
{
}

void TopValuationFormula::setNumericValuesToBoundVariables(int val)
{
}

std::string TopValuationFormula::toString(const Context&) const
{
	return "true";
}

NamedValuationFormula::NamedValuationFormula(const std::string& n, const ValuationFormula& f) : ValuationFormula(f), name(new std::string(n)), vf(f.clone())
{
}

NamedValuationFormula::NamedValuationFormula(const std::string* n, ValuationFormula* f) : ValuationFormula(), name(n), vf(f)
{
	assert(vf != 0);

	types.insert(vf->getTypeVariables().begin(), vf->getTypeVariables().end());
	relvars.insert(vf->getRelationVariables().begin(), vf->getRelationVariables().end());
	atomvars.insert(vf->getAtomVariables().begin(), vf->getAtomVariables().end());
}

NamedValuationFormula::NamedValuationFormula(const NamedValuationFormula& f) : ValuationFormula(f), name(new std::string(*(f.name))), vf(f.vf->clone())
{
}

NamedValuationFormula& NamedValuationFormula::operator=(const NamedValuationFormula& f)
{
	if(&f == this) return *this;
	NamedValuationFormula nvf(f);
	std::swap(types, nvf.types);
	std::swap(relvars, nvf.relvars);
	std::swap(atomvars, nvf.atomvars);
	std::swap(name, nvf.name);
	std::swap(vf, nvf.vf);
	return *this;
}

ValuationFormula* NamedValuationFormula::clone() const
{
	assert(vf != 0 && name != 0);

	return new NamedValuationFormula(*this);
}

ValuationFormula* NamedValuationFormula::createPrimedVersion(const AtomVariable& av) const
{
	assert(vf != 0 && name != 0);

	return new NamedValuationFormula(new std::string(*name), vf->createPrimedVersion(av));
}

NamedValuationFormula::~NamedValuationFormula()
{
	delete vf;
	delete name;
}

const std::string& NamedValuationFormula::getName() const
{
	assert(name != 0);

	return *name;
}

const ValuationFormula& NamedValuationFormula::getValuationFormula() const
{
	assert(vf != 0);

	return *vf;
}

bool NamedValuationFormula::instance(const Valuation& val) const
{
	assert(vf != 0);

	return (vf->instance(val));
}

std::list<ValuationFormula*> NamedValuationFormula::conjuncts() const
{
	assert(vf != 0);

	return vf->conjuncts();
}

bool NamedValuationFormula::isExistentialFree(int numOfNegations) const
{
	assert(vf != 0);

	return (vf->isExistentialFree(numOfNegations));
}

bool NamedValuationFormula::isMembershipNegated(int numOfNegations) const
{
	assert(vf != 0);

	return (vf->isMembershipNegated(numOfNegations));
}

bool NamedValuationFormula::isUnquantified() const
{
	assert(vf != 0);

	return vf->isUnquantified();
}

void NamedValuationFormula::setNumericValue(VarType vt, int id, int var_id)
{
	assert(vf != 0);

	ValuationFormula::setNumericValue(vt,id,var_id);
	vf->setNumericValue(vt,id,var_id);
}

void NamedValuationFormula::setNumericValuesToBoundVariables(int val)
{
	assert(vf != 0);

	vf->setNumericValuesToBoundVariables(val);
}

std::string NamedValuationFormula::toString(const Context& c) const
{
	assert(name != 0);
	
	return *name;
}
