#include "reduce.h"
#include <string>
#include <iostream>
#include <vector>
#include <algorithm>
#include <map>
#include <ctime>
#include <sstream>
#include <limits>
#include <iomanip>

#ifdef _OPENMP
	#include "omp.h"
#endif

quartet<std::map<TypeVariable,int>*, std::map<TypeVariable,bool>*, bool, bool> determineCutoffs(const Context& ctxt, ValuationFormula*& vf, LTSSchema*& spec, LTSSchema*& sys, VerType vtype)
{
	bool sound = true;
	bool complete = true;
	
	std::set<TypeVariable, VariableCompare<TypeVariable> > typevars;
	typevars.insert(vf->getTypeVariables().begin(), vf->getTypeVariables().end());
	typevars.insert(spec->getTypeVariables().begin(), spec->getTypeVariables().end());
	typevars.insert(sys->getTypeVariables().begin(), sys->getTypeVariables().end());
	for(std::set<TypeVariable, VariableCompare<TypeVariable> >::iterator cit = typevars.begin(); cit != typevars.end(); ++cit)
	{
		if(!((sys->isProcessTypeVariable(*cit) && spec->isProcessTypeVariable(*cit)) || (sys->isDataTypeVariable(*cit) && spec->isDataTypeVariable(*cit))))
		{
			LTSSchema* modsys = sys->convertChoiceToParallelComposition(*cit);
			LTSSchema* modspec = spec->convertChoiceToParallelComposition(*cit);
			delete sys;
			delete spec;
			sys = modsys;
			spec = modspec;
//			std::cout << "Sys = " << sys->toString(ctxt) << std::endl << std::endl;
//			std::cout << "Spec = " << spec->toString(ctxt) << std::endl << std::endl;
		}
	}

	std::set<RelationVariable, VariableCompare<RelationVariable> > relvars;
	relvars.insert(vf->getRelationVariables().begin(), vf->getRelationVariables().end());
	relvars.insert(spec->getRelationVariables().begin(), spec->getRelationVariables().end());
	relvars.insert(sys->getRelationVariables().begin(), sys->getRelationVariables().end());
	std::set<AtomVariable, VariableCompare<AtomVariable> > atomvars;
	atomvars.insert(vf->getAtomVariables().begin(), vf->getAtomVariables().end());
	atomvars.insert(spec->getAtomVariables().begin(), spec->getAtomVariables().end());
	atomvars.insert(sys->getAtomVariables().begin(), sys->getAtomVariables().end());
	std::map<TypeVariable,int>* cutoffs = new std::map<TypeVariable,int>();
	std::map<TypeVariable,bool>* cutoffisexact = new std::map<TypeVariable,bool>;
	LTSSchema* sysspec = new ParallelLTSSchema(*sys, *spec);
	int i = 0;
	for(std::set<TypeVariable, VariableCompare<TypeVariable> >::iterator cit = typevars.begin(); cit != typevars.end(); ++cit)
	{
		std::cout << "Determining a cut-off (size) for the type " << ctxt.typeVariableToString(*cit) << "..." << std::unitbuf << std::endl;
		const_cast<TypeVariable&>(*cit).id = i;
		spec->setNumericValue(TYPEVAR,cit->first, i);
		sys->setNumericValue(TYPEVAR,cit->first, i);
		vf->setNumericValue(TYPEVAR,cit->first, i);
		int proccutoff = sysspec->numOfNestedOccurrences(*cit);
		int datacutoff = sysspec->numOfParallelOccurrences(*cit,*sysspec,*cutoffs);
		int cutoff = std::min(proccutoff,datacutoff);
		int freevars = 0;
		for(std::set<AtomVariable, VariableCompare<AtomVariable> >::iterator cait = atomvars.begin(); cait != atomvars.end(); ++cait)
		{
			const_cast<AtomVariable&>(*cait).setNumericValue(TYPEVAR,cit->first,i);
			if(*cit == cait->second) freevars++;
		}
		cutoff+=freevars;
		(*cutoffs)[*cit] = std::max(1,cutoff);

		ValuationFormula* relevantConjunct = new TopValuationFormula();
		std::list<ValuationFormula*> conjuncts = vf->conjuncts();
		int conjunctCount;
		std::set<TypeVariable> tvs;
		tvs.insert(*cit);
		do
		{
			conjunctCount = conjuncts.size();
			std::list<ValuationFormula*>::iterator lit = conjuncts.begin();
			while(lit != conjuncts.end())
			{
				if(disjoint<TypeVariable>((*lit)->getTypeVariables(), tvs))
				{
					++lit;
				}
				else	
				{
					relevantConjunct = new ConjunctiveValuationFormula(relevantConjunct, *lit);
					tvs.insert((*lit)->getTypeVariables().begin(), (*lit)->getTypeVariables().end());
					lit = conjuncts.erase(lit);
				}
			}
		}
		while(conjunctCount != conjuncts.size());		
		for(std::list<ValuationFormula*>::iterator lit = conjuncts.begin(); lit != conjuncts.end(); lit++)
		{
			delete *lit;
		}
		conjuncts.clear();
		
		int relevantRelvars = 0;
		for(std::set<RelationVariable>::iterator crit = relvars.begin(); crit != relvars.end(); crit++)
		{
			const_cast<RelationVariable&>(*crit).setNumericValue(TYPEVAR,cit->first,i);
			if(crit->second == *cit || crit->third == *cit) relevantRelvars++;
		}
		
		(*cutoffisexact)[*cit] = true;
		if(!(spec->isHidingFree()))
		{
			std::cout << "...Failed. The specification involves hiding." << std::endl;
			(*cutoffisexact)[*cit] = false;
			cutoff = CUTOFFMAX+1;
			sound = false;
		}
		else if(sys->involvesConditionalHiding())
		{
			(*cutoffisexact)[*cit] = false;
			std::cout << "...Failed. The system involves conditional hiding." << std::endl;
			cutoff = CUTOFFMAX+1;
			sound = false;
		}
		else if(vtype == TRACEREF && !(relevantConjunct->isExistentialFree()))
		{
			(*cutoffisexact)[*cit] = false;
			std::cout << "...Failed. Relations related to the type " << ctxt.typeVariableToString(*cit) << " involve existential quantification." << std::endl;
			cutoff = CUTOFFMAX+1;
			sound = false;
		}
		else if(vtype == COMPAT && dynamic_cast<TopValuationFormula*>(vf) == 0)
		{
			(*cutoffisexact)[*cit] = false;
			std::cout << "...Failed. Relations related to parametric types are not allowed when checking for compatibility." << std::endl;
			cutoff = CUTOFFMAX+1;
			sound = false;
		}
		else if(vtype == ALTSIM && dynamic_cast<TopValuationFormula*>(vf) == 0)
		{
			(*cutoffisexact)[*cit] = false;
			std::cout << "...Failed. Relations related to parametric types are not allowed when checking for alternating simulation." << std::endl;
			cutoff = CUTOFFMAX+1;
			sound = false;
		}
		else if(relevantRelvars > 0 && !sysspec->isProcessTypeVariable(*cit))
		{
			(*cutoffisexact)[*cit] = false;
			std::cout << "...Failed. There are both relations with the domain " << ctxt.typeVariableToString(*cit) << " and variables of the type " << ctxt.typeVariableToString(*cit) << " that are bound in an LTS." << std::endl;
			cutoff = CUTOFFMAX+1;
			sound = false;
		}
		else if(!((sys->isProcessTypeVariable(*cit) && spec->isProcessTypeVariable(*cit)) || (sys->isDataTypeVariable(*cit) && spec->isDataTypeVariable(*cit))))
		{
			std::cout << "...Failed. A variable of the type " << ctxt.typeVariableToString(*cit) << " appear bound both in a parallel composition and within an LTS." << std::endl;
			std::string input;
			do
			{
				if(vtype == TRACEREF && spec->isProcessTypeVariable(*cit))
				{
					std::cout << "Enter 'A' to apply abstraction or provide a cut-off (size) for the type " << ctxt.typeVariableToString(*cit) << std::endl;
					std::getline(std::cin, input);
					(*cutoffisexact)[*cit] = (input.compare("A") == 0 || input.compare("a") == 0);
					(*cutoffs)[*cit] = strtol(input.c_str(),0,0);
				}
				else
				{
					std::cout << "Please, provide a cut-off (size) for the type " << ctxt.typeVariableToString(*cit) << std::endl;
					(*cutoffisexact)[*cit] = false;
					std::cin >> (*cutoffs)[*cit];
				}		
			}
			while(((*cutoffs)[*cit] < 1 || (*cutoffs)[*cit] > CUTOFFMAX) && !(*cutoffisexact)[*cit]);
			if((*cutoffisexact)[*cit])
			{
				LTSSchema* modsys = sys->convertChoiceToParallelComposition(*cit,true);
//				std::cout << "Sys = " << modsys->toString(ctxt) << std::endl << std::endl;
				if(!modsys->isProcessTypeVariable(*cit))
				{
					(*cutoffisexact)[*cit] = false;
					std::cout << "...Failed. The abstraction did not yield a cut-off." << std::endl;				
					delete modsys;
					sound = false;
				}
				else
				{
					delete sys;
					delete sysspec;
					sys = modsys;
					sysspec = new ParallelLTSSchema(*sys, *spec);
					proccutoff = sysspec->numOfNestedOccurrences(*cit);
					datacutoff = sysspec->numOfParallelOccurrences(*cit,*sysspec,*cutoffs);
					cutoff = std::min(proccutoff,datacutoff) + freevars;
					(*cutoffs)[*cit] = std::max(1,cutoff);
					complete = false;
				}
			}
			else
			{
				sound = false;
			}
		}
		if((*cutoffisexact)[*cit] && cutoff > CUTOFFMAX)
		{
			(*cutoffisexact)[*cit] = false;
			std::cout << "...Failed. The computed cut-off succeeds the maximum value of " << CUTOFFMAX << '.' << std::endl;
			sound = false;
		}
		else if((*cutoffisexact)[*cit] && cutoff <= CUTOFFMAX)
		{
			std::cout << "...Succeeded. The cut-off size is " << std::max(1,cutoff) << "." << std::endl;
			std::string input;
			do
			{
				std::cout << "Press enter to accept it or provide a smaller cut-off to change it " << std::endl;
				std::getline(std::cin, input);
				(*cutoffisexact)[*cit] = (input.compare("") == 0);
				(*cutoffs)[*cit] = strtol(input.c_str(),0,0);
			}
			while(((*cutoffs)[*cit] < 1 || (*cutoffs)[*cit] > cutoff) && !(*cutoffisexact)[*cit]);
			if((*cutoffisexact)[*cit])
			{
				(*cutoffs)[*cit] = cutoff;
			}
			else
			{
				sound &= (cutoff <= (*cutoffs)[*cit]);
			}
		}
		if((*cutoffisexact)[*cit] == false && cutoff > CUTOFFMAX)
		{
			do
			{
				std::cout << "Please, provide a cut-off (size) for the type " << ctxt.typeVariableToString(*cit) << std::endl;
				std::cin >> (*cutoffs)[*cit];
			}
			while((*cutoffs)[*cit] < 1 || (*cutoffs)[*cit] > CUTOFFMAX);
			sound = false;
		}
		delete relevantConjunct;
		i++;
	}
	delete sysspec;
	
	i = 0;
	for(std::set<RelationVariable, VariableCompare<RelationVariable> >::iterator cit = relvars.begin(); cit != relvars.end(); ++cit)
	{
		const_cast<RelationVariable&>(*cit).id = i;
		spec->setNumericValue(RELVAR,cit->first, i);
		sys->setNumericValue(RELVAR,cit->first, i);
		vf->setNumericValue(RELVAR,cit->first, i);
		i++;
	}
	
	i = 0;
	for(std::set<AtomVariable, VariableCompare<AtomVariable> >::iterator cit = atomvars.begin(); cit != atomvars.end(); ++cit)
	{
		const_cast<AtomVariable&>(*cit).id = i;
		spec->setNumericValue(ATOMVAR, cit->first, i);
		sys->setNumericValue(ATOMVAR, cit->first, i);
		vf->setNumericValue(ATOMVAR, cit->first, i);
		i++;
	}
	spec->setNumericValuesToBoundVariables(atomvars.size());
	sys->setNumericValuesToBoundVariables(atomvars.size());
	vf->setNumericValuesToBoundVariables(atomvars.size());
	
	return quartet<std::map<TypeVariable,int>*, std::map<TypeVariable,bool>*, bool, bool>(cutoffs,cutoffisexact,sound,complete);
}

	
quartet<std::set<Valuation*,PointerCompare<Valuation> >*, int, int, int> computeValuations(const Context& ctxt, ValuationFormula*& vf, LTSSchema*& spec, LTSSchema*& sys, const std::map<TypeVariable,int>& cutoffs)
{	
	double computed = 0;
	double portion = 100;
	int valCount = 0;
	int canCount = 0;
	int storedCount = 0;
	int storedMax = 0;

	std::set<TypeVariable, VariableCompare<TypeVariable> > typevars;
	typevars.insert(spec->getTypeVariables().begin(), spec->getTypeVariables().end());
	typevars.insert(sys->getTypeVariables().begin(), sys->getTypeVariables().end());
	typevars.insert(vf->getTypeVariables().begin(), vf->getTypeVariables().end());
	std::set<RelationVariable, VariableCompare<RelationVariable> > relvars;
	relvars.insert(spec->getRelationVariables().begin(), spec->getRelationVariables().end());
	relvars.insert(sys->getRelationVariables().begin(), sys->getRelationVariables().end());
	relvars.insert(vf->getRelationVariables().begin(), vf->getRelationVariables().end());
	std::set<AtomVariable, VariableCompare<AtomVariable> > atomvars;
	atomvars.insert(vf->getAtomVariables().begin(), vf->getAtomVariables().end());
	atomvars.insert(spec->getAtomVariables().begin(), spec->getAtomVariables().end());
	atomvars.insert(sys->getAtomVariables().begin(), sys->getAtomVariables().end());

	portion = portion / (typevars.size() + relvars.size() + atomvars.size());

	ValuationFormula* msnvf = new TopValuationFormula();
	ValuationFormula* pvf = new TopValuationFormula();

	std::list<ValuationFormula*> conjuncts = vf->conjuncts();
	std::list<ValuationFormula*> relnegConjuncts = vf->conjuncts();
	std::list<ValuationFormula*>::iterator lit = relnegConjuncts.begin();
	while(lit != relnegConjuncts.end())
	{
		if((*lit)->isMembershipNegated())
		{
			++lit;
		}
		else
		{
			delete *lit;
			lit = relnegConjuncts.erase(lit);
		}	
	
	}

//	std::cout << std::endl << "Cut-offs for the (size of the values of) type variables: " << mapToString<TypeVariable, int>(cutoffs) << std::unitbuf << std::endl << std::endl;
	
	std::set<Valuation*,PointerCompare<Valuation> >* vals = new std::set<Valuation*,PointerCompare<Valuation> >();
	vals->insert(new Valuation());
	valCount = 1;
	storedCount = 1;
	
	std::cout << "Computing the values of parameters..." << std::endl;
	
	vals = computeValuations(typevars, relvars, atomvars, cutoffs, msnvf, pvf, conjuncts, relnegConjuncts, computed, portion, valCount, canCount, storedCount, storedMax, vals, false);

	std::cout << "...Done!" << std::setw(70) << " " << std::endl;
	
	std::cout << std::endl << "Found " << vals->size() << " non-isomorphic valuations." << std::endl;
	std::cout << "(#valuations: " << valCount << ", #canonical forms: " << canCount << ", max #valuations stored: " << storedMax << ')' << std::endl << std::endl;
	
	return quartet<std::set<Valuation*,PointerCompare<Valuation> >*, int, int, int>(vals, valCount, canCount, storedMax);
}


std::set<Valuation*,PointerCompare<Valuation> >* computeValuations(std::set<TypeVariable, VariableCompare<TypeVariable> > typevars, std::set<RelationVariable, VariableCompare<RelationVariable> > relvars, std::set<AtomVariable, VariableCompare<AtomVariable> > atomvars, const std::map<TypeVariable,int>& cutoffs, ValuationFormula* msnvf, ValuationFormula* pvf, std::list<ValuationFormula*>& conjuncts, std::list<ValuationFormula*>& relnegConjuncts, double& completed, double portion, int& valCount, int& canCount, int& storedCount, int& storedMax, std::set<Valuation*,PointerCompare<Valuation> >* vals, bool parallelised)
{
	boolean varpicked = false;

	if(typevars.size() > 0 || relvars.size() > 0 || atomvars.size() > 0)
	{	
		int avid = std::numeric_limits<int>::max();
		int rvid = std::numeric_limits<int>::max();
		int tvid = std::numeric_limits<int>::max();
		if(atomvars.size() > 0)
		{
			avid = atomvars.begin()->id;
		}
		if(relvars.size() > 0)
		{
			rvid = relvars.begin()->id;
		}
		if(typevars.size() > 0)
		{
			tvid = typevars.begin()->id;
		}
		if(avid < std::min(tvid,rvid) || (avid == std::min(rvid,tvid) && atomvars.size() > 0))
		{
			AtomVariable av = *(atomvars.begin());
			TypeVariable tv = av.second;
			if(typevars.find(tv) ==  typevars.end())
			{
//				std::cout << "Computing values for atom variable " << av.first << " ..." << std::endl;
	
				atomvars.erase(atomvars.begin());
				std::list<ValuationFormula*>::iterator lit = relnegConjuncts.begin();
				while(lit != relnegConjuncts.end())
				{
					if(disjoint<TypeVariable, VariableCompare<TypeVariable> >(typevars, (*lit)->getTypeVariables()) && disjoint<RelationVariable, VariableCompare<RelationVariable> >(relvars, (*lit)->getRelationVariables()) && disjoint<AtomVariable, VariableCompare<AtomVariable> >(atomvars, (*lit)->getAtomVariables()))
					{
						msnvf = new ConjunctiveValuationFormula(msnvf, *lit);
						lit = relnegConjuncts.erase(lit);
					}
					else
					{
						++lit;
					}
				}
				msnvf->setNumericValuesToBoundVariables(avid+1);
				//std::cout << "Relation-negated valuation formula is now " << msnvf->toString() << std::endl;
				lit = conjuncts.begin();
				while(lit != conjuncts.end())
				{
					if(disjoint<TypeVariable, VariableCompare<TypeVariable> >(typevars, (*lit)->getTypeVariables()) && disjoint<RelationVariable, VariableCompare<RelationVariable> >(relvars, (*lit)->getRelationVariables()) && disjoint<AtomVariable, VariableCompare<AtomVariable> >(atomvars, (*lit)->getAtomVariables()))
					{
						pvf = new ConjunctiveValuationFormula(pvf, *lit);
						lit = conjuncts.erase(lit);
					}
					else
					{
						++lit;
					}
				}
				pvf->setNumericValuesToBoundVariables(avid+1);
				//std::cout << "Valuation formula is now " << pvf->toString() << std::endl;
			
				int count = 0;
				int total = vals->size();
				std::set<Valuation*,PointerCompare<Valuation> >* tempvals = new std::set<Valuation*,PointerCompare<Valuation> >();
				std::swap(tempvals, vals);
				for(std::set<Valuation*,PointerCompare<Valuation> >::iterator sit = tempvals->begin(); sit != tempvals->end(); sit++)
				{
					//std::cout << "   " << std::setw(3) << std::right << 100*count++/total << "%   #val:" << std::setw(12) << std::left << valCount << " #can:" << std::setw(12) << canCount << " #sto:" << std::setw(12) << storedCount << '\r';
					std::list<Valuation*> vals2 = (*sit)->addAtomVariable(av.second);	
					delete *sit;
					#pragma omp atomic
					valCount += vals2.size();
					updateStoredCount(storedCount, vals2.size()-1, storedMax);
					printProgress(completed, portion*count++/total, valCount, canCount, storedCount);
					
					//std::cout << "   " << std::setw(3) << std::right << 100*count++/total << "%   #val:" << std::setw(12) << std::left << valCount << " #can:" << std::setw(12) << canCount << " #sto:" << std::setw(12) << storedCount << '\r';
					for(std::list<Valuation*>::iterator lit = vals2.begin(); lit != vals2.end(); ++lit)
					{
						if(pvf->instance(**lit))
						{
							(*lit)->computeCanonicalGraph();
							std::set<Valuation*, PointerCompare<Valuation> >::iterator it = vals->find(*lit);
							if(it == vals->end())
							{
								vals->insert(it, *lit);
							}
							else
							{
								delete *lit;
								updateStoredCount(storedCount, -1, storedMax);
							}
							#pragma omp atomic
							canCount++;
						}
						else
						{
							delete *lit;
							updateStoredCount(storedCount, -1, storedMax);
						}
					}
				}
				#pragma omp atomic
				completed += portion;
				delete tempvals;
				
				//std::cout << "...Done." << std::setw(60) << " " << std::endl;

				varpicked = true;
			}
		}
		
		if(!varpicked && (rvid < tvid || (rvid == tvid && relvars.size() > 0)))
		{
			RelationVariable rv = *(relvars.begin());
			if(typevars.find(rv.second) == typevars.end() && typevars.find(rv.third) == typevars.end())
			{
//				std::cout << "Computing values for relation variable " << rv.first << " ..." << std::endl;
	
				relvars.erase(relvars.begin());
				//spec->setNumericValue(rv, relvarCount);
				//sys->setNumericValue(rv, relvarCount);
				std::list<ValuationFormula*>::iterator lit = relnegConjuncts.begin();
				while(lit != relnegConjuncts.end())
				{
					//(*lit)->setNumericValue(rv, relvarCount);
					if(disjoint<TypeVariable, VariableCompare<TypeVariable> >(typevars, (*lit)->getTypeVariables()) && disjoint<RelationVariable, VariableCompare<RelationVariable> >(relvars, (*lit)->getRelationVariables()) && disjoint<AtomVariable, VariableCompare<AtomVariable> >(atomvars, (*lit)->getAtomVariables()))
					{
						msnvf = new ConjunctiveValuationFormula(msnvf, *lit);
						lit = relnegConjuncts.erase(lit);
					}
					else
					{
						++lit;
					}
				}
				msnvf->setNumericValuesToBoundVariables((*(vals->begin()))->getAtomVariableCount());
				//std::cout << "Relation-negated valuation formula is now " << msnvf->toString() << std::endl;
				lit = conjuncts.begin();
				while(lit != conjuncts.end())
				{
					//(*lit)->setNumericValue(rv, relvarCount);
					if(disjoint<TypeVariable, VariableCompare<TypeVariable> >(typevars, (*lit)->getTypeVariables()) && disjoint<RelationVariable, VariableCompare<RelationVariable> >(relvars, (*lit)->getRelationVariables()) && disjoint<AtomVariable, VariableCompare<AtomVariable> >(atomvars, (*lit)->getAtomVariables()))
					{
						pvf = new ConjunctiveValuationFormula(pvf, *lit);
						lit = conjuncts.erase(lit);
					}
					else
					{
						++lit;
					}
				}
				pvf->setNumericValuesToBoundVariables((*(vals->begin()))->getAtomVariableCount());
				//std::cout << "Valuation formula is now " << pvf->toString() << std::endl;
	
				int count = 0;
				int total = vals->size();
				std::set<Valuation*,PointerCompare<Valuation> >* tempvals = new std::set<Valuation*,PointerCompare<Valuation> >();
				std::swap(tempvals, vals);
				for(std::set<Valuation*,PointerCompare<Valuation> >::const_iterator sit = tempvals->begin(); sit != tempvals->end(); sit++)
				{
					//std::cout << "   " << std::setw(3) << std::right << 100*count++/total << "%   #val:" << std::setw(12) << std::left << valCount << " #can:" << std::setw(12) << canCount << " #sto:" << std::setw(12) << storedCount << '\r';
					std::list<Valuation*> vallist;
					Valuation* initval = (*sit)->addRelationVariable(rv.second, rv.third);
					delete *sit;
					int valCountIncrement = search(initval, &vallist, msnvf);
					#pragma omp atomic
					valCount += valCountIncrement;
					updateStoredCount(storedCount, vallist.size()-1, storedMax);
					printProgress(completed, portion*count++/total, valCount, canCount, storedCount);
										
					//std::cout << "   " << std::setw(3) << std::right << 100*count++/total << "%   #val:" << std::setw(12) << std::left << valCount << " #can:" << std::setw(12) << canCount << " #sto:" << std::setw(12) << storedCount << '\r';
					for(std::list<Valuation*>::iterator lit = vallist.begin(); lit != vallist.end(); ++lit)
					{
						if(pvf->instance(**lit))
						{
							(*lit)->computeCanonicalGraph();
							std::set<Valuation*, PointerCompare<Valuation> >::iterator it = vals->find(*lit);
							if(it == vals->end())
							{
								vals->insert(it, *lit);
							}
							else
							{
								delete *lit;
								updateStoredCount(storedCount, -1, storedMax);
							}
							#pragma omp atomic
							canCount++;
						}
						else
						{
							delete *lit;
							updateStoredCount(storedCount, -1, storedMax);
						}
					}
				}
				#pragma omp atomic
				completed += portion;
				delete tempvals;
				//std::cout << "...Done." << std::setw(60) << " " << std::endl;
				
				varpicked = true;
			}
		}
		
		if(!varpicked && (typevars.size() > 0))
		{
			TypeVariable tv = *(typevars.begin());
			//std::cout << "Computing values for type variable " << tv << " ..." << std::endl;

			typevars.erase(typevars.begin());
			/*
			type2int[tv] = typeCount;
			spec->setNumericValue(tv, typeCount);
			sys->setNumericValue(tv, typeCount);
			for(std::list<ValuationFormula*>::iterator lit = relnegConjuncts.begin(); lit != relnegConjuncts.end(); ++lit)
			{
				(*lit)->setNumericValue(tv, typeCount);
			}
			for(std::list<ValuationFormula*>::iterator lit = conjuncts.begin(); lit != conjuncts.end(); ++lit)
			{
				(*lit)->setNumericValue(tv, typeCount);
			}
			*/

			int count = 0;
			int total = vals->size();
			std::set<Valuation*,PointerCompare<Valuation> >* tempvals = new std::set<Valuation*,PointerCompare<Valuation> >();
			std::swap(tempvals, vals);
			for(std::set<Valuation*,PointerCompare<Valuation> >::iterator sit = tempvals->begin(); sit != tempvals->end(); sit++)
			{
				std::list<Valuation*> vals2 = (*sit)->addTypeVariable(cutoffs.find(tv)->second);
				delete *sit;
				#pragma omp atomic
				valCount += vals2.size();
				updateStoredCount(storedCount, vals2.size()-1, storedMax);
				printProgress(completed, portion*count++/total, valCount, canCount, storedCount);
				vals->insert(vals2.begin(), vals2.end());
			}
			#pragma omp atomic
			completed += portion;
/*			int cutoff = cutoffs.find(tv)->second;
			std::set<Valuation*,PointerCompare<Valuation> >* vals2[cutoff];
			for(int size = cutoff - 1; size >= 0; size--)
			{
//				#pragma omp task shared(cutoffs,conjuncts,relnegConjuncts,valCount,canCount,storedCount,storedMax,vals2)
				{
					vals2[size] = new std::set<Valuation*,PointerCompare<Valuation> >();
					for(std::set<Valuation*,PointerCompare<Valuation> >::iterator sit = tempvals->begin(); sit != tempvals->end(); sit++)
					{
						std::cout << "   " << std::setw(3) << std::right << 100*count++/total << "%   #val:" << std::setw(12) << std::left << valCount << " #can:" << std::setw(12) << canCount << " #sto:" << std::setw(12) << storedCount << '\r';
						vals2[size]->insert((*sit)->addTypeVariable(size+1));
						valCount++;
						storedCount++;
						storedMax = std::max(storedCount, storedMax);
					}
					std::list<ValuationFormula*> newConjuncts = cloneConjuncts(conjuncts);
					std::list<ValuationFormula*> newRelnegConjuncts = cloneConjuncts(relnegConjuncts);
					vals2[size] = computeValuations(spec, sys, typevars, relvars, atomvars, type2int, cutoffs, msnvf->clone(), pvf->clone(), newConjuncts, newRelnegConjuncts, valCount, canCount, storedCount, storedMax, typeCount, relvarCount, atomvarCount, vals2[size]);
				}
			}
//			#pragma omp taskwait
			deleteConjuncts(conjuncts);
			deleteConjuncts(relnegConjuncts);			
			storedCount -= tempvals->size();
			for(std::set<Valuation*,PointerCompare<Valuation> >::iterator sit = tempvals->begin(); sit != tempvals->end(); sit++)
			{
					delete *sit;
			}
*/
			delete tempvals;

/*			for(int size = 0; size < cutoff; size++)
			{
				vals->insert(vals2[size]->begin(), vals2[size]->end());
				delete vals2[size];
			}
*/
			//std::cout << "...Done." << std::setw(60) << " " << std::endl;
			varpicked = true;
		}

		int thread_count = 1;
		#ifdef _OPENMP
			thread_count = omp_get_num_threads();
			//std::cout << "Thread count " << thread_count << std::endl;
		#endif
		if(!parallelised && vals->size() >= 8*thread_count)
		{
			int task_count = vals->size();
//			std::cout << "about to start " << task_count << " threads\n";
			std::set<Valuation*,PointerCompare<Valuation> >* vals_part[task_count];
			int partindex;
			for(partindex = 0; partindex < task_count; partindex++)
			{
				vals_part[partindex] = new std::set<Valuation*,PointerCompare<Valuation> >();
			}
			partindex = 0;
			for(std::set<Valuation*,PointerCompare<Valuation> >::iterator sit = vals->begin(); sit != vals->end(); sit++)
			{
				vals_part[partindex]->insert(*sit);
				partindex = (partindex + 1) % task_count;
			}
			vals->clear();
			
//			#pragma omp parallel for 
			for(partindex = 0; partindex < task_count; partindex++)
			{
				#pragma omp task firstprivate(typevars,relvars,atomvars) shared(cutoffs,conjuncts,relnegConjuncts,completed,valCount,canCount,storedCount,storedMax,msnvf,pvf,portion,task_count,vals_part)
				{
					std::list<ValuationFormula*> newConjuncts = cloneConjuncts(conjuncts);
					std::list<ValuationFormula*> newRelnegConjuncts = cloneConjuncts(relnegConjuncts);
					vals_part[partindex] = computeValuations(typevars, relvars, atomvars, cutoffs, msnvf->clone(), pvf->clone(), newConjuncts, newRelnegConjuncts, completed, portion/task_count, valCount, canCount, storedCount, storedMax, vals_part[partindex], true);
				}
			}
			#pragma omp taskwait
			deleteConjuncts(conjuncts);
			deleteConjuncts(relnegConjuncts);			
			for(partindex = 0; partindex < task_count; partindex++)
			{
				for(std::set<Valuation*,PointerCompare<Valuation> >::iterator sit = vals_part[partindex]->begin(); sit != vals_part[partindex]->end(); sit++)
				{
					vals->insert(*sit);
				}
				delete vals_part[partindex];
			}
		}
		else if(vals->size() > 0)
		{		
			vals = computeValuations(typevars, relvars, atomvars, cutoffs, msnvf->clone(), pvf->clone(), conjuncts, relnegConjuncts, completed, portion, valCount, canCount, storedCount, storedMax, vals, parallelised);
		}
	}
	
	delete msnvf;
	delete pvf;
	
	return vals;
}

void updateStoredCount(int& storedCount, int storedCountIncrement, int& storedMax)
{
	#pragma omp critical
	{
		storedCount += storedCountIncrement;
		storedMax = std::max(storedCount, storedMax);
	}
}

void printProgress(double& completed, double increment, int& valCount, int& canCount, int& storedCount)
{
	#pragma omp critical
	{
		std::cout << "   " << std::setw(3) << std::right << int(completed + increment) << "%   #val:" << std::setw(12) << std::left << valCount << " #can:" << std::setw(12) << canCount << " #sto:" << std::setw(12) << storedCount << '\r';
	}
}

int search(Valuation* val, std::list<Valuation*>* vals, const ValuationFormula* vf)
{	
	int iter = 1;
	if(vf->instance(*val))
	{
		vals->push_back(val);
		std::list<Valuation*> children = val->children();
		for(std::list<Valuation*>::iterator lit = children.begin(); lit != children.end(); ++lit)
		{
			iter+= search(*lit, vals, vf);
		}
	}
	else
	{
		delete val;
	}
	return iter;
}

std::list<ValuationFormula*> cloneConjuncts(const std::list<ValuationFormula*>& conjuncts)
{
	std::list<ValuationFormula*> newConjuncts = std::list<ValuationFormula*>();
	for(std::list<ValuationFormula*>::const_iterator clit = conjuncts.begin(); clit != conjuncts.end(); clit++)
	{
		newConjuncts.push_back((*clit)->clone());
	}
	return newConjuncts;
}

void deleteConjuncts(std::list<ValuationFormula*>& conjuncts)
{
	for(std::list<ValuationFormula*>::iterator clit = conjuncts.begin(); clit != conjuncts.end(); clit++)
	{
		delete *clit;
	}
	conjuncts.clear();
}
