#include "fdr2output4trref.h"
#include <fstream>
#include <sstream>
#include <list>
#include <set>
#include <map>
#include <iomanip>
#include <ctime>
#include <cstdio>
#include "reduce.h"

int TraceRefinementOutput::generateInstances(const Context& ctxt, const LTSSchema* spec, const LTSSchema* sys, const ValuationFormula* vf, const std::set<Valuation*,PointerCompare<Valuation> >* vals, const std::string& filenameprefix, const std::map<TypeVariable,bool>& cutoffisexact, bool sound, bool complete)
{
	std::set<TypeVariable> types = sys->getTypeVariables();
	types.insert(spec->getTypeVariables().begin(), spec->getTypeVariables().end());
	types.insert(vf->getTypeVariables().begin(), vf->getTypeVariables().end());
	std::set<RelationVariable> relvars = sys->getRelationVariables();
	relvars.insert(spec->getRelationVariables().begin(), spec->getRelationVariables().end());
	relvars.insert(vf->getRelationVariables().begin(), vf->getRelationVariables().end());
	std::set<AtomVariable> atomvars = sys->getAtomVariables();
	atomvars.insert(spec->getAtomVariables().begin(), spec->getAtomVariables().end());
	atomvars.insert(vf->getAtomVariables().begin(), vf->getAtomVariables().end());
	std::set<TypeVariable> datatypes;
	std::set<TypeVariable> proctypes;
	for(std::set<TypeVariable>::const_iterator ctit = types.begin(); ctit != types.end(); ctit++)
	{
		if(spec->isDataTypeVariable(*ctit) && sys->isDataTypeVariable(*ctit)) datatypes.insert(*ctit);
		if(spec->isProcessTypeVariable(*ctit) && sys->isProcessTypeVariable(*ctit)) proctypes.insert(*ctit);
	}
	std::list<triplet<InstanceOfLTSSchema*, InstanceOfLTSSchema*, int> > instances;
	bool isdeterministic = true;
	bool isCorrect = true;
	int instCount = 0;
	int preservedInstCount = 0;

	bool isDeleted = true;
	int delCount = 0;
	while(isDeleted)
	{
		std::ostringstream filename;
		filename << filenameprefix << "_instance_" << delCount++ << ".csp";
		isDeleted = isDeleted && (std::remove(filename.str().c_str()) == 0);
	}

	for(std::set<Valuation*,PointerCompare<Valuation> >::iterator sit = vals->begin(); sit != vals->end(); sit++)
	{
		#pragma omp task shared(types,atomvars,relvars,spec,sys,instances,isdeterministic,filenameprefix,isCorrect,preservedInstCount,ctxt,cutoffisexact)
		{
			std::stringstream output;
			output << "Generating Instance " << instCount << " generated by valuation" << std::endl;
			for(std::set<TypeVariable>::const_iterator ctit = types.begin(); ctit != types.end(); ++ctit)
			{
				AtomSet* as = (*sit)->getValue(*ctit);
				output << ctit->toString(ctxt) << " -> " << as->toString(ctxt) << std::endl;
				delete as;
			}
			for(std::set<RelationVariable>::const_iterator crit = relvars.begin(); crit != relvars.end(); ++crit)
			{
				AtomTupleSet* ats = (*sit)->getValue(*crit);
				output << crit->toString(ctxt) << " -> " << ats->toString(ctxt) << std::endl;
				delete ats;
			}
			for(std::set<AtomVariable>::const_iterator cait = atomvars.begin(); cait != atomvars.end(); ++cait)
			{
				output << cait->toString(ctxt) << " -> " << ctxt.atomToString((*sit)->getValue(*cait)) << std::endl;
			}
			output << std::endl;
			printString(output.str());

			bool keep = true;

			for(std::set<TypeVariable>::const_iterator ctit = proctypes.begin(); ctit != proctypes.end(); ++ctit)
			{
				if(cutoffisexact.at(*ctit))
				{	
					AtomSet* typeval = (*sit)->getValue(*ctit);
					for(std::set<AtomVariable>::const_iterator cavit = atomvars.begin(); cavit != atomvars.end(); ++cavit)
					{
						typeval->erase((*sit)->getValue(*cavit));
					}
					if(sys->instanceCoveredBySmallerOnes(**sit,*typeval) && spec->instanceCoveredBySmallerOnes(**sit,*typeval))
					{
						delete typeval;
						output.str("");
						output << "Discarding Instance " << instCount << ". The instance has no behaviour or is already covered by smaller ones." << std::endl << std::endl;
						printString(output.str());
						keep = false;
						break;
					}
					delete typeval;
				}
			}

			if(keep)
			{
				PartialInstanceOfLTSSchema* syspinst = sys->partialInstance(**sit);
				PartialInstanceOfLTSSchema* specpinst = spec->partialInstance(**sit);

				for(std::set<TypeVariable>::const_iterator ctit = datatypes.begin(); ctit != datatypes.end(); ++ctit)
				{
					if(cutoffisexact.at(*ctit))
					{
						AtomSet* typeval = (*sit)->getValue(*ctit);
						int typesize = typeval->size();
						AtomSet* freeatoms = new AtomSet();
						for(std::set<AtomVariable>::const_iterator cavit = atomvars.begin(); cavit != atomvars.end(); ++cavit)
						{
							Atom a = (*sit)->getValue(*cavit);
							if(typeval->erase(a))
							{
								freeatoms->insert(a);
							}
						}
						int freecount = freeatoms->size();
						int sysidtr, specidst, specidtr;
						int sysidtr2 = syspinst->numOfOccurrencesInTransitions(*typeval);
						int specidst2 = specpinst->numOfOccurrencesInStates(*typeval);
						int specidtr2 = specpinst->numOfOccurrencesInTransitions(*typeval);
						delete freeatoms;
						do
						{
							sysidtr = sysidtr2;
							specidst = specidst2;
							specidtr = specidtr2;
							if(typesize > 1 && typesize > freecount + std::max(specidtr, sysidtr) + specidst)
							{	
								output.str("");
								output << "Discarding Instance " << instCount << ". The instance is already covered by smaller ones." << std::endl << std::endl;
								printString(output.str());
								keep = false;
								break;
							}
							else
							{
								syspinst->composeLeaves(MAX_SYS_COMP_SIZE);
								specpinst->composeLeaves(MAX_SPEC_COMP_SIZE);
								sysidtr2 = syspinst->numOfOccurrencesInTransitions(*typeval);
								specidst2 = specpinst->numOfOccurrencesInStates(*typeval);
								specidtr2 = specpinst->numOfOccurrencesInTransitions(*typeval);
							}
						}
						while(keep && (sysidtr2 < sysidtr || specidst2 < specidst || specidtr2 < specidtr));
						delete typeval;
					}
				}

				if(keep)
				{
					InstanceOfLTSSchema* sysinst = syspinst->instance();
					InstanceOfLTSSchema* specinst = specpinst->instance();

					if(datatypes.size() > 0 && isdeterministic)
					{
						bool thisisdet = specinst->isDeterministic();
						#pragma omp atomic
						isdeterministic &= thisisdet;
					}

					std::ostringstream filename;
					#pragma omp critical
					{
						filename << filenameprefix << "_instance_" << preservedInstCount << ".csp";
						preservedInstCount++;
					}
					bool isCorrect1 = generateInstance(ctxt, specinst, sysinst, instCount, filename.str(), sound, complete);
					#pragma omp atomic
					isCorrect &= isCorrect1;						
					
					delete sysinst;
					delete specinst;
				}
				
				delete syspinst;
				delete specpinst;
			}
			
			delete *sit;
		}
		instCount++;
	}
	#pragma omp taskwait
	
	if(sound && !isdeterministic)
	{
		std::stringstream output;
		output << "NOTE! The specification is not deterministic. The reduction is not sound." << std::endl << std::endl;
		printString(output.str());
		sound = false;
	}
	
	std::ostringstream filename;
	filename << filenameprefix << "_instance_" << preservedInstCount << ".csp";
	std::ofstream file(filename.str().c_str(), std::fstream::out | std::fstream::trunc);
	if(file.is_open())
	{
		file << "-- Bounds2.2 end of trace refinement check --" << std::endl;
		file << "-- The model is " << (complete ? "" : "*NOT* ") << "complete --" << std::endl;
		file << "-- The model is " << (sound ? "" : "*NOT* ") << "sound --" << std::endl << std::endl;
		file.close();
	}
	else
	{
		std::stringstream output;
		output << "Error in opening file " << filename << std::endl << std::endl;
		printString(output.str());
	}

	return preservedInstCount;
}

bool TraceRefinementOutput::generateInstance(const Context& ctxt, const InstanceOfLTSSchema* specinst, const InstanceOfLTSSchema* sysinst, int num, const std::string& filename, bool sound, bool complete)
{
	bool isCorrect = false;
	std::stringstream output;

	std::ofstream file(filename.c_str(), std::fstream::out | std::fstream::trunc);
//	std::stringstream file(std::fstream::out | std::fstream::trunc);
	if(file.is_open())
	{
		file << "-- Bounds2.2 trace refinement check --" << std::endl;
		file << "-- The model is " << (complete ? "" : "*NOT* ") << "complete --" << std::endl;
		file << "-- The model is " << (sound ? "" : "*NOT* ") << "sound --" << std::endl << std::endl;

		file << "transparent normalise" << std::endl << std::endl;

		ActionSet chans = sysinst->getActionSet();
		ActionSet chans2 = specinst->getActionSet();
		chans.insert(chans2.begin(), chans2.end());
		chans2.clear();
		for(ActionSet::const_iterator cit = chans.begin(); cit != chans.end(); ++cit)
		{
			file << "channel " << cit->toString(ctxt,'_','_','_',' ',' ') << std::endl;
		}
		file << "channel tau__" << std::endl;
		file << std::endl;

		toFDRFormattedStream(ctxt, file, sysinst, "Sys");
		toFDRFormattedStream(ctxt, file, specinst, "Spec");

		file << "assert CHAOS(alph_Spec_Proc_1) [T= CHAOS(alph_Sys_Proc_1)" << std::endl;
		file << "assert CHAOS(alph_Sys_Proc_1) [T= CHAOS(alph_Spec_Proc_1)" << std::endl;
		file << "assert Spec_Proc_1 [T= Sys_Proc_1\n";
		
		
		if(file.good())
		{
			output << "Instance " << num << " written successfully to file " << filename << std::endl << std::endl;
			printString(output.str());
			isCorrect = true;
		}
		else
		{
			output << "An error has occurred while writing Instance " << num << " to a file." << std::endl << std::endl;
			printString(output.str());
		}
		file.close();
	}
	else
	{
		output << "Error in opening file " << filename << std::endl << std::endl;
		printString(output.str());
	}

	return isCorrect;
}


void TraceRefinementOutput::printString(const std::string& ss)
{
	#pragma omp critical
	std::cout << ss;
};


std::ostream& TraceRefinementOutput::toFDRFormattedStream(const Context& ctxt, std::ostream& os, const InstanceOfLTSSchema* i, const std::string& name, bool compress, int depth)
{
	if(const ParallelInstanceOfLTSSchema* pi = dynamic_cast<const ParallelInstanceOfLTSSchema*>(i))
	{
		toFDRFormattedStream(ctxt, os, *pi, name, compress, depth);
	}
/*	else if(const NamedInstanceOfLTSSchema* ni = dynamic_cast<const NamedInstanceOfLTSSchema*>(i))
	{
		toFDRFormattedStream(ctxt, os, *ni, name, compress, depth);
	}
*/	else if(const UnitLTS* id = dynamic_cast<const UnitLTS*>(i))
	{
		toFDRFormattedStream(ctxt, os, *id, name, compress, depth);
	}
	else if(const HidingInstanceOfLTSSchema* hi = dynamic_cast<const HidingInstanceOfLTSSchema*>(i))
	{
		toFDRFormattedStream(ctxt, os, *hi, name, compress, depth);
	}
	else if(const ElementaryInstanceOfLTSSchema* ei = dynamic_cast<const ElementaryInstanceOfLTSSchema*>(i))
	{
		toFDRFormattedStream(ctxt, os, *ei, name, compress, depth);
	}
	return os;
}

std::ostream& TraceRefinementOutput::toFDRFormattedStream(const Context& ctxt, std::ostream& os, const UnitLTS& i, const std::string& name, bool compress, int depth)
{
	os << name << "_Proc_" << depth << " = STOP" << std::endl;
	os << "alph_" << name << "_Proc_" << depth << " = {}" << std::endl << std::endl;
	return os;
}

std::ostream& TraceRefinementOutput::toFDRFormattedStream(const Context& ctxt, std::ostream& os, const ElementaryInstanceOfLTSSchema& i, const std::string& name, bool compress, int depth)
{
	std::ostringstream procname;
	procname << name << "_Proc_" << depth;
	os << procname.str() << " =" << std::endl;
	os << "\tlet" << std::endl;
	const TransitionSet& tran = i.getTransitions();
	StateSet visited;
	State prevstate(ControlState(-1,-1));
	for(TransitionSet::const_iterator ctit = tran.begin(); ctit != tran.end(); ++ctit)
	{
		visited.insert(ctit->first);
		if(ctit->first != prevstate)
		{
//			os << "\t\t" << *(ctit->first) << " =\t(" << toFDRFormattedString(*(ctit->second), val) << " -> " << *(ctit->third) << ")" << std::endl;
			os << "\t\t" << ctit->first.toString(ctxt,'_','_','_') << " =\t(" << ctit->second.toString(ctxt,'_','_','_',' ',' ') << " -> " << ctit->third.toString(ctxt,'_','_','_') << ")" << std::endl;
			prevstate = ctit->first;
		}
		else
		{
//			os << "\t\t\t[] (" << toFDRFormattedString(*(ctit->second), val) << " -> " << *(ctit->third) << ")" << std::endl;
			os << "\t\t\t[] (" << ctit->second.toString(ctxt,'_','_','_',' ',' ') << " -> " << ctit->third.toString(ctxt,'_','_','_') << ")" << std::endl;
		}
	}
	const StateSet& states = tran.getStateSet();
	for(StateSet::const_iterator csit = states.begin(); csit != states.end(); ++csit)
	{
		if(visited.find(*csit)==visited.end())
		{
			os << "\t\t" << csit->toString(ctxt,'_','_','_') << " = STOP" << std::endl;
		}
	}
	os << "\twithin normalise(" << i.getInitialState().toString(ctxt,'_','_','_') << " \\ {tau__})" << std::endl << std::endl;
	os << "alph_" << procname.str() << " = ";
//	toFDRFormattedStream(os, l.getActionSet(), val);
	toFDRFormattedStream(ctxt, os, i.getActionSet());
	os << std::endl << std::endl;
	return os;
}

std::ostream& TraceRefinementOutput::toFDRFormattedStream(const Context& ctxt, std::ostream& os, const ParallelInstanceOfLTSSchema& i, const std::string& name, bool compress, int depth)
{
	std::string normalise;
	if(compress) normalise = "normalise";
	std::ostringstream procname;
	procname << name << "_Proc_" << depth;
	int leftdepth = 2*depth;
	int rightdepth = leftdepth + 1;
	std::ostringstream leftname;
	leftname << name << "_Proc_" << leftdepth;
	std::ostringstream rightname;
	rightname << name << "_Proc_" << rightdepth;
	toFDRFormattedStream(ctxt, os, &(i.getOneInstance()), name, compress, leftdepth);
	toFDRFormattedStream(ctxt, os, &(i.getOtherInstance()), name, compress, rightdepth);
	os << procname.str() << " = " << normalise << '(' << leftname.str() << " [alph_" << leftname.str() << "||alph_" << rightname.str() << "] " << rightname.str() << ')' << std::endl;
	os << "alph_" << procname.str() << " = union(alph_" << leftname.str() << ", alph_" << rightname.str() << ')' << std::endl << std::endl;
	return os;
}

std::ostream& TraceRefinementOutput::toFDRFormattedStream(const Context& ctxt, std::ostream& os, const HidingInstanceOfLTSSchema& i, const std::string& name, bool compress, int depth)
{
	std::string normalise;
	if(compress) normalise = "normalise";
	std::ostringstream procname;
	procname << name << "_Proc_" << depth;
	int leftdepth = 2*depth;
	int rightdepth = leftdepth + 1;
	std::ostringstream leftname;
	leftname << name << "_Proc_" << leftdepth;
	std::ostringstream rightname;
	rightname << name << "_Set_" << rightdepth;
	toFDRFormattedStream(ctxt, os, &(i.getInstanceOfLTSSchema()), name, compress, leftdepth);
	toFDRFormattedStream(ctxt, os, i.getHidingSet(), name, rightdepth);
	os << procname.str() << " = " << normalise << '(' << leftname.str() << " \\ " << rightname.str() << ')' << std::endl;
	os << "alph_" << procname.str() << " = diff(alph_" << leftname.str() << ", " << rightname.str() << ')' << std::endl << std::endl;
	return os;
}

std::ostream& TraceRefinementOutput::toFDRFormattedStream(const Context& ctxt, std::ostream& os, const ActionSet& as, const std::string& name, int depth)
{
	os << name << "_Set_" << depth << " = ";
	toFDRFormattedStream(ctxt, os, as);
	os << std::endl << std::endl;
	return os;
}

/*
std::ostream& TraceRefinementOutput::toFDRFormattedStream(const Context& ctxt, std::ostream& os, const NamedInstanceOfLTSSchema& i, const std::string& name, bool compress, int depth)
{
	toFDRFormattedStream(ctxt, os, &(i.getInstanceOfLTSSchema()), name, compress, depth);
	return os;
}
*/

std::ostream& TraceRefinementOutput::toFDRFormattedStream(const Context& ctxt, std::ostream& os, const ActionSet& as)
{
	ActionSet::const_iterator cait = as.begin();
	os << '{';
	if(cait != as.end())
	{
		os << cait->toString(ctxt,'_','_','_',' ',' ');
		cait++;
	}
	for(; cait != as.end(); cait++)
	{
		os << ',' << cait->toString(ctxt,'_','_','_',' ',' ');
	}
	os << '}';
	return os;
}


