;;; Helper procedures for handling streams

;; Return element n from the stream s (the first element is numbered
;; 0)
(define (stream-ref s n)
  (if (= n 0)
      (stream-car s)
      (stream-ref (stream-cdr s) (- n 1))))

;; The equivalent of map for streams.
(define (stream-map proc . argstreams)
  (if (stream-null? (car argstreams))
      the-empty-stream
      (cons-stream
       (apply proc (map stream-car argstreams))
       (apply stream-map
	      (cons proc (map stream-cdr argstreams))))))

;; Call proc on each element of the stream s (which must be finite for
;; this to terminate)
(define (stream-for-each proc s)
  (if (stream-null? s)
      'done
      (begin (proc (stream-car s))
             (stream-for-each proc (stream-cdr s)))))

;; Display all the elements of a stream (which must be finite...)
(define (display-stream s)
  (stream-for-each
   (lambda (x)
     (display x)
     (newline))
   s))

;; Return a (finite) stream low, low+1, ..., high-1, high
(define (stream-enumerate-interval low high)
  (if (> low high)
      the-empty-stream
      (cons-stream
       low
       (stream-enumerate-interval (+ low 1) high))))

;; Filter a stream, keeping only elements for which pred returns true
(define (stream-filter pred stream)
  (cond ((stream-null? stream) the-empty-stream)
        ((pred (stream-car stream))
         (cons-stream (stream-car stream)
                      (stream-filter pred
                                     (stream-cdr stream))))
        (else (stream-filter pred (stream-cdr stream)))))

;; Return a stream of integers n, n+1, n+2, ...
(define (integers-starting-from n)
  (cons-stream n (integers-starting-from (+ n 1))))

;; The stream 1, 2, 3, 4, ...
(define integers (integers-starting-from 1))

;; The stream 1, 1, 1, 1, ...
(define ones (cons-stream 1 ones))

;; Add two streams of numbers together, elementwise
(define (add-streams s1 s2)
  (stream-map + s1 s2))

;; Multiply all the elements of a stream of numbers with a constant
;; factor
(define (scale-stream stream factor)
  (stream-map (lambda (x) (* x factor)) stream))

;; Append two streams together directly. s1 must be a finite stream
;; for this to be useful.
(define (stream-append s1 s2)
  (if (stream-null? s1)
      s2
      (cons-stream (stream-car s1)
                   (stream-append (stream-cdr s1) s2))))

;; Create a stream which includes all the elements of s1 and s2
;; interleaved
(define (interleave s1 s2)
  (if (stream-null? s1)
      s2
      (cons-stream (stream-car s1)
                   (interleave s2 (stream-cdr s1)))))

;; Pairs
(define (pairs s t)
  (cons-stream
   (list (stream-car s) (stream-car t))
   (interleave
    (stream-map (lambda (x) (list (stream-car s) x))
                (stream-cdr t))
    (pairs (stream-cdr s) (stream-cdr t)))))


;; Convert a list to a (finite) stream
(define (list->stream l)
  (if (null? l)
      the-empty-stream
      (cons-stream (car l)
		   (list->stream (cdr l)))))

;; Return elements i..j (inclusive) of the stream s as a list (the
;; first element is numbered 0)
(define (part-of-stream->list stream i j)
  (define (recurse this rest)
    (cond ((or (> this j) (stream-null? rest))
	   '())
	  ((< this i)
	   (recurse (+ this 1) (stream-cdr rest)))
	  (else
	   (cons (stream-car rest)
		 (recurse (+ this 1) (stream-cdr rest))))))
  (recurse 0 stream))

'STREAMS-SUPPORT-LOADED
