;;;
;;; Hierarkia - Scheme Implementation
;;; ai.scm: AI object implementation
;;;
;;; Copyright 2003 Sami Virpioja, Ilari Lhteenmki
;;;
;;; ---
;;;
;;; This file is part of Hierarkia - Scheme Implementation.
;;;
;;; Hierarkia - Scheme Implementation is free software; you can redistribute 
;;; it and/or modify it under the terms of the GNU General Public License as 
;;; published by the Free Software Foundation; either version 2 of the 
;;; License, or (at your option) any later version.
;;;
;;; Hierarkia - Scheme Implementation  is distributed in the hope that it will 
;;; be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
;;; Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License along 
;;; with Hierarkia - Scheme Implementation; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;
;;; ---
;;;
;;; See README for more details.
;;;

;;
;; A message dispatching style object representing an AI player
;;

(define (make-pvs-ai gui . args)

  (define max-depth 3)
  (define max-rdepth 3)

  ;; transitions :: hash-table for saving evaluation of redundant states
  ;;
  ;; Pair (board, turn) (identifies the state of game) is used as a key and 
  ;; evaluation of the state (calculated utility) and corresponding node in 
  ;; the move tree are consed together to make the data

  ;; ((value . fail-ind) depth . node)
  ;; caar - value
  ;; cdar - 0: between bounds, 1: fail high, -1: fail low
  ;; cadr - depth
  ;; cddr - node

  (define transitions (make-hash-table 'equal))
  
  (define saved-transitions (make-hash-table 'equal))

  ;; get-saved-successors :: Searches both transition tables and
  ;; returns a list of tree elements. If first value is false,
  ;; state wasn't found in tables. If it is true, state was found
  ;; but it was calculated deeper in the search tree than depth.
  ;; Otherwise it is the value of the saved node. Second return
  ;; value is depth of the node in the table or false. Third
  ;; value is node's successors, either from the table or from the
  ;; current state using get-successors -procedure.
  
  (define (get-saved-successors board turn depth)
    (let ((new (hash-table-get transitions (cons board turn)
			       (lambda () 'undef)))
	  (saved (hash-table-get saved-transitions (cons board turn)
				 (lambda () 'undef))))
      (cond ((and (eq? new 'undef) (eq? saved 'undef))
	     (list #f #f (lambda args (get-successors board turn))))
	    ((eq? new 'undef)
	     (list (if (<= (cadr saved) depth) (car saved) #t)
		   (cadr saved)
		   (if (null? (cddr saved))
		       (lambda args (get-successors board turn))
		       (cddr saved))))
	    ((eq? saved 'undef)
	     (list (if (<= (cadr new) depth) (car new) #t)
		   (cadr new)
		   (if (null? (cddr new))
		       (lambda args (get-successors board turn))
		       (cddr new))))
	    (else 
	     (let ((saved-depth (cadr saved)) (new-depth (cadr new)))
	       (if (< saved-depth new-depth)
		   (list (if (<= (cadr saved) depth) (car saved) #t)
			 (cadr saved)
			 (if (null? (cddr saved))
			     (lambda args (get-successors board turn))
			     (cddr saved)))
		   (list (if (<= (cadr new) depth) (car new) #t)
			 (cadr new)
			 (if (null? (cddr new))
			     (lambda args (get-successors board turn))
			     (cddr new)))))))))
  
  (define (get-old-value board turn)
    (let ((new (hash-table-get transitions (cons board turn)
                               (lambda () 'undef)))
          (saved (hash-table-get saved-transitions (cons board turn)
                                 (lambda () 'undef))))
      (cond ((and (eq? new 'undef) (eq? saved 'undef)) 0)
            ((eq? new 'undef) (caar saved))
            ((eq? saved 'undef) (caar new))
            (else 
             (let ((saved-depth (cadr saved)) (new-depth (cadr new)))
               (if (< saved-depth new-depth)
                   (caar saved)
                   (caar new)))))))
  

  

  ;;
  ;; aspirate :: Aspiration window
  ;;
  (define (aspirate board turn prev-move)
    (let* ((basevalue (get-old-value board turn))
	   (window-size 0.8)
	   (alpha (- basevalue window-size))
	   (beta (+ basevalue window-size)))
      (alfa-beta-pure-pvs board turn prev-move alpha beta)))
    
  ;;
  ;; alfa-beta-pure-pvs :: Aspiration window only initially, otherwise pvs.
  ;;
  (define (alfa-beta-pure-pvs board turn prev-move alpha beta)
    
    (set! transitions (make-hash-table 'equal))
    
    (let ((movelist ((caddr (get-saved-successors board turn 0)) 
		     'get-values))
	  (best-value -inf.0)
	  (best-move 'pass))
      
      
      ;;
      ;; First successor ::
      (let* ((first-move (car movelist))
	     (first-board (fork-move board first-move))
	     (a alpha)
	     (b beta)
	     (depth 1) (rd 1))
	(gui 'display-ai-action "** Using pure pvs")
	(set! best-value (min-value first-board (next turn) 
				    alpha beta 1 1 first-move))
	(set! best-move first-move)
	
	(set! a (max best-value alpha))
	
	(gui 'display-ai-action first-move best-value)
	;; rest of the moves
	(for-each
	 (lambda (move)
	   (let* ((newboard (fork-move board move))
		  (inc (if (and (>= (+ depth 1) max-depth)
				(not (quiescent-state? 
				      board turn prev-move
				      move)))
			   0 1))
		  (min-val 
		   (let ((try 
			  (min-value 
			   newboard (next turn) a (+ a 0.1) 
			   (+ depth inc) (+ rd 1) move)))
		     (if (and (> try (+ a 0.1)) (< try b))
			 ;;
			 ;; retry needed
			 (min-value newboard (next turn) try b 
				    (+ depth inc) (+ rd 1)
				    move)
			 try))))
	     (gui 'display-ai-action move " " min-val)
	     (if (> min-val best-value) 
		 (begin
		   (set! best-value min-val)
		   (set! best-move move)))
	     (set! a (max best-value a))))
	 (cdr movelist))    

	(if (or (< best-value alpha) (> best-value beta))
	    (alfa-beta-pure-pvs board turn prev-move -inf.0 +inf.0)
	    best-move))))
  




  ;; max-value :: Chooses the best move for max
  ;;
  ;; uses : * min-value to evaluate opponents response to move
  ;;        * fork-move to make an artificial board to make the move on
  ;;        * terminal-test to recognize the end of the game
  ;;        * utility to evaluate the value of the state for player max.
  ;;
  ;; AI-stuff: * transition-table for redundant states
  ;;           * Using evaluations of recent round when choosing the next
  ;;             node to expand
  ;;           * Quiescent state cutoff
  
  (define (max-value board turn alfa beta depth rd prev-move)
    (let ((node (make-priority-list >)))

      (define (each-successor slist v a b)

	(define recursive-call
	  (lambda ()
	    (let* ((newboard (fork-move board (car slist)))
		   (inc (if (and (>= (+ depth 1) max-depth)
				 (not (quiescent-state? 
				       board turn prev-move
				       (car slist))))
			    0 1))
		   (min-val (let ((try 
				   (min-value 
				    newboard (next turn) a b 
				    (+ depth inc) (+ rd 1) (car slist))))
			      (if (and (> try b) (< try beta))
				  ;;
				  ;; retry needed
				  
				  (min-value newboard (next turn) try beta 
					     (+ depth inc) (+ rd 1)
					     (car slist))
				  try))))


	      ;; Add move and its value to the node
	      (node 'put! min-val (car slist))
	      
	      ;; Alfa-beta pruning. If rest of the successors are  
	      ;; pruned, add them to the current tree.
	      (let ((u (max v min-val)))
		(if (>= u beta)                      
                    (let ((f (cond ((>= u beta) 1) ((<= u alfa) -1) (else 0))))
                      (node 'append! -inf.0 (cdr slist))
                      ;; Add new state to transition table.
                      (hash-table-remove! saved-transitions (cons board turn))
                      (hash-table-remove! transitions (cons board turn))
                      (hash-table-put! transitions (cons board turn)
                                       (cons (cons u f) (cons rd node)))
                      (list u))

		    (each-successor (cdr slist) u (max a u) 
				    (+ (max a u) 0.1)))))))
	
        (cond ((null? slist)
               (let ((f (cond ((>= v beta) 1) ((<= v alfa) -1) (else 0))))
                 (hash-table-remove! saved-transitions (cons board turn))
                 (hash-table-remove! transitions (cons board turn))
                 (hash-table-put! transitions (cons board turn)
                                  (cons (cons v f) (cons rd node)))
                 (list v)))
	      (else (recursive-call))))
      
      ;; If terminal test matches or search depth exceeds, get utility
      ;; value from hash or calculate it. Else calculate values for
      ;; each successor.
      ;;(gui 'display-ai-action "max-value " alfa " " beta)
    
      (if (or (>= depth max-depth)
              (>= rd max-rdepth)
	      (terminal-test board turn))
          (let ((data (hash-table-get 
                       transitions (cons board turn) 
                       (lambda () 'undef))))
            (if (eq? data 'undef)
                (let* ((util (utility board turn rd))
                       (f (cond ((>= util beta) 1) ((<= util alfa) -1) 
                                (else 0))))
                  (hash-table-put! transitions (cons board turn) 
                                   (cons (cons util f) (cons rd '())))
                  util)
                (if (= (cdar data) 0) 
                    (caar data)
                    (utility board turn rd))))
          
	  (let* ((tr-data (get-saved-successors board turn rd))
		 (tr-value (car tr-data))
		 (tr-depth (cadr tr-data))
		 (tr-node (caddr tr-data)))
	    (if (pair? tr-value)
                (let ((val (car tr-value)) (fail (cdr tr-value)))
                  (cond ((= fail 0)
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        ((and (< fail 0)
                              (<= val alfa)) 
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        ((and (> fail 0)
                              (>= val beta))
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        (else
                         (if (< fail 0)
                             (set! beta (min beta val))
                             (set! alfa (max alfa val)))
                         (car (each-successor (tr-node 'get-values) 
                                              -inf.0 alfa beta)))))
		(car (each-successor (tr-node 'get-values) 
				     -inf.0 alfa beta)))))))

  ;;
  ;; min-value :: Chooses the best move for min
  ;;
  ;; uses : * max-value to evaluate opponents response to move
  ;;        * fork-move to make an artificial board to make the move on
  ;;        * terminal-test to recognize the end of the game
  ;;        * utility to evaluate the value of the state for player max
  ;;          (min tries to make it as small as possible)
  ;;
  ;; AI-stuff: * transition-table for redundant states
  ;;           * Using evaluations of recent round when choosing the next
  ;;             node to expand (tree)
  ;;           * Quiescent state cutoff 

  (define (min-value board turn alfa beta depth rd prev-move)
    (let ((node (make-priority-list <)))

      (define (each-successor slist v a b)

	(define recursive-call
	  (lambda ()
	    (let* ((newboard (fork-move board (car slist)))
		   (inc (if (and (>= (+ depth 1) max-depth)
				 (not (quiescent-state? 
				       board turn prev-move 
				       (car slist))))
			    0 1))
		   (max-val     
		    (let ((try (max-value newboard (next turn) 
					  a b 
					  (+ depth inc) (+ rd 1) 
					  (car slist))))
		      (if (and (> try alfa) (< try a))
			  ;;
			  ;; retry needed
			  (max-value newboard (next turn)
				     alfa try (+ depth inc) 
				     (+ rd 1) 
				     (car slist))
			  try))))
	      
	      ;; Add move and its value to the node
	      (node 'put! max-val (car slist))
	      
	      ;; Alfa-beta pruning. If rest of the successors are 
	      ;; pruned, add them to the current tree.
	      (let ((u (min v max-val)))
		(if (<= u alfa)
                    (let ((f (cond ((>= u beta) 1) ((<= u alfa) -1) (else 0))))
                      (node 'append! +inf.0 (cdr slist))
                      ;; Add new state to transition table.
                      (hash-table-remove! saved-transitions (cons board turn))
                      (hash-table-remove! transitions (cons board turn))
                      (hash-table-put! transitions (cons board turn)
                                       (cons (cons u f) (cons rd node)))
                      (list u))
		    
		    (each-successor (cdr slist) u 
				    (- (min b u) 0.1) 
				    (min b u)))))))
  
        (cond ((null? slist) 
               (let ((f (cond ((>= v beta) 1) ((<= v alfa) -1) (else 0))))
                 (hash-table-remove! saved-transitions (cons board turn))
                 (hash-table-remove! transitions (cons board turn))
                 (hash-table-put! transitions (cons board turn)
                                  (cons (cons v f) (cons rd node)))
                 (list v)))
	      (else (recursive-call))))
      
      ;; If terminal test matches or search depth exceeds, get utility
      ;; value from hash or calculate it. Else calculate values for
      ;; each successor.
      ;;(gui 'display-ai-action "min-value " alfa " " beta)

      (if (or (>= depth max-depth)
              (>= rd max-rdepth)
	      (terminal-test board turn))
          (let ((data (hash-table-get 
                       transitions (cons board turn) 
                       (lambda () 'undef))))
            (if (eq? data 'undef)
                (let* ((util (utility board (next turn) rd))
                       (f (cond ((>= util beta) 1) ((<= util alfa) -1) 
                                (else 0))))
                  (hash-table-put! transitions (cons board turn) 
                                   (cons (cons util f) (cons rd '())))
                  util)
                (if (= (cdar data) 0) 
                    (caar data)
                    (utility board (next turn) rd))))

	  (let* ((tr-data (get-saved-successors board turn rd))
		 (tr-value (car tr-data))
		 (tr-depth (cadr tr-data))
		 (tr-node (caddr tr-data)))
	    (if (pair? tr-value)
                (let ((val (car tr-value)) (fail (cdr tr-value)))
                  (cond ((= fail 0)
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        ((and (< fail 0)
                              (<= val alfa)) 
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        ((and (> fail 0)
                              (>= val beta))
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        (else
                         (if (< fail 0)
                             (set! beta (min beta val))
                             (set! alfa (max alfa val)))
                         (car (each-successor (tr-node 'get-values) 
                                              +inf.0 alfa beta)))))
		(car (each-successor (tr-node 'get-values)
				     +inf.0 alfa beta)))))))

  ;; 
  ;; returns ai's next move 
  ;;

  (define (get-move board turn prev-move allowed-moves)
    ;; (gui 'display-ai-action "prev-move: " prev-move)

    ;; Alfa-beta search won't be started at all if only one move is
    ;; available
    (if (= (hash-size allowed-moves) 1)
	(let ((moves (get-successors board turn)))
	  (car moves))
        (let ((move (aspirate board turn prev-move)))
          (set! saved-transitions transitions)
          move)))


  ;; Dispatch prosedure for message passing
  (define (dispatch m . args)
    (cond ((eq? m 'get-move) (apply get-move args))
	  ((eq? m 'get-player-type) 'ai)
	  ((eq? m 'get-successors) (apply get-successors args))
	  (else (error "Invalid operation for ai-player"))))


  ;; Set optional arguments if given
  (if (not (null? args)) (set! max-depth (car args)))
  (if (> (length args) 1) (set! max-rdepth (cadr args)))
  dispatch)
