;;;
;;; Hierarkia - Scheme Implementation
;;; ai_mtd.scm: AI object implementation
;;;
;;; Copyright 2003 Sami Virpioja, Ilari Lhteenmki
;;;
;;; ---
;;;
;;; This file is part of Hierarkia - Scheme Implementation.
;;;
;;; Hierarkia - Scheme Implementation is free software; you can redistribute 
;;; it and/or modify it under the terms of the GNU General Public License as 
;;; published by the Free Software Foundation; either version 2 of the 
;;; License, or (at your option) any later version.
;;;
;;; Hierarkia - Scheme Implementation  is distributed in the hope that it will 
;;; be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
;;; Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License along 
;;; with Hierarkia - Scheme Implementation; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;
;;; ---
;;;
;;; See README for more details.
;;;

;;
;; A message dispatching style object representing an AI player
;;

(define (make-mtd-ai gui . args)

  (define max-depth 3)
  (define max-rdepth 3)

  ;; transitions :: hash-table for saving evaluation of redundant states
  ;;
  ;; Pair (board, turn) (identifies the state of game) is used as a key and 
  ;; evaluation of the state (calculated utility) and corresponding node in 
  ;; the move tree are consed together to make the data

  ;; ((value . fail-ind) depth . node)
  ;; caar - value
  ;; cdar - 0: between bounds, 1: fail high, -1: fail low
  ;; cadr - depth
  ;; cddr - node

  (define transitions (make-hash-table 'equal))
  
  (define saved-transitions (make-hash-table 'equal))

  ;; get-saved-successors :: Searches both transition tables and
  ;; returns a list of tree elements. If first value is false,
  ;; state wasn't found in tables. If it is true, state was found
  ;; but it was calculated deeper in the search tree than depth.
  ;; Otherwise it is the value of the saved node. Second return
  ;; value is depth of the node in the table or false. Third
  ;; value is node's successors, either from the table or from the
  ;; current state using get-successors -procedure.
  
  (define (get-saved-successors board turn depth)
    (let ((new (hash-table-get transitions (cons board turn)
			       (lambda () 'undef)))
	  (saved (hash-table-get saved-transitions (cons board turn)
				 (lambda () 'undef))))
      (cond ((and (eq? new 'undef) (eq? saved 'undef))
	     (list #f #f (lambda args (get-successors board turn))))
	    ((eq? new 'undef)
	     (list (if (<= (cadr saved) depth) (car saved) #t)
		   (cadr saved)
		   (if (null? (cddr saved))
		       (lambda args (get-successors board turn))
		       (cddr saved))))
	    ((eq? saved 'undef)
	     (list (if (<= (cadr new) depth) (car new) #t)
		   (cadr new)
		   (if (null? (cddr new))
		       (lambda args (get-successors board turn))
		       (cddr new))))
	    (else 
	     (let ((saved-depth (cadr saved)) (new-depth (cadr new)))
	       (if (< saved-depth new-depth)
		   (list (if (<= (cadr saved) depth) (car saved) #t)
			 (cadr saved)
			 (if (null? (cddr saved))
			     (lambda args (get-successors board turn))
			     (cddr saved)))
		   (list (if (<= (cadr new) depth) (car new) #t)
			 (cadr new)
			 (if (null? (cddr new))
			     (lambda args (get-successors board turn))
			     (cddr new)))))))))
  
  (define (get-old-value board turn)
    (let ((new (hash-table-get transitions (cons board turn)
                               (lambda () 'undef)))
          (saved (hash-table-get saved-transitions (cons board turn)
                                 (lambda () 'undef))))
      (cond ((and (eq? new 'undef) (eq? saved 'undef)) 0)
            ((eq? new 'undef) (caar saved))
            ((eq? saved 'undef) (caar new))
            (else 
             (let ((saved-depth (cadr saved)) (new-depth (cadr new)))
               (if (< saved-depth new-depth)
                   (caar saved)
                   (caar new)))))))
  
  ;; alfa-beta-search :: Base function for ai, makes the first move
  ;;
  ;; uses: * min-value to evaluate opponents response to possible moves.
  ;;       * fork-move to make an artificial board to plan future moves on.
  ;;       * get-successors to get possible moves
  ;;       * search-tree to get the best moves according to previous 
  ;;         evaluation first

  (define (alfa-beta-search board turn) 
    (let ((best-value -inf.0) (best-move 'pass))

      ;; Clear transition table
      (set! transitions (make-hash-table 'equal))

      ;;     (display-line "Plotting tree on start of alfa-beta")
      ;;     (plot-tree search-tree 0) 

      (for-each 
       (lambda (move)
	 (let* ((newboard (fork-move board move))
		(oldval (get-old-value newboard (next turn))))

           ;; In MTD-loop, value got from previous round will be used to
           ;; make an estimate of the future value of 
           ;; min-value. MTD uses zero-size window and 
           ;; iterates closer to real value.
           (define (MTD-loop f-value upperbound lowerbound n)
	     (gui 'display-ai-action 
		  "** round " n ", bounds "
		  (list (round-decimal upperbound 3)
			(round-decimal lowerbound 3))
                  ", f-value " (round-decimal f-value 3))
             (cond ((>= lowerbound upperbound) 
		    f-value)
		   ((> n 20) 
		    (gui 'display-ai-action "backing to alfa-beta")
                    (min-value newboard (next turn)
                               best-value +inf.0 1 1 move))
                   (else 
                    (let* ((beta (if (= f-value lowerbound)
                                     (+ f-value 0.1)
                                     f-value))
                           (g-value (min-value newboard (next turn) 
                                               (- beta 0.1) beta	
                                               1 1 move)))
                      (MTD-loop g-value 
                                (if (< g-value beta) 
                                    g-value
                                    upperbound)
                                (if (< g-value beta)
                                    lowerbound
                                    g-value)
                                (+ n 1))))))
           
	   (gui 'display-ai-action "starting MTD, initial guess " 
                (round-decimal oldval 3))

           (let ((value (MTD-loop oldval +inf.0 -inf.0 0)))

	     (gui 'display-ai-action move " " 
		  (round-decimal value 3) " " 
                  (hash-size transitions))
	     
	     (if (> value best-value) 
		 (begin (set! best-value value)
			(set! best-move move))))))
       
       ;; Get successors either from the saved tree or directly from board
       ((caddr (get-saved-successors board turn 0)) 'get-values))
      
      best-move))

  ;; max-value :: Chooses the best move for max
  ;;
  ;; uses : * min-value to evaluate opponents response to move
  ;;        * fork-move to make an artificial board to make the move on
  ;;        * terminal-test to recognize the end of the game
  ;;        * utility to evaluate the value of the state for player max.
  ;;
  ;; AI-stuff: * transition-table for redundant states
  ;;           * Using evaluations of recent round when choosing the next
  ;;             node to expand
  ;;           * Quiescent state cutoff
  
  (define (max-value board turn alfa beta depth rd prev-move)
    (let ((node (make-priority-list >)))

      (define (each-successor slist v a b)

	(define recursive-call
	  (lambda ()
	    (let* ((newboard (fork-move board (car slist)))
		   (inc (if (and (>= (+ depth 1) max-depth)
				 (not (quiescent-state? 
				       board turn prev-move
				       (car slist))))
			    0 1))
		   (min-val (min-value 
			     newboard (next turn) a b 
			     (+ depth inc) (+ rd 1) (car slist))))
	      
	      ;; Add move and its value to the node
	      (node 'put! min-val (car slist))
	      
	      ;; Alfa-beta pruning. If rest of the successors are  
	      ;; pruned, add them to the current tree.
	      (let ((u (max v min-val)))
		(if (>= u b)                      
                    (let ((f (cond ((>= u beta) 1) ((<= u alfa) -1) (else 0))))
                      (node 'append! -inf.0 (cdr slist))
                      ;; Add new state to transition table.
                      (hash-table-remove! saved-transitions (cons board turn))
                      (hash-table-remove! transitions (cons board turn))
                      (hash-table-put! transitions (cons board turn)
                                       (cons (cons u f) (cons rd node)))
                      (list u))

		    (each-successor (cdr slist) u (max a u) b))))))
	
        (cond ((null? slist)
               (let ((f (cond ((>= v beta) 1) ((<= v alfa) -1) (else 0))))
                 (hash-table-remove! saved-transitions (cons board turn))
                 (hash-table-remove! transitions (cons board turn))
                 (hash-table-put! transitions (cons board turn)
                                  (cons (cons v f) (cons rd node)))
                 (list v)))
	      (else (recursive-call))))
      
      ;; If terminal test matches or search depth exceeds, get utility
      ;; value from hash or calculate it. Else calculate values for
      ;; each successor.
      ;;(gui 'display-ai-action "max-value " alfa " " beta)
    
      (if (or (>= depth max-depth)
              (>= rd max-rdepth)
	      (terminal-test board turn))
          (let ((data (hash-table-get 
                       transitions (cons board turn) 
                       (lambda () 'undef))))
            (if (eq? data 'undef)
                (let* ((util (utility board turn rd))
                       (f (cond ((>= util beta) 1) ((<= util alfa) -1) 
                                (else 0))))
                  (hash-table-put! transitions (cons board turn) 
                                   (cons (cons util f) (cons rd '())))
                  util)
                (if (= (cdar data) 0) 
                    (caar data)
                    (utility board turn rd))))
          
	  (let* ((tr-data (get-saved-successors board turn rd))
		 (tr-value (car tr-data))
		 (tr-depth (cadr tr-data))
		 (tr-node (caddr tr-data)))
	    (if (pair? tr-value)
                (let ((val (car tr-value)) (fail (cdr tr-value)))
                  (cond ((= fail 0)
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        ((and (< fail 0)
                              (<= val alfa)) 
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        ((and (> fail 0)
                              (>= val beta))
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        (else
                         (if (< fail 0)
                             (set! beta (min beta val))
                             (set! alfa (max alfa val)))
                         (car (each-successor (tr-node 'get-values) 
                                              -inf.0 alfa beta)))))
		(car (each-successor (tr-node 'get-values) 
				     -inf.0 alfa beta)))))))

  ;;
  ;; min-value :: Chooses the best move for min
  ;;
  ;; uses : * max-value to evaluate opponents response to move
  ;;        * fork-move to make an artificial board to make the move on
  ;;        * terminal-test to recognize the end of the game
  ;;        * utility to evaluate the value of the state for player max
  ;;          (min tries to make it as small as possible)
  ;;
  ;; AI-stuff: * transition-table for redundant states
  ;;           * Using evaluations of recent round when choosing the next
  ;;             node to expand (tree)
  ;;           * Quiescent state cutoff 

  (define (min-value board turn alfa beta depth rd prev-move)
    (let ((node (make-priority-list <)))

      (define (each-successor slist v a b)

	(define recursive-call
	  (lambda ()
	    (let* ((newboard (fork-move board (car slist)))
		   (inc (if (and (>= (+ depth 1) max-depth)
				 (not (quiescent-state? 
				       board turn prev-move 
				       (car slist))))
			    0 1))
		   (max-val (max-value 
			     newboard (next turn) a b 
			     (+ depth inc) (+ rd 1) (car slist))))
	      
	      ;; Add move and its value to the node
	      (node 'put! max-val (car slist))
	      
	      ;; Alfa-beta pruning. If rest of the successors are 
	      ;; pruned, add them to the current tree.
	      (let ((u (min v max-val)))
		(if (<= u a)
                    (let ((f (cond ((>= u beta) 1) ((<= u alfa) -1) (else 0))))
                      (node 'append! +inf.0 (cdr slist))
                      ;; Add new state to transition table.
                      (hash-table-remove! saved-transitions (cons board turn))
                      (hash-table-remove! transitions (cons board turn))
                      (hash-table-put! transitions (cons board turn)
                                       (cons (cons u f) (cons rd node)))
                      (list u))
		    
		    (each-successor (cdr slist) u a (min b u)))))))
  
        (cond ((null? slist) 
               (let ((f (cond ((>= v beta) 1) ((<= v alfa) -1) (else 0))))
                 (hash-table-remove! saved-transitions (cons board turn))
                 (hash-table-remove! transitions (cons board turn))
                 (hash-table-put! transitions (cons board turn)
                                  (cons (cons v f) (cons rd node)))
                 (list v)))
	      (else (recursive-call))))
      
      ;; If terminal test matches or search depth exceeds, get utility
      ;; value from hash or calculate it. Else calculate values for
      ;; each successor.
      ;;(gui 'display-ai-action "min-value " alfa " " beta)

      (if (or (>= depth max-depth)
              (>= rd max-rdepth)
	      (terminal-test board turn))
          (let ((data (hash-table-get 
                       transitions (cons board turn) 
                       (lambda () 'undef))))
            (if (eq? data 'undef)
                (let* ((util (utility board (next turn) rd))
                       (f (cond ((>= util beta) 1) ((<= util alfa) -1) 
                                (else 0))))
                  (hash-table-put! transitions (cons board turn) 
                                   (cons (cons util f) (cons rd '())))
                  util)
                (if (= (cdar data) 0) 
                    (caar data)
                    (utility board (next turn) rd))))

	  (let* ((tr-data (get-saved-successors board turn rd))
		 (tr-value (car tr-data))
		 (tr-depth (cadr tr-data))
		 (tr-node (caddr tr-data)))
	    (if (pair? tr-value)
                (let ((val (car tr-value)) (fail (cdr tr-value)))
                  (cond ((= fail 0)
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        ((and (< fail 0)
                              (<= val alfa)) 
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        ((and (> fail 0)
                              (>= val beta))
                         (hash-table-remove! saved-transitions 
                                             (cons board turn))
                         (hash-table-remove! transitions (cons board turn))
                         (hash-table-put! 
                          transitions (cons board turn)
                          (cons tr-value (cons tr-depth tr-node)))
                         val)
                        (else
                         (if (< fail 0)
                             (set! beta (min beta val))
                             (set! alfa (max alfa val)))
                         (car (each-successor (tr-node 'get-values) 
                                              +inf.0 alfa beta)))))
		(car (each-successor (tr-node 'get-values)
				     +inf.0 alfa beta)))))))

  ;; 
  ;; returns ai's next move 
  ;;

  (define (get-move board turn prev-move allowed-moves)
    ;; (gui 'display-ai-action "prev-move: " prev-move)

    ;; Alfa-beta search won't be started at all if only one move is
    ;; available
    (if (= (hash-size allowed-moves) 1)
	(let ((moves (get-successors board turn)))
	  (car moves))
        (let ((move (alfa-beta-search board turn)))
          (set! saved-transitions transitions)
          move)))


  ;; Dispatch prosedure for message passing
  (define (dispatch m . args)
    (cond ((eq? m 'get-move) (apply get-move args))
	  ((eq? m 'get-player-type) 'ai)
	  ((eq? m 'get-successors) (apply get-successors args))
	  (else (error "Invalid operation for ai-player"))))


  ;; Set optional arguments if given
  (if (not (null? args)) (set! max-depth (car args)))
  (if (> (length args) 1) (set! max-rdepth (cadr args)))
  dispatch)
