;;;
;;; Hierarkia - Scheme Implementation
;;; ngui.scm: Non-graphical user interface object
;;;
;;; Copyright 2003 Sami Virpioja, Ilari Lhteenmki
;;;
;;; ---
;;;
;;; This file is part of Hierarkia - Scheme Implementation.
;;;
;;; Hierarkia - Scheme Implementation is free software; you can redistribute 
;;; it and/or modify it under the terms of the GNU General Public License as 
;;; published by the Free Software Foundation; either version 2 of the 
;;; License, or (at your option) any later version.
;;;
;;; Hierarkia - Scheme Implementation  is distributed in the hope that it will 
;;; be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
;;; Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License along 
;;; with Hierarkia - Scheme Implementation; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;
;;; ---
;;;
;;; See README for more details.
;;;
;;; This file includes make-ngui procedure, which makes an object
;;; for non-graphical user inferface.
;;;

(define (make-ngui . args)
  (define debug #t)

  (define (display-move move . args)
    (if (eq? move 'pass) 
        (display-line 'pass)
        (display-line (move-unit move) " " 
                      (move-type move) "s " 
                      (move-dest move))))

  (define (display-board board turn a-moves btime rtime . args)
    (print-board board)
    (display-line "Used times: " (list 'blue (round (* 0.001 btime))) 
                  " " (list 'red (round (* 0.001 rtime)))))

  (define (get-game-action . args)
    (display "Game command: ")
    (let ((input (read)))
      (cond ((or (eq? input 'quit) (eq? input 'q)) 'quit)
            ((and (list? input) (eq? (car input) 'play)
                  (not (null? (cdr input))) (number? (cadr input)))
             input)
            ((and (list? input) (eq? (car input) 'human)
                  (not (null? (cdr input))) (number? (cadr input)))
             input)
            ((and (list? input) (eq? (car input) 'ai)
                  (not (null? (cdr input))) (number? (cadr input)))
             input)
            (else (display-line "Unknown command, try quit or (play <rounds>)")
                  (get-game-action)))))

  (define (get-player-action board turn allowed-moves)
    (define (read-unit) (display "Unit to move: ") (read))
    (define (read-target) (display "Choose targer square: ") (read))    
    (let unit-loop ((unit-input (read-unit)))
      (let ((move-list (hash-table-get 
                        allowed-moves unit-input
                        (lambda () 'none))))
        (cond ((eq? move-list 'none)
               (cond ((eq? unit-input 'pass) 'pass)
                     ((eq? unit-input 'quit) 'quit)
                     ((and (list unit-input)
                           (or (eq? (car unit-input) 'ai)
                               (eq? (car unit-input) 'human))
                           (not (null? (cdr unit-input))) 
                           (number? (cadr unit-input)))
                      unit-input)
                     (else (display-line "Unknown unit given.")
                           (unit-loop (read-unit)))))
              ((null? move-list)
               (display-line "That unit cannot move.")
               (unit-loop (read-unit)))
              (else (display-line "Allowed moves are:")
                    (for-each (lambda (m) (display-line (car m) " " (cdr m))) 
                              move-list)
                    (let target-loop ((target-input (read-target)))
                      (if (not (list? target-input))
                          (begin (display-line 
                                  "Target and unit formats are (x y)")
                                 (unit-loop (read-unit)))
                          (let find-move ((ml move-list))
                            (cond ((null? ml)
                                   (display-line 
                                    "Illegal target for unit.")
                                   (target-loop (read-target)))
                                  ((equal? target-input (cdr (car ml)))
                                   (list unit-input (car ml)))
                                  (else (find-move (cdr ml))))))))))))

  (define (display-ai-action . args)
    (if debug
        (begin
          (for-each (lambda (a) (display a)) args)
          (newline))))

  (define (dispatch m . args)
    (cond ((eq? m 'display-move) (apply display-move args))
          ((eq? m 'display-board) (apply display-board args))
          ((eq? m 'get-game-action) (apply get-game-action args))
          ((eq? m 'get-player-action) (apply get-player-action args))
          ((eq? m 'display-ai-action) (apply display-ai-action args))
          ((eq? m 'event-type) 'sequential)
	  ((eq? m 'quit) (apply display-line args))
	  ((eq? m 'init) (void))
          (else (error "MAKE-NGUI: unknown message" m))))
  
  (if (and (not (null? args)) (eq? (car args) 'nodebug))
      (set! debug #f))
  
  dispatch)
