;;;
;;; Hierarkia - Scheme Implementation
;;; mred-gui.scm: Graphical user interface object
;;;
;;; Copyright 2003 Sami Virpioja, Ilari Lhteenmki
;;;
;;; ---
;;;
;;; This file is part of Hierarkia - Scheme Implementation.
;;;
;;; Hierarkia - Scheme Implementation is free software; you can redistribute 
;;; it and/or modify it under the terms of the GNU General Public License as 
;;; published by the Free Software Foundation; either version 2 of the 
;;; License, or (at your option) any later version.
;;;
;;; Hierarkia - Scheme Implementation  is distributed in the hope that it will 
;;; be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
;;; Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License along 
;;; with Hierarkia - Scheme Implementation; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;
;;; ---
;;;
;;; See README for more details.
;;;
;;; This file includes make-gui procedure, which makes an object
;;; for graphical user inferface. MrEd GUI toolbox is required,
;;; as well as MzScheme string library "string.ss".
;;;

(require (lib "string.ss"))

(define pi (acos -1))

(define (get-move moves pos)
  (cond ((null? moves) '())
        ((equal? (cdr (car moves)) pos) 
         (car moves))
        (else (get-move (cdr moves) pos))))

(define (make-gui board-size max-rank board . args)
  (let* ((square-size 50)
         (circle-size (inexact->exact (round (* square-size 7/8))))
         (square-side (round (/ (* (sqrt 2) square-size) 2)))
         (unit-size (inexact->exact (round (/ circle-size (sqrt 2) 1.3))))
         (board-xsize (inexact->exact (* 2 board-size square-side)))
         (board-ysize (inexact->exact (* 2 board-size square-side)))
         (textarea-width 200)
         (menubar-height 20)
         (textarea-height board-ysize)
         (window-xsize (+ board-xsize textarea-width))
         (window-ysize (+ board-ysize menubar-height))

	 (popup-ai-menu 'undef)

         (debug #t)
         (player-action-state #f)
         (game-action-state #f)
	 (direct-quit #f)

         (selected-square '())
         (action '())
         (turn 'none)
         (allowed-moves '())
         (player1 'none)
         (player2 'none)

         (square-center 
          (lambda (sx sy)
            (list (+ (* board-size square-side) (* square-side (- sx sy)))
                  (+ square-side (* square-side (+ sx sy))))))
         
         (coord->square 
          (lambda (x y)
            (list 
	     (inexact->exact
	      (round (* 0.5 (- (/ (+ y x) square-side) board-size 1))))
	     (inexact->exact 
	      (round (* 0.5 (+ (/ (- y x) square-side) board-size -1)))))))
         
         (board-bitmap (instantiate 
                           bitmap% (board-xsize board-ysize)))
         (board-dc (instantiate bitmap-dc% (board-bitmap)))
    
         (frame (let ((f (instantiate frame% ("Hierarkia") 
                           (alignment '(center top))
                           (width window-xsize) (height window-ysize))))
                  (send f set-control-font 
                        (make-object font% 10 'default 'normal 'normal #f))
                  f))
         (menubar-panel (instantiate vertical-panel% (frame)))
    
         (menu-bar (instantiate menu-bar% (frame)))
         (main-menu (instantiate menu% () 
                      (label "&Main") (parent menu-bar)))
         (players-menu (instantiate menu% () 
                         (label "&Players") (parent menu-bar)))
    
         (board-panel (instantiate horizontal-panel% (menubar-panel)
                        (alignment '(center center))))
         (msg (instantiate message% ()
                (label "")
                (parent menubar-panel)
                (stretchable-width #t)))
    
         (draw-update (lambda args 'done))
         (board-canvas%
          (class canvas% ; The base class is canvas%
            ;; Declare overrides:
            (override on-event)
            ;; Define overriding method to handle mouse events
            (define on-event 
              (lambda (event) 
                (let* ((sq (coord->square (send event get-x) 
                                          (send event get-y)))
                       (m (string-append 
                           (number->string (car sq)) " x "
                           (number->string (cadr sq)))))
                  (let ((t (send event get-event-type)))
                    (cond ((and (eq? t 'left-down) 
                                (< -1 (car sq) board-size)
                                (< -1 (cadr sq) board-size)
                                game-action-state)
                           (let ((selected 
                                  (if (null? selected-square) '()
                                      (board-ref board 
                                                 (car selected-square) 
                                                 (cadr selected-square))))
                                 (hit (board-ref board (car sq) (cadr sq))))
                             (cond ((and (null? selected)
                                         (not (null? hit))
                                         (eq? (unit-side hit) turn))
                                    (set! selected-square sq))
                                   ((and (not (null? selected))
                                         (equal? selected-square hit))
                                    (set! selected-square '()))
                                   ((and (not (null? hit))
                                         (not (null? selected))
                                         (eq? (unit-side hit) turn)
                                         (eq? (unit-side selected) turn))
                                    (set! selected-square sq))
                                   ((and (not (null? selected))
                                         (not (eq? (unit-side selected) turn)))
                                    (if (and (not (null? hit))
                                             (eq? (unit-side hit) turn))
                                        (set! selected-square sq)
                                        (set! selected-square '())))
                                   ((and (not (null? selected))
                                         (eq? (unit-side selected) turn))
                                    (let* ((moves (hash-table-get 
                                                   allowed-moves 
						   selected-square
                                                   (lambda () '())))
                                           (m (get-move moves sq)))
                                      (if (not (null? m))
                                          (begin 
                                            (set! action 
                                                  (list selected-square m))
                                            (set! selected-square '()))))))
                             (draw-update board))))
                    
                    (send msg set-label (string-append m))))))
            ;; Call the superclass initialization (and pass on all init args)
            (super-instantiate ())))

         (canvas (instantiate board-canvas% (board-panel)
                   (min-width board-xsize) (min-height board-ysize)
                   (paint-callback 
                    (lambda (canvas dc) 
                      (send dc draw-bitmap board-bitmap 0 0)))))
         (textarea (instantiate text-field% 
                     () (label #f) (parent board-panel) 
                     (callback (lambda (field event) 
                                 (void)))
                     (min-width textarea-width)
                     (min-height textarea-height)))
         
         (main-start (instantiate menu-item% ()
                      (label "Restart game") (parent main-menu)
                      (callback 
                       (lambda (button event)
                         (send (send textarea get-editor) lock #f)
                         (send (send textarea get-editor) insert 
                               "Game restarted.\n")
                         (send (send textarea get-editor) lock #t)
                         (set! selected-square '())
                         (set! action 'restart)))))
         (main-abort (instantiate menu-item% ()
                      (label "Abort game") (parent main-menu)
                      (callback 
		       (lambda (button event)
			 (send (send textarea get-editor) lock #f)
                         (send (send textarea get-editor) insert 
                               "Game aborted.\n")
                         (send (send textarea get-editor) lock #t)
                         (set! selected-square '())
			 (set! action 'quit)))))
	 (main-sep (instantiate separator-menu-item% () (parent main-menu)))
	 (main-quit (instantiate menu-item% ()
				  (label "Quit") (parent main-menu)
				  (callback 
				   (lambda (button event)
				     (set! direct-quit #t)
				     (set! action 'quit)
				     (send frame show #f)))))         
	 (players-1-human (instantiate checkable-menu-item% ()
			    (label "P1 - Human") (parent players-menu)
                            (callback (lambda (button event)
                                        (send button check #t)
                                        (set! action '(human 1))))))
         (players-1-ai (instantiate checkable-menu-item% ()
                         (label "P1 - AI") (parent players-menu)
                         (callback (lambda (button event)
                                     (send button check #t)
                                     (set! action '(ai 1))))))
	 (players-sep1 (instantiate separator-menu-item% () 
				    (parent players-menu)))
         (players-2-human (instantiate checkable-menu-item% ()
                            (label "P2 - Human") (parent players-menu)
                            (callback (lambda (button event)
                                        (send button check #t)
                                        (set! action '(human 2))))))
         (players-2-ai (instantiate checkable-menu-item% ()
                         (label "P2 - AI") (parent players-menu)
                         (callback (lambda (button event)
                                     (send button check #t)
                                     (set! action '(ai 2))))))
	 (players-sep2 (instantiate separator-menu-item% () 
				    (parent players-menu)))
	 (players-ai-set (instantiate 
			  menu-item% ()
			  (label "Config AI") (parent players-menu)
			  (callback 
			   (lambda (button event) (popup-ai-menu))))))


    (define ai-1-type 'none)
    (define ai-1-depth 1)
    (define ai-1-maxdepth 1)
    (define ai-1-mod #f)
    (define ai-2-type 'none)
    (define ai-2-depth 1)
    (define ai-2-maxdepth 1)
    (define ai-2-mod #f)
    (define blue-time 0)
    (define red-time 0)
    
    (define no-pen (instantiate pen% ("BLACK" 1 'transparent)))
    (define no-brush (instantiate brush% ("BLACK" 'transparent))) 
    (define blue-brush (instantiate brush% ("BLUE" 'solid))) 
    (define lightblue-brush (instantiate brush% 
                              ((make-object color% 160 160 255) 'solid))) 
    (define yellow-brush (instantiate brush% ("YELLOW" 'solid))) 
    (define red-brush (instantiate brush% ("RED" 'solid))) 
    (define lightred-brush (instantiate brush% 
                             ((make-object color% 255 160 160) 'solid))) 
    (define white-brush (instantiate brush% ("WHITE" 'solid))) 
    (define red-pen (instantiate pen% ("RED" 1 'solid))) 
    (define red-thick-pen (instantiate pen% ("RED" 3 'solid))) 
    (define black-pen (instantiate pen% ("BLACK" 1 'solid))) 
    (define black-2-pen (instantiate pen% ("BLACK" 2 'solid))) 
    (define black-thick-pen (instantiate pen% ("BLACK" 3 'solid))) 
    (define unit-font (make-object font% 12 'default 'normal 'bold #f))
    (define default-font (make-object font% 10 'default 'normal 'normal #f))

    (define (draw-gameboard dc)
      (define (draw-outlines)
        (send dc set-brush no-brush)
        (send dc set-pen black-pen)
        (send dc draw-polygon
              (list (make-object point% 0 (/ board-ysize 2))
                    (make-object point% (/ board-xsize 2) 0)
                    (make-object point% board-xsize (/ board-ysize 2))
                    (make-object point% (/ board-xsize 2) board-ysize))
              0 0 'odd-even))
      (define (draw-grid)
        (send dc set-brush no-brush)
        (send dc set-pen black-pen)
        (let xloop ((x (- board-size 1)))
          (if (< x 0) 'done
              (let ((p1 (square-center x 0)) 
                    (p2 (square-center x (- board-size 1))))
                (send dc draw-line (car p1) (cadr p1) (car p2) (cadr p2))
                (xloop (- x 1)))))
        (let yloop ((y (- board-size 1)))
          (if (< y 0) 'done
              (let ((p1 (square-center 0 y)) 
                    (p2 (square-center (- board-size 1) y)))
                (send dc draw-line (car p1) (cadr p1) (car p2) (cadr p2))
                (yloop (- y 1)))))                
        'done)
      (define (draw-squares)
        (let loop ((x 0) (y 0))
          (cond ((>= y board-size) 'done)
                ((>= x board-size) (loop 0 (+ y 1)))
                (else
                 (cond ((and (not (null? selected-square))
                             (= x (car selected-square))
                             (= y (cadr selected-square)))
                        (send dc set-brush yellow-brush))
                       ((and (not (null? selected-square))
                             (not (null? allowed-moves))
                             (let ((moves (hash-table-get 
                                           allowed-moves selected-square
                                           (lambda () '()))))
                               (member (list x y) (map (lambda (x) (cdr x))
                                                       moves))))
                        (send dc set-brush yellow-brush))
                       ((inside-blue? x y) (send dc set-brush lightblue-brush))
                       ((inside-red? x y) (send dc set-brush lightred-brush))
                       (else (send dc set-brush white-brush)))
                 (send dc set-pen black-pen)
                 (let ((c (square-center x y)))
                   (send dc draw-ellipse 
                         (- (car c) (/ circle-size 2))
                         (- (cadr c) (/ circle-size 2))
                         circle-size circle-size)
                   (loop (+ x 1) y)))))
        'done)
      (draw-outlines)
      (draw-grid)
      (draw-squares)
      (send dc set-font unit-font)
      (send dc set-text-foreground (make-object color% 0 0 0))
      (let ((x circle-size) (y circle-size))
        (send dc draw-text (number->string (round (* 0.001 blue-time))) x y))
      (let ((y (- board-ysize circle-size)) (x circle-size))
        (send dc draw-text (number->string (round (* 0.001 red-time))) x y))
      (send dc set-pen black-pen)
      (send dc set-brush (if (eq? turn 'r) lightred-brush lightblue-brush))
      (let ((x (- board-xsize circle-size)) (y circle-size))
        (send dc draw-ellipse 
              (- x (/ circle-size 2))
              (- y (/ circle-size 2))
              circle-size circle-size)))
    
    (define (draw-units dc)
      (define (draw-unit x y unit)
        (let ((pos (square-center x y))
              (side (unit-side unit)) (rank (unit-rank unit)))
          (send dc set-pen black-pen)
          (send dc set-brush (if (eq? side 'r) red-brush blue-brush))
          (send dc draw-rectangle 
                (- (car pos) (* unit-size 0.5)) 
		(- (cadr pos) (* unit-size 0.5))
                unit-size unit-size)
          (send dc set-font unit-font)
          (send dc set-text-foreground (make-object color% 255 255 255))
          (let ((str (number->string rank)))
            (let-values (((w h d s) 
			  (send dc get-text-extent str default-font)))
              (send dc draw-text str 
                    (- (car pos) (* w 0.5)) (- (cadr pos) (* h 0.5)))))
          (cond ((unit-defends? unit)
                 (send dc set-brush no-brush)
                 (let ((csize (* 0.7 circle-size)))
                   (send dc set-pen black-thick-pen)
                   (send dc draw-ellipse 
                         (- (car pos) (/ csize 2) -1)
                         (- (cadr pos) (/ csize 2) -1)
                         csize csize)
                   (send dc draw-line 
                         (- (car pos) (/ csize 2) -1) (cadr pos)
                         (- (car pos) (/ csize 2) -1 -6) (cadr pos))
                   (send dc draw-line 
                         (+ (car pos) (/ csize 2) -1 -6) (cadr pos)
                         (+ (car pos) (/ csize 2) -1) (cadr pos))
                   (send dc draw-line 
                         (car pos) (- (cadr pos) (/ csize 2) -1)
                         (car pos) (- (cadr pos) (/ csize 2) -1 -6))
                   (send dc draw-line 
                         (car pos) (+ (cadr pos) (/ csize 2) -1 -6)
                         (car pos) (+ (cadr pos) (/ csize 2) -1))
                   (send dc set-pen red-pen)
                   (send dc draw-ellipse 
                         (- (car pos) (/ csize 2) -1)
                         (- (cadr pos) (/ csize 2) -1)
                         csize csize)
                   (send dc draw-line 
                         (- (car pos) (/ csize 2)) (cadr pos)
                         (- (car pos) (/ csize 2) -7) (cadr pos))
                   (send dc draw-line 
                         (+ (car pos) (/ csize 2) -7) (cadr pos)
                         (+ (car pos) (/ csize 2)) (cadr pos))
                   (send dc draw-line 
                         (car pos) (- (cadr pos) (/ csize 2))
                         (car pos) (- (cadr pos) (/ csize 2) -7))
                   (send dc draw-line 
                         (car pos) (+ (cadr pos) (/ csize 2) -7)
                         (car pos) (+ (cadr pos) (/ csize 2)))))
                ((unit-attacks? unit)
                 (send dc set-brush no-brush)
                 (send dc set-pen black-2-pen)
                 (let ((csize (* 0.4 circle-size)))
                   (send dc draw-polygon
                         (list 
                          (make-object point% (car pos) (+ (cadr pos) csize))
                          (make-object point% (+ (car pos) csize) (cadr pos))
                          (make-object point% (car pos) (- (cadr pos) csize))
                          (make-object point% (- (car pos) csize) (cadr pos)))
                         0 0 'odd-even)))
                ((unit-blocked? unit)
                 (send dc set-text-foreground 
                       (if (eq? side 'r) 
                           (make-object color% 255 155 155)
                           (make-object color% 155 155 255)))
                 (let ((str (number->string rank)))
                   (let-values (((w h d s) 
                                 (send dc get-text-extent str default-font)))
                     (send dc draw-text str 
                           (- (car pos) (* w 0.5)) 
                           (- (cadr pos) (* h 0.5)))))))))
      (let loop ((x 0) (y 0))
        (cond ((>= y board-size) 'done)
              ((>= x board-size) (loop 0 (+ y 1)))
              ((null? (board-ref board x y)) (loop (+ x 1) y))
              (else (let ((unit (board-ref board x y))) 
                      (draw-unit x y unit))
                    (loop (+ x 1) y)))))

    (define (display-board b . args)
      (set! board b)
      (if (> (length args) 0) (set! turn (car args)))
      (if (> (length args) 1) (set! allowed-moves (cadr args)))
      (send board-dc clear)
      (draw-gameboard board-dc)
      (draw-units board-dc)
      (send (send canvas get-dc) draw-bitmap board-bitmap 0 0))

    (define (display-line . args)
      (send (send textarea get-editor) lock #f)
      (send (send textarea get-editor) insert 
            (apply string-append 
                   (append (map (lambda (e) 
                                  (if (string? e) e (expr->string e))) 
                                args)
                           (list "\n"))))
      (send (send textarea get-editor) lock #t))
          
    (define (display-move move b)
      (if (eq? move 'pass)
          (display-line "Pass")
          (display-line (expr->string (move-unit move)) " "
                        (symbol->string (move-type move)) "s " 
                        (expr->string (move-dest move))))
      (set! board b)
      (send board-dc clear)
      (draw-gameboard board-dc)
      (draw-units board-dc)
      (send (send canvas get-dc) draw-bitmap board-bitmap 0 0))
    
    (define (display-ai-action . args)
      (if debug (apply display-line (cons "** " args))))
    
    (define (get-game-action p1 p2)
      (if (not (eq? player1 p1))
          (if (eq? p1 'ai)
              (begin (set! player1 p1)
                     (send players-1-ai check #t)
                     (send players-1-human check #f))
              (begin (set! player1 p1)
                     (send players-1-ai check #f)
                     (send players-1-human check #t))))
      (if (not (eq? player2 p2))
          (if (eq? p2 'ai)
              (begin (set! player2 p2)
                     (send players-2-ai check #t)
                     (send players-2-human check #f))
              (begin (set! player2 p2)
                     (send players-2-ai check #f)
                     (send players-2-human check #t))))
      (cond (ai-1-mod
	     (set! ai-1-mod #f)	     
	     (list 'ai 1 ai-1-type ai-1-depth ai-1-maxdepth))
	    (ai-2-mod
	     (set! ai-2-mod #f)	     
	     (list 'ai 2 ai-2-type ai-2-depth ai-2-maxdepth))
	    ((and (or (and (eq? p1 'ai) (eq? turn 'b))
                      (and (eq? p2 'ai) (eq? turn 'r)))
                  (or (null? action) (is-move? action)))
             (send (send textarea get-editor) lock #f)
             (send (send textarea get-editor) insert "** AI running...\n")
             (send (send textarea get-editor) lock #t)
             (set! selected-square '())
             (set! game-action-state #f)
             '(play 1))
            ((and (not (null? allowed-moves))
                  (let ((moves (hash-table-map allowed-moves 
                                               (lambda (key value) 
                                                 (list key value)))))
                    (or (null? moves)
                        (and (= (length moves) 1)
                             (list? (cadr (car moves)))
                             (eq? (car (car (cadr (car moves)))) 'destroy)))))
             '(play 1))
            (else
             (set! game-action-state #t)
             (let wait ()
               (sleep/yield 0.05)
               (if (null? action)
                   (wait)
                   (let ((a action))
                     (cond ((is-move? a)
                            (set! action a)
                            '(play 1))
                           (else
                            (set! action '())
                            a))))))))
    
    (define (get-player-action b t a-moves)
      (if (is-move? action) 
          (let ((a action))
            (set! player-action-state #f)
            (set! action '())
            a)
          (let wait ()
            (set! player-action-state #t)
            (sleep/yield 0.05)
            (if (null? action) (wait) 
                (begin (set! player-action-state #f)
                       (let ((a action))
                         (set! action '())
                         a))))))
 
    (define (get-immediate-action bt rt)
      (set! blue-time bt)
      (set! red-time rt)
      (draw-update board)
      (sleep/yield 0.05)
      (cond ((null? action) '())
            ((is-move? action) '())
            (else action)))

    (define (quit-menu message info)
      (if direct-quit 
	  info
	  (let* ((quit #f)
		 (restart #f)
		 (dialog (instantiate frame% () (label "Game over")
				      (width 300) (height 100)))
		 (msg (instantiate message% ()
				   (label (string-append
					   message "\n"
				       (expr->string info)))
				   (parent dialog)))
		 (button-panel (instantiate horizontal-panel% (dialog)
					    (alignment '(center center))))
		 (restart-button (instantiate 
				  button% 
				  ("Restart" button-panel 
				   (lambda (button event) 
				     (set! restart #t)
				     (send dialog show #f)))))
		 (quit-button (instantiate 
			       button% 
			       ("Quit" button-panel 
				(lambda (button event) 
				  (set! quit #t)
				  (send dialog show #f))))))
	    (send dialog show #t)
	    (let wait ()
	      (sleep/yield 0.05)
	      (cond (quit 
		     (send frame show #f)
		     info)
		    (restart
		     (set! action '())
		     #f)
		    (else (wait)))))))

    (define (popup-ai-menu!)
      (let* ((accept #f)
	     (type-labels (list "Alfa-beta" "Iterative alfa-beta" 
                                "MTD" "Iterative MTD" "PVS"))
	     (types (list 'plain 'iter 'mtd 'imtd 'pvs))
	     (1-type ai-1-type)
	     (1-depth ai-1-depth)
	     (1-maxdepth ai-1-maxdepth)
	     (2-type ai-2-type)
	     (2-depth ai-2-depth)
	     (2-maxdepth ai-2-maxdepth)
	     (dialog (instantiate dialog% () (label "AI configuration")
				  (width 200) (height 300)
				  (alignment '(left top))))
	     (msg (instantiate message% ()
			       (label "Blue AI")
			       (parent dialog)))
	     (type-select-1 (instantiate 
			     choice% () 
			     (label "Type:") 
			     (choices type-labels) 
			     (parent dialog) 
			     (callback (lambda (button event)
					 (set! ai-1-mod #t)))))
	     (depth-select-1 (instantiate
			      slider% () 
			      (label "Depth:") 
			      (min-value 1) 
			      (max-value 8) 
			      (parent dialog) 
			      (callback (lambda (button event)
					 (set! ai-1-mod #t))) 
			      (init-value 1-depth)))
	     (maxdepth-select-1 (instantiate
				 slider% () 
				 (label "Max. depth:") 
				 (min-value 4) 
				 (max-value 10) 
				 (parent dialog) 
				 (callback (lambda (button event)
					     (set! ai-1-mod #t))) 
				 (init-value 1-maxdepth)))
	     (msg (instantiate message% ()
			       (label "Red AI")
			       (parent dialog)))
	     (type-select-2 (instantiate 
			     choice% () 
			     (label "Type:") 
			     (choices type-labels) 
			     (parent dialog) 
			     (callback (lambda (button event)
					 (set! ai-2-mod #t)))))
	     (depth-select-2 (instantiate
			      slider% () 
			      (label "Depth:") 
			      (min-value 1) 
			      (max-value 8) 
			      (parent dialog) 
			      (callback (lambda (button event)
					 (set! ai-2-mod #t))) 
			      (init-value 2-depth)))
	     (maxdepth-select-2 (instantiate
				 slider% () 
				 (label "Max. depth:") 
				 (min-value 4) 
				 (max-value 10) 
				 (parent dialog) 
				 (callback (lambda (button event)
					     (set! ai-2-mod #t))) 
				 (init-value 2-maxdepth)))
	     (button-panel (instantiate horizontal-panel% (dialog)
					(alignment '(center center))))
	     (accept-button (instantiate 
			     button% 
			     ("Accept" button-panel 
			      (lambda (button event) 
				(set! accept #t)
				(send dialog show #f)))))
	     (cancel-button (instantiate 
			     button% 
			     ("Cancel" button-panel 
			      (lambda (button event) 
				(send dialog show #f))))))
	(send dialog show #t)
	(cond ((and accept (or ai-1-mod ai-2-mod))
	       (set! ai-1-type 
		     (list-ref types (send type-select-1 get-selection)))
	       (set! ai-1-depth (send depth-select-1 get-value))
	       (set! ai-1-maxdepth (send maxdepth-select-1 get-value))
	       (set! ai-2-type
		     (list-ref types (send type-select-2 get-selection)))
	       (set! ai-2-depth (send depth-select-2 get-value))
	       (set! ai-2-maxdepth (send maxdepth-select-2 get-value))
	       (display-line "AI options updated.")))))
    
    (define (init! i-player1 i-player2 
		   i-ai-1-type i-ai-1-depth i-ai-1-maxdepth
		   i-ai-2-type i-ai-2-depth i-ai-2-maxdepth)
      (set! player1 i-player1)
      (set! player2 i-player2)
      (if (eq? player1 'ai)
	  (begin (send players-1-ai check #t)
		 (send players-1-human check #f))
	  (begin (send players-1-ai check #f)
		 (send players-1-human check #t)))
      (if (eq? player2 'ai)
	  (begin (send players-2-ai check #t)
		 (send players-2-human check #f))
	  (begin (send players-2-ai check #f)
		 (send players-2-human check #t)))
      (set! ai-1-type i-ai-1-type)
      (set! ai-1-depth i-ai-1-depth)
      (set! ai-1-maxdepth i-ai-1-maxdepth)
      (set! ai-2-type i-ai-2-type)
      (set! ai-2-depth i-ai-2-depth)
      (set! ai-2-maxdepth i-ai-2-maxdepth))
    
    (define (dispatch m . args)
      (cond ((eq? m 'display-move) (apply display-move args))
            ((eq? m 'display-board) (apply display-board args))
            ((eq? m 'display-ai-action) (apply display-ai-action args))
            ((eq? m 'get-immediate-action) (apply get-immediate-action args))
            ((eq? m 'get-game-action) (apply get-game-action args))
            ((eq? m 'get-player-action) (apply get-player-action args))
            ((eq? m 'event-type) 'thread)
            ((eq? m 'redraw-frame) (send frame show #t))
	    ((eq? m 'quit) (apply quit-menu args))
	    ((eq? m 'init) (apply init! args))
            (else (error "MAKE-GUI: unknown message" m))))
    
    (if (member 'nodebug args) (set! debug #f))

    (set! popup-ai-menu popup-ai-menu!)
    (set! draw-update display-board)
    (display-board board)
    (send (send textarea get-editor) lock #t)
    (send frame show #t)
    (sleep/yield 0.5)
    
    dispatch))


