;;;
;;; Hierarkia - Scheme Implementation
;;; hierarkia.scm: Core of the game
;;;
;;; Copyright 2003 Sami Virpioja, Ilari Lhteenmki
;;;
;;; ---
;;;
;;; This file is part of Hierarkia - Scheme Implementation.
;;;
;;; Hierarkia - Scheme Implementation is free software; you can redistribute 
;;; it and/or modify it under the terms of the GNU General Public License as 
;;; published by the Free Software Foundation; either version 2 of the 
;;; License, or (at your option) any later version.
;;;
;;; Hierarkia - Scheme Implementation  is distributed in the hope that it will 
;;; be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
;;; Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License along 
;;; with Hierarkia - Scheme Implementation; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;
;;; ---
;;;
;;; See README for more details.
;;;
;;; This file contains the core of the game. The game should be started
;;; by evaluating rungame.scm.
;;;

;;
;; Miscellaneous procedures & definitions
;;

(define (display-line . args)
  (for-each display args)
  (newline))

(define (sign x) (if (< x 0) -1 1))

;; Array definitions

(define (make-array x y value)
  (let ((v (make-vector y)))
    (let xset ((n 0))
      (if (< n y) 
	  (begin (vector-set! v n (make-vector x value)) 
		 (xset (+ n 1)))))
    v))
(define (array-xsize array)
  (vector-length array))
(define (array-ysize array)
  (vector-length (vector-ref array 0)))
(define (array-ref array x y)
  (vector-ref (vector-ref array y) x))
(define (array-set! array x y value)
  (vector-set! (vector-ref array y) x value))

;; Queue definitions

(define (make-queue) (cons '() '()))
(define (front-ptr q) (car q))
(define (rear-ptr q) (cdr q))
(define (set-front-ptr! q i) (set-car! q i))
(define (set-rear-ptr! q i) (set-cdr! q i))
(define (empty-queue? q) (null? (front-ptr q)))
(define (front-queue q)
  (if (empty-queue? q)
      (error "FRONT called with an empty queue" q)
      (car (front-ptr q))))
(define (insert-queue! q i)
  (let ((new-pair (cons i '())))
    (cond ((empty-queue? q)
	   (set-front-ptr! q new-pair)
	   (set-rear-ptr! q new-pair)
	   q)
	  (else
	   (set-cdr! (rear-ptr q) new-pair)
	   (set-rear-ptr! q new-pair)
	   q))))
(define (delete-queue! q)
  (cond ((empty-queue? q)
	 (error "DELETE! called with an empty queue" q))
	(else
	 (set-front-ptr! q (cdr (front-ptr q)))
	 q)))

;; List operations

(define (accumulate op init seq)
  (if (null? seq) init (op (car seq) (accumulate op init (cdr seq)))))

(define (flatmap proc seq)
  (accumulate append '() (map proc seq)))

(define (filter pred seq)
  (cond ((null? seq) '())
        ((pred (car seq)) (cons (car seq) (filter pred (cdr seq))))
        (else (filter pred (cdr seq)))))

;;
;; Game procedures
;;

(define dirs '((1 0) (0 1) (-1 0) (0 -1)))

(define (next t) (if (eq? t 'r) 'b 'r))

(define (board-coordinates board-size)
  (define vec (let loop ((i 0)) 
                (if (< i board-size) (cons i (loop (+ i 1))) '())))
  (flatmap (lambda (x) (map (lambda (y) (list x y)) vec)) vec))

;; Unit definitions

(define (make-unit . params) 
  (if (> 6 (length params) 1)
      (let ((v (make-vector 5 #f)))
	(let loop ((plist params) (i 0))
	  (cond ((null? plist) v)
		(else (vector-set! v i (car plist))
		      (loop (cdr plist) (+ i 1))))))
      (error "procedure make-unit: expects 2 to 5 arguments, given" params)))

(define (copy-unit u)
  (if (not (vector? u)) '()
      (let ((v (make-vector 5 #f)))
	(let loop ((i 0))
	  (cond ((> i 4) v)
		(else (vector-set! v i (vector-ref u i))
		      (loop (+ i 1))))))))

(define (unit-side u) (vector-ref u 0))
(define (unit-rank u) (vector-ref u 1))
(define (unit-target u) (vector-ref u 2))
(define (unit-attacks? u) (unit-target u))
(define (unit-defends? u) (vector-ref u 3))
(define (unit-blocked? u) (vector-ref u 4))

(define (unit-attack! u t) (vector-set! u 2 t))
(define (unit-defend! u) (vector-set! u 3 #t))
(define (unit-block! u) (vector-set! u 4 #t))
(define (unit-restore! u)
  (vector-set! u 2 #f) (vector-set! u 3 #f) (vector-set! u 4 #f))

;; Board procedures

(define (make-board) (make-array board-size board-size '()))

(define (board-ref board x y)
  (let ((s (array-ref board x y)))
    (if (null? s) '()
        (if (vector? s) s 
            (if (> s 0) (make-unit 'b s) (make-unit 'r (- s)))))))

(define (board-set! board x y u) 
  (if (and (vector? u) (not (vector-ref u 2)) 
           (not (vector-ref u 3)) (not (vector-ref u 4)))
      (array-set! board x y 
                  (if (eq? (vector-ref u 0) 'b)
                      (vector-ref u 1)
                      (- (vector-ref u 1))))
      (array-set! board x y u)))

(define (init-board! board)
  (if (not (< 0 max-rank board-size)) 
      (error "init-board: illegal board parameters"))
  (let loop ((x 0) (y 0) (br max-rank) 
	     (rr (- max-rank (* 2 (- board-size 1))))) 
    (cond ((> y (- board-size 1)) (void))
	  ((> x (- board-size 1)) (loop 0 (+ y 1) br rr))
	  (else (cond ((> (- br x y) 0) 
		       (board-set! board x y (- br x y)))
		      ((> (+ rr x y) 0)
		       (board-set! board x y (- 0 rr x y)))
		      (else 
		       (board-set! board x y '())))
		(loop (+ x 1) y br rr)))))

(define (board-copy board)
  (define (copy-sq sq)
    (if (null? sq) '() 
        (if (vector? sq) (copy-unit sq)
            sq)))
  (let ((new (make-board)))
    (let loop ((x 0) (y 0))
      (cond ((> y (- board-size 1)) new)
	    ((> x (- board-size 1)) (loop 0 (+ y 1)))
	    (else (array-set! new x y (copy-sq (array-ref board x y)))
		  (loop (+ x 1) y))))))

;; Coordinates inside game board
(define inside? (lambda (x y) (and (< -1 x board-size) 
                                   (< -1 y board-size))))

;; Coordinates inside blue's area
(define inside-blue? 
  (lambda (x y) (and (inside? x y) (< (+ x y) max-rank))))

;; Coordinates inside red's area
(define inside-red? 
  (lambda (x y) (and (inside? x y) 
                     (> (+ x y) (- (* 2 (- board-size 1)) max-rank)))))

;; Returns a list representing given board
(define (board->list d)
  (map 
   (lambda (v)
     (map (lambda (p)
            (if (null? p) '--
                (if (vector? p)
                    (string->symbol 
                     (string-append (symbol->string (unit-side p))
                                    (number->string (unit-rank p))))
                    (if (> p 0) 
                        (string-append "b" (number->string p))
                        (string-append "r" (number->string (- p)))))))
          (vector->list v)))
   (vector->list d)))

;; Procedure for printing out the given board
(define (print-board l)
  (for-each (lambda (r) (display r) (newline)) (board->list l)))


;; Finds shortest route for start to destination coordinates, moving
;; only in empty squares. If no route is found, returns value two times
;; of the board's size.
(define (get-min-route board sx sy dx dy)
  (let* ((start (list sx sy)) 
	 (dest (list dx dy))
	 (q (make-queue)) 
	 (visit (list start)))
    (define (legal? pos) 
      (and (inside? (car pos) (cadr pos))
	   (if (equal? pos dest) #t
	       (null? (board-ref board (car pos) (cadr pos))))))
    (define (seek pos len)
      (cond ((equal? pos dest) len)
	    (else (for-each 
		   (lambda (p) 
		     (cond ((and (legal? p)
                                 (not (member p visit)))
			    (set! visit (cons p visit))
			    (insert-queue! q (list p (+ len 1))))))
		   (map (lambda (dir) (map + pos dir)) dirs))
		  (if (empty-queue? q) 
		      (* 2 board-size)  ; won't get any bigger :)
		      (let ((new (front-queue q)))
			(delete-queue! q)
			(seek (car new) (cadr new)))))))
   (seek start 0)))


;; Return all units that can be fired or moved from unit in (sx, sy) and 
;; lengths of the routes in a list.
(define (get-route-lengths board sx sy side)
  (let* ((start (list sx sy))
	 (q (make-queue))
	 (visit (list start))
         (attacklist '())
         (movelist '()))
    (define (legal? pos)
      (if (inside? (car pos) (cadr pos))
          (let ((square (board-ref board (car pos) (cadr pos))))
            (if (null? square)
                #t
                (not (eq? (unit-side square) side))))
          #f))
    (define (seek pos len)
      (if (and (> len 0) (not (null? (board-ref board (car pos) (cadr pos)))))
          (set! attacklist (cons (cons pos len) attacklist))
          (begin
            (if (> len 0) 
                (let ((mr (free-route? board sx sy (car pos) (cadr pos))))
                  (if mr (set! movelist (cons (cons pos mr) movelist)))))
            (for-each 
             (lambda (p) 
               (cond ((and (legal? p)
                           (not (member p visit)))
                      (set! visit (cons p visit))
                      (insert-queue! q (list p (+ len 1))))))
             (map (lambda (dir) (map + pos dir)) dirs))))
      (if (empty-queue? q) 
          (cons movelist attacklist)
          (let ((new (front-queue q)))
            (delete-queue! q)
            (seek (car new) (cadr new)))))
    (seek start 0)))


;; Unit in its home area
(define (in-home? u x y)
  (if (eq? (unit-side u) 'r) (inside-red? x y) (inside-blue? x y)))

;; Checks if there exist a free direct route from (x1 y1) to (x2 y2) and 
;; returns its length
(define (free-route? board x1 y1 x2 y2)
  (cond ((= x1 x2)
         (if (= y1 y2) 0
             (let* ((dir (sign (- y2 y1))))
               (let loop ((y (+ y1 dir)) (len 1))
                 (cond ((= y y2) (if (null? (board-ref board x1 y)) len #f))
                       ((not (null? (board-ref board x1 y))) #f)
                       (else (loop (+ y dir) (+ len 1))))))))
	((= y1 y2)
	 (let* ((dir (sign (- x2 x1))))
	   (let loop ((x (+ x1 dir)) (len 1))
	     (cond ((= x x2) (if (null? (board-ref board x y1)) len #f))
		   ((not (null? (board-ref board x y1))) #f)
		   (else (loop (+ x dir) (+ len 1)))))))
	(else #f)))

;; Unit can move to given coordinates
(define (legal-move? board ux uy tx ty dist)
  (and (not (null? (board-ref board ux uy)))
       (let* ((unit (board-ref board ux uy))
	      (rank (unit-rank unit))
	      (max-dist (if (> rank 1) rank (if (in-home? unit ux uy) 2 1))))
	 (and (not (unit-defends? unit))
	      (>= max-dist dist)))))

;; Gets firepower modifier from square (mx, my) when side is attackers side
(define (get-firepower-modifier board mx my side turn)
  (cond ((not (inside? mx my)) 0)
        ((null? (board-ref board mx my)) 0)
        (else (let ((modifier (board-ref board mx my)))
                (if (eq? side (unit-side modifier))
                    (if (and (unit-defends? modifier)
                             (eq? (unit-side modifier) turn))
                        0
                        (unit-rank modifier))
                    (- (unit-rank modifier)))))))

;; Checks if unit can attack to given coordinates.
;; Parameter turn is given so that if unit that is attacked in opponent's 
;; turn, it still can continue its own attack.
(define (legal-attack? board ux uy tx ty turn dist)
  (and (not (null? (board-ref board ux uy)))
       (let* ((unit (board-ref board ux uy)) (side (unit-side unit)))
	 (and (not (and (unit-defends? unit) (eq? side turn)))
	      (not (unit-blocked? unit))
	      (not (in-home? unit ux uy))
	      (let ((max-dist 
		     (apply + (map (lambda (p) 
				     (get-firepower-modifier 
				      board (car p) (cadr p) side turn))
				   (map (lambda (dir) (map + (list ux uy) dir))
					dirs)))))
		(>= max-dist dist))))))

;; Returns all allowed moves of a unit in (ux, uy), including shooting
(define (get-unit-moves board ux uy turn)
  (let* ((unit (board-ref board ux uy))
         (routes (get-route-lengths board ux uy turn))
         (side (unit-side unit))
         (moves '()))
    (let move-loop ((l (car routes)))
      (cond ((null? l) (void))
            ((legal-move? board ux uy 
                          (car (car (car l))) (cadr (car (car l)))
                          (cdr (car l)))
             (set! moves (cons (cons 'move (car (car l))) moves))
             (move-loop (cdr l)))
            (else (move-loop (cdr l)))))
    (let attack-loop ((l (cdr routes)))
      (cond ((null? l) (void))
            ((legal-attack? board ux uy 
                            (car (car (car l))) (cadr (car (car l)))
                            turn (cdr (car l)))
             (set! moves (cons (cons 'attack (car (car l))) moves))
             (attack-loop (cdr l)))
            (else (attack-loop (cdr l)))))
    moves))

;; Gets winner's side from a board with an ended game
(define (get-winner board)
  (let ((red-general? #f) (blue-general? #f))
    (let loop ((x 0) (y 0))
      (cond ((and red-general? blue-general?) #f)
            ((> y (- board-size 1)) (if red-general? 'r 'b))
            ((> x (- board-size 1)) (loop 0 (+ y 1)))
            (else 
             (if (null? (board-ref board x y)) 
                 (loop (+ x 1) y)
                 (let ((u (board-ref board x y)))
                   (if (= (unit-rank u) max-rank)
                       (if (eq? (unit-side u) 'r) 
                           (set! red-general? #t)
                           (set! blue-general? #t)))
                   (loop (+ x 1) y))))))))

;; Returns a hash table of all possible moves for player of the given side.
;; Hash keys are unit coordinates, values are allowed moves for that unit.
(define (get-allowed-moves board turn)
  (let ((moves (make-hash-table 'equal)))
    (let ((attacker (find-attacker board turn)))
      (if attacker 
          (let* ((ax (car attacker)) (ay (cadr attacker))
                 (att-unit (board-ref board ax ay))
                 (tx (car (unit-target att-unit)))
                 (ty (cadr (unit-target att-unit))))
            (hash-table-put! 
             moves (list ax ay) (list (list 'destroy tx ty)))
            moves)
          (let loop ((x 0) (y 0))
            (cond ((> y (- board-size 1)) moves)
                  ((> x (- board-size 1)) (loop 0 (+ y 1)))
                  (else 
                   (cond ((null? (board-ref board x y)) (loop (+ x 1) y))
                         ((eq? turn (unit-side (board-ref board x y)))
                          (let* ((this-moves (get-unit-moves board x y turn)))
                            (if (not (null? this-moves))
                                (hash-table-put! moves (list x y) this-moves))
                            (loop (+ x 1) y)))
                         (else (loop (+ x 1) y))))))))))

;; Structure for movement data
(define (make-move ux uy type tx ty)
  (list (list ux uy) (list type tx ty)))
(define (is-move? m)
  (or (eq? m 'pass)
      (and (list? m) (= (length m) 2) (list? (car m)) (list? (cadr m))
           (= (length (car m)) 2) (= (length (cadr m)) 3))))
(define (move-unit-x move) (car (car move)))
(define (move-unit-y move) (cadr (car move)))
(define (move-type move) (car (cadr move)))
(define (move-target-x move) (cadr (cadr move)))
(define (move-target-y move) (caddr (cadr move)))
(define (move-unit move) (car move))
(define (move-move move) (cadr move))
(define (move-dest move) (cdr (cadr move)))

;; Makes a move in the given board
(define (apply-move! board move)
  (if (eq? move 'pass) 
      board
      (let* ((ux (move-unit-x move)) (uy (move-unit-y move))
             (unit (board-ref board ux uy)) (type (move-type move))
             (tx (move-target-x move)) (ty (move-target-y move))
             (turn (unit-side unit)))
        (if (null? unit) (error "apply-move! no unit found for move" move))
        (cond ((eq? type 'move)
               (let ((attacker (find-attacker board (next turn))))
                 (board-set! board ux uy '())
                 (board-set! board tx ty unit)
                 (if attacker
                     (let* ((ax (car attacker)) (ay (cadr attacker))
                            (att-unit (board-ref board ax ay))
                            (tx (car (unit-target att-unit)))
                            (ty (cadr (unit-target att-unit))))
                       (if (not (legal-attack? board ax ay tx ty turn
                                               (get-min-route 
                                                board ax ay tx ty)))
                           (begin (unit-restore! att-unit)
                                  (unit-block! att-unit)
                                  (board-set! board ax ay att-unit)
                                  (let ((tg-unit (board-ref board tx ty)))
                                    (unit-restore! tg-unit)
                                    (board-set! board tx ty tg-unit)))))))
               (remove-blocks! board turn)
               board)
              ((eq? type 'attack)
               (let ((target (board-ref board tx ty)))
                 (unit-defend! target)
                 (unit-attack! unit (list tx ty))
                 (board-set! board ux uy unit)
                 (board-set! board tx ty target))
               (remove-blocks! board turn)
               board)
              ((eq? type 'destroy)
               (remove-blocks! board turn)
               (unit-restore! unit)
               (unit-block! unit)
               (board-set! board ux uy unit)
               (board-set! board tx ty '())
               board)
              (else (error "apply-move!: unknown move type" move))))))
  
(define (fork-move board move) (apply-move! (board-copy board) move))

;; Does given move block an attack?
(define (is-block-move? board turn move)
  (and (find-attacker board (next turn))
       (not (find-attacker (fork-move board move) (next turn)))))

;; Is there unit on turn's side that attacks?
(define (find-attacker board turn)
  (let loop ((x 0) (y 0))
    (cond ((> y (- board-size 1)) #f)
          ((> x (- board-size 1)) (loop 0 (+ y 1)))
          (else 
           (if (null? (board-ref board x y))
               (loop (+ x 1) y)
               (let* ((u (board-ref board x y)))
                 (if (and (eq? (unit-side u) turn)
                          (unit-attacks? u))
                     (list x y)
                     (loop (+ x 1) y))))))))

;; Remove block from turn's unit
(define (remove-blocks! board turn)
  (let loop ((x 0) (y 0))
    (cond ((> y (- board-size 1)) 'done)
          ((> x (- board-size 1)) (loop 0 (+ y 1)))
          (else 
           (if (null? (board-ref board x y))
               (loop (+ x 1) y)
               (let* ((u (board-ref board x y)))
                 (if (and (eq? (unit-side u) turn)
                          (unit-blocked? u))
                     (begin (unit-restore! u)
                            (board-set! board x y u))
                     (loop (+ x 1) y))))))))


;; Counts number of allowed moves in a hash, used in calculation of 
;; branching factor
(define (count-moves allowed-moves)
  (let ((hash-list
         (hash-table-map allowed-moves (lambda (key value) value))))
    (let each-unit ((hl hash-list) (n 0))
      (cond ((null? hl) n)
            ((null? (car hl)) (each-unit (cdr hl) n))
            (else (each-unit (cdr hl) (+ n (length (car hl)))))))))

(define (hash-size h)
  (let ((n 0)) (hash-table-for-each h (lambda (k v) (set! n (+ n 1)))) n))

;;
;; Procedure for starting the game. Arguments are player and GUI objects.
;;

(define (start-game p1 p2 gui . args)


  (define ai-1-depth 2)
  (define ai-1-maxdepth 4)
  (define ai-1-type 'iter)
  (define ai-2-depth 2)
  (define ai-2-maxdepth 4)
  (define ai-2-type 'iter)

  (define timer 0)
  (define btime 0)
  (define rtime 0)

  (define (quit-action? a) (eq? a 'quit))
  (define (restart-action? a) (eq? a 'restart))
  (define (play-action? a) 
    (and (list? a) (eq? (car a) 'play) 
	 (not (null? (cdr a))) (number? (cadr a))))
  (define (set-human-action? a) 
    (and (list? a) (eq? (car a) 'human)
	 (not (null? (cdr a))) (number? (cadr a))))
  (define (set-ai-action? a) 
    (and (list? a) (eq? (car a) 'ai)
	 (not (null? (cdr a))) (number? (cadr a))))

  
  (define (main-loop board turn prev-move rounds-to-play branchs n)

    (define (handle-game-action action)
      (cond ((quit-action? action)
             (let ((q (gui 'quit "Game aborted\n"
			   (if (> n 0)
			       (list 'turns n 'branch-factor (/ branchs n 1.0))
			       (list 'turns 0)))))
	       (if q q (handle-game-action 'restart))))

            ((restart-action? action)
             (if (eq? (p1 'get-player-type) 'ai)
                 (set! p1 (make-ai-player ai-1-type gui 
					  ai-1-depth ai-1-maxdepth))
                 (set! p1 (make-human-player gui)))
             (if (eq? (p2 'get-player-type) 'ai)
                 (set! p2 (make-ai-player ai-2-type gui 
					  ai-2-depth ai-2-maxdepth))
                 (set! p2 (make-human-player gui)))
             (set! timer (current-milliseconds)) 
             (set! btime 0) (set! btime 0)
             (init-board! board)
             (main-loop board 'b '() 0 0 0))

            ((set-human-action? action)
             (if (= (cadr action) 1)
                 (set! p1 (make-human-player gui))
                 (set! p2 (make-human-player gui)))
             (main-loop board turn prev-move 0 branchs n))

            ((set-ai-action? action)
             (if (= (cadr action) 1)
                 (begin
		   (if (> (length action) 2)
		       (set! ai-1-type (list-ref action 2)))
		   (if (> (length action) 3)
		       (set! ai-1-depth (list-ref action 3)))
		   (if (> (length action) 4)
		       (set! ai-1-maxdepth (list-ref action 4)))
		   (set! p1 (make-ai-player ai-1-type gui 
					    ai-1-depth ai-1-maxdepth)))
                 (begin 
		   (if (> (length action) 2)
		       (set! ai-2-type (list-ref action 2)))
		   (if (> (length action) 3)
		       (set! ai-2-depth (list-ref action 3)))
		   (if (> (length action) 4)
		       (set! ai-2-maxdepth (list-ref action 4)))
		   (set! p2 (make-ai-player ai-2-type gui 
					    ai-2-depth ai-2-maxdepth))))
             (main-loop board turn prev-move 0 branchs n))

            ((play-action? action)
             (if (> (cadr action) 0)
                 (main-loop board turn prev-move (cadr action) 
                            branchs n)
                 (main-loop board turn prev-move 0 branchs n)))

            (else (error "unknown action" action))))

    (collect-garbage)
    (gui 'display-ai-action (current-memory-use))
    (let ((allowed-moves (get-allowed-moves board turn)))
      (gui 'display-board board turn allowed-moves 
           (if (eq? turn 'b)
               (+ btime (- (current-milliseconds) 
                           timer))
               btime)
           (if (eq? turn 'r)
               (+ rtime (- (current-milliseconds) 
                           timer))
               rtime))
      (cond 
       ((get-winner board) 
	(let ((q (gui 'quit 
		      (if (eq? (get-winner board) 'b)
			  "Blue player won\n"
			  "Red player won\n")
		      (list 'turns n 'branch-factor (/ branchs n 1.0)))))
	  (if q q (handle-game-action 'restart))))
       ((> rounds-to-play 0)
        (let* ((player (if (eq? turn 'b) p1 p2))
               (bf (count-moves allowed-moves))
               (new-move 'undef))
          (cond ((eq? (gui 'event-type) 'thread)
                 (let ((t (thread 
                           (lambda () 
                             (set! new-move 
                                   (player 'get-move board turn 
                                           prev-move allowed-moves))))))
		   (let wait ((action (gui 'get-immediate-action
                                           btime rtime)))
		     (cond ((not (null? action))
			    (kill-thread t)
			    (set! new-move action))
			   ((eq? new-move 'undef) 
			    ;; (sleep/yield 0.05) 
			    (wait (gui 'get-immediate-action 
                                       (if (eq? turn 'b)
                                           (+ btime (- (current-milliseconds) 
                                                       timer))
                                           btime)
                                       (if (eq? turn 'r)
                                           (+ rtime (- (current-milliseconds) 
                                                       timer))
                                           rtime))))
			   (else 'done)))))
                (else 
                 (set! timer (current-milliseconds))
                 (set! new-move (player 'get-move board turn 
                                        prev-move allowed-moves))))
          (if (not (is-move? new-move))
              (handle-game-action new-move)
              (begin
                (if (eq? turn 'b)
                    (set! btime (+ btime (- (current-milliseconds) timer))) 
                    (set! rtime (+ rtime (- (current-milliseconds) timer))))
                (if (not (eq? new-move 'pass))
                    (let ((u (move-unit new-move)) (m (move-move new-move)))
                      (let ((ul (hash-table-get 
                                 allowed-moves u
                                 (lambda () 
                                   (error "main-loop: illegal move from player"
                                          new-move)))))
                        (let ((ml (member m ul)))
                          (if (not ml) 
                              (error "main-loop: illegal move from player" 
                                     new-move))))))                
                (let ((new-board (if (eq? new-move 'pass)
                                     board 
                                     (fork-move board new-move))))
                  (gui 'display-move new-move new-board)
                  (set! timer (current-milliseconds)) 
                  (main-loop new-board (next turn) new-move 
                             (- rounds-to-play 1) (+ branchs bf) 
                             (+ n 1)))))))
       (else (let ((action (gui 'get-game-action 
                                (p1 'get-player-type)
                                (p2 'get-player-type))))
               (handle-game-action action))))))

  ;; Create players and start the game

  (if (eq? p1 'ai)
      (set! p1 (make-ai-player ai-1-type gui ai-1-depth ai-1-maxdepth)) 
      (set! p1 (make-human-player gui)))
  (if (eq? p2 'ai) 
      (set! p2 (make-ai-player ai-2-type gui ai-2-depth ai-2-maxdepth)) 
      (set! p2 (make-human-player gui)))


  (for-each 
   (lambda (action)
     (if (set-ai-action? action)
	 (if (= (cadr action) 1)
	     (begin
	       (if (> (length action) 2)
		   (set! ai-1-type (list-ref action 2)))
	       (if (> (length action) 3)
		   (set! ai-1-depth (list-ref action 3)))
	       (if (> (length action) 4)
		   (set! ai-1-maxdepth (list-ref action 4)))
	       (set! p1 (make-ai-player ai-1-type gui 
					ai-1-depth ai-1-maxdepth)))
	     (begin 
	       (if (> (length action) 2)
		   (set! ai-2-type (list-ref action 2)))
	       (if (> (length action) 3)
		   (set! ai-2-depth (list-ref action 3)))
	       (if (> (length action) 4)
		   (set! ai-2-maxdepth (list-ref action 4)))
	       (set! p2 (make-ai-player ai-2-type gui 
					ai-2-depth ai-2-maxdepth))))))
   args)

  (gui 'init 
       (p1 'get-player-type) (p2 'get-player-type)
       ai-1-type ai-1-depth ai-1-maxdepth
       ai-2-type ai-2-depth ai-2-maxdepth)

  (set! timer (current-milliseconds))

  (main-loop board 'b '() 0 0 0))


;;
;; Test procedure for calculating branching factor
;;

(define (test-game initial-board first-move)
  (define gui (make-ngui 'nodebug))
  (let ((p1 (make-ai-player 'mtd gui 4 4 )) 
	(p2 (make-ai-player 'mtd gui 4 4 )))
    (let main-loop ((board initial-board) (turn 'r) (prev-move first-move)
		    (rounds-to-play 2000) (branchs 0) (n 0))  
      (let ((allowed-moves (get-allowed-moves board turn)))
	(cond 
	 ((get-winner board) (/ branchs n 1.0))
	 ((> rounds-to-play 0)
	  (let* ((player (if (eq? turn 'b) p1 p2))
		 (bf (count-moves allowed-moves))
		 (new-move 
		  (player 'get-move board turn prev-move allowed-moves))) 
	    (begin
	      (if (not (eq? new-move 'pass))
		  (let ((u (move-unit new-move)) (m (move-move new-move)))
		    (let ((ul (hash-table-get 
			       allowed-moves u
			       (lambda () 
				 (error "main-loop: illegal move from player"
					new-move)))))
		      (let ((ml (member m ul)))
			(if (not ml) 
			    (error "main-loop: illegal move from player" 
				   new-move))))))                
	      (let ((new-board (if (eq? new-move 'pass)
				   board 
				   (fork-move board new-move))))
		(main-loop new-board (next turn) new-move 
			   (- rounds-to-play 1) (+ branchs bf) 
			   (+ n 1))))))
	 (else (/ branchs n 1.0)))))))
  



;; Opening-list contains all possible openings for the given board.
(define (opening-list board)
  (map (lambda (move) 
	 (fork-move board move))
       ((make-ai-player 'any (make-ngui 'nodebug)) 'get-successors board 'b)))


;; Each board is run in a test-game, which returns branching factor.
;; Result is list of branching-factors
(define (branching-factor-loop board-list branch-list)
  (if (or (null? board-list)  
	  (null? (car board-list))) 
      branch-list  
      (branching-factor-loop (cdr board-list) 
			     (cons (test-game (car board-list) '())
				   branch-list))))

;;
;; Branch-test calculates interesting statistical information of the branching
;; factors:
;; min, max, average, sample standard deviation, 95%-confidence interval
(define (branch-test)
  (let ((board (make-board)))
    (init-board! board)
    (let ((branches (branching-factor-loop (opening-list board) '())))
      (let stat-loop ((sample (car branches))
		      (n 0) (max 0) (min 100) (sum 0) (average 0) 
		      (square-sum 0) (sum-square 0)
		      (rest (cdr branches)))
	(if (null? sample)
	    (let* ((ssd (/ (- (* n square-sum) (* sum sum))
			   (* (- n 1) n)))
		   (lower (- average (* 2.069 (/ (sqrt ssd) (sqrt n)))))
		   (upper (+ average (* 2.069 (/ (sqrt ssd) (sqrt n))))))
	      (display-line "n: " n)
	      (display-line "max: " max)
	      (display-line "min: " min)
	      (display-line "average: " average)
	      (display-line "std. dev.: " ssd)
	      (display-line "lower 95%: " lower)
	      (display-line "upper 95%: " upper)
	      'done)
	    (let ((new-sum (+ sum sample))
		  (new-square-sum (+ square-sum (* sample sample))))
	      (stat-loop (if (null? rest) '() (car rest)) (+ n 1) 
			 (if (> sample max) sample max) 
			 (if (< sample min) sample min)
			 new-sum
			 (/ new-sum (+ n 1))
			 new-square-sum
			 (* new-sum new-sum)
			 (if (null? rest) '() (cdr rest)))))))))


