;;;
;;; Hierarkia - Scheme Implementation
;;; ai_iter.scm: AI object implementation
;;;
;;; Copyright 2003 Sami Virpioja, Ilari Lhteenmki
;;;
;;; ---
;;;
;;; This file is part of Hierarkia - Scheme Implementation.
;;;
;;; Hierarkia - Scheme Implementation is free software; you can redistribute 
;;; it and/or modify it under the terms of the GNU General Public License as 
;;; published by the Free Software Foundation; either version 2 of the 
;;; License, or (at your option) any later version.
;;;
;;; Hierarkia - Scheme Implementation  is distributed in the hope that it will 
;;; be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
;;; Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License along 
;;; with Hierarkia - Scheme Implementation; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;
;;; ---
;;;
;;; See README for more details.
;;;

;;
;; A message dispatching style object representing an AI player
;;

(define (make-iter-ai gui . args)

  ;; 5/5 6 turns (mred -z, oregano.niksula):
  ;; current 0:49, 0:42, 0:49
  ;; without tr-table fix 0:53, 0:47, 0:46

  ;; 5/5 6 turns (drscheme, mauste.niksula):
  ;; without tr-table fix 2:07
  ;; do not use current transitions in ordering 2:07, 2:09
  ;; without saved transitions (2:06 min) 2:34
  ;; without transition table: 5:15

  (define max-depth 2)
  (define max-rdepth 4)
  
  ;; transitions :: hash-table for saving evaluation of redundant states
  ;;
  ;; Pair (board, turn) (identifies the state of game) is used as a key and 
  ;; evaluation of the state (calculated utility) and corresponding node in 
  ;; the move tree are consed together to make the data

  (define transitions (make-hash-table 'equal))
  
  (define (get-saved-successors board turn depth)
    (let ((new (hash-table-get transitions (cons board turn)
			       (lambda () 'undef))))
      (cons (if (eq? new 'undef) 
		#t 
		(>= (cadr new) depth))
	    (if (or (eq? new 'undef) (null? (cddr new)))
		(get-successors board turn)
		((cddr new) 'get-values)))))

  ;; alfa-beta-search :: Base function for ai, makes the first move
  ;;
  ;; uses: * min-value to evaluate opponents response to possible moves.
  ;;       * fork-move to make an artificial board to plan future moves on.
  ;;       * get-successors to get possible moves
  ;;       * search-tree to get the best moves according to previous 
  ;;         evaluation first

  (define (alfa-beta-search board turn prev-move) 
    (let ((best-value -inf.0) (best-move 'pass)
	  (root '()))
      
      ;; Clear transition table
      (set! transitions (make-hash-table 'equal))

      ;;     (display-line "Plotting tree on start of alfa-beta")
      ;;     (plot-tree search-tree 0) 

      (let iterate ((idepth (if (even? max-rdepth) 2 1))
		    (slist (get-successors board turn)))
	(set! root (make-priority-list >))
	(if (> idepth max-rdepth) best-move
	    (begin
	      (gui 'display-ai-action "Iterating depth " idepth
		   (if (> idepth max-depth) " (attacks only)" ""))
	      (set! best-value -inf.0)
	      (set! best-move 'pass)
	      (for-each 
	       (lambda (move)
		 (let ((next-move (fork-move board move)))
		   
		   (let ((value (min-value (fork-move board move) 
					   (next turn) 
					   best-value
					   +inf.0
					   idepth 1 move prev-move)))
		     
		     (gui 'display-ai-action move " " 
			  (round-decimal value 3) " " 
			  (hash-size transitions))

		     (root 'put! value move)
		     
		     (if (> value best-value) 
			 (begin (set! best-value value)
				(set! best-move move))))))

	       ;; Get successors 
	       slist)

	      (iterate (+ idepth 2) (root 'get-values)))))))
  
  ;; max-value :: Chooses the best move for max
  ;;
  ;; uses : * min-value to evaluate opponents response to move
  ;;        * fork-move to make an artificial board to make the move on
  ;;        * terminal-test to recognize the end of the game
  ;;        * utility to evaluate the value of the state for player max.
  ;;
  ;; AI-stuff: * transition-table for redundant states
  ;;           * Using evaluations of recent round when choosing the next
  ;;             node to expand
  ;;           * Quiescent state cutoff
  
  (define (max-value board turn alfa beta depth rd prev-move pprev)
    (let ((node (make-priority-list >)))
      
      (define (each-successor slist v a b use-tr?)

	(define recursive-call
	  (lambda ()
	    (let* ((newboard (fork-move board (car slist)))
		   (min-val (min-value 
			     newboard (next turn) a b 
			     depth (+ rd 1) (car slist) prev-move)))
	      
	      ;; Add move and its value to the node
	      (node 'put! min-val (car slist))
	      
	      ;; Alfa-beta pruning. If rest of the successors are  
	      ;; pruned, add them to the current tree.
	      (let ((u (max v min-val)))
		(if (>= u b)
		    (begin 
		      (node 'append! -inf.0 (cdr slist))
		      ;; Add new state to transition table.
		      ;; If use-tr? is false, it may already exist.
		      (if use-tr?
			  (hash-table-put! transitions (cons board turn)
					   (cons u (cons (- depth rd) node)))
			  (begin
			    (hash-table-remove! transitions (cons board turn))
			    (hash-table-put! transitions (cons board turn)
					     (cons u (cons (- depth rd) 
							   node)))))
		      (list u))
		    (each-successor (cdr slist) u (max a u) b use-tr?))))))
	
        (cond ((null? slist) 
               (hash-table-put! transitions (cons board turn)
                                (cons v (cons (- depth rd) node)))
               (list v))
              (use-tr?
	       (hash-table-get transitions (cons board turn)              
			       recursive-call))
	      (else (recursive-call))))
      
      ;; If terminal test matches or search depth exceeds, get utility
      ;; value from hash or calculate it. Else calculate values for
      ;; each successor.
      ;;(gui 'display-ai-action "max-value " alfa " " beta)
    
      (if (or (terminal-test board turn) 
              (>= rd depth)
	      (and (>= rd max-depth)
		   (quiescent-state? board turn pprev prev-move)))
          (car
           (hash-table-get 
            transitions (cons board turn) 
            (lambda () 
              (let ((util (utility board turn rd)))
                (hash-table-put! transitions (cons board turn) 
				 (cons util (cons (- depth rd) '())))
                (list util)))))
	  (let ((saved-suc (get-saved-successors board turn (- depth rd))))
	    (car (each-successor (cdr saved-suc)
				 -inf.0 alfa beta 
				 (car saved-suc)))))))
  
  ;;
  ;; min-value :: Chooses the best move for min
  ;;
  ;; uses : * max-value to evaluate opponents response to move
  ;;        * fork-move to make an artificial board to make the move on
  ;;        * terminal-test to recognize the end of the game
  ;;        * utility to evaluate the value of the state for player max
  ;;          (min tries to make it as small as possible)
  ;;
  ;; AI-stuff: * transition-table for redundant states
  ;;           * Using evaluations of recent round when choosing the next
  ;;             node to expand (tree)
  ;;           * Quiescent state cutoff 

  (define (min-value board turn alfa beta depth rd prev-move pprev)
    (let ((node (make-priority-list <)))

      (define (each-successor slist v a b use-tr?)

	(define recursive-call
	  (lambda ()
	    (let* ((newboard (fork-move board (car slist)))
		   (max-val (max-value 
			     newboard (next turn) a b 
			     depth (+ rd 1) (car slist) prev-move)))
	      
	      ;; Add move and its value to the node
	      (node 'put! max-val (car slist))
	      
	      ;; Alfa-beta pruning. If rest of the successors are 
	      ;; pruned, add them to the current tree.
	      (let ((u (min v max-val)))
		(if (<= u a)
		    (begin 
		      (node 'append! +inf.0 (cdr slist))
		      ;; Add new state to transition table.
		      ;; If use-tr? is false, it may already exist.
		      (if use-tr?
			  (hash-table-put! transitions (cons board turn)
					   (cons u (cons (- depth rd) node)))
			  (begin
			    (hash-table-remove! transitions (cons board turn))
			    (hash-table-put! transitions (cons board turn)
					     (cons u (cons (- depth rd) 
							   node)))))
		      (list u))
		    
		    (each-successor (cdr slist) u a (min b u) use-tr?))))))
  
        (cond ((null? slist) 
               (hash-table-put! transitions (cons board turn)
                                (cons v (cons (- depth rd) node)))
               (list v))
              (use-tr?
	       (hash-table-get transitions (cons board turn)
			       recursive-call))
	      (else (recursive-call))))

      ;; If terminal test matches or search depth exceeds, get utility
      ;; value from hash or calculate it. Else calculate values for
      ;; each successor.
      ;;(gui 'display-ai-action "min-value " alfa " " beta)

      (if (or (terminal-test board turn) 
              (>= rd depth)
	      (and (>= rd max-depth)
		   (quiescent-state? board turn pprev prev-move)))
          (car
           (hash-table-get 
            transitions (cons board turn) 
            (lambda () 
              (let ((util (utility board (next turn) rd)))
                (hash-table-put! transitions (cons board turn) 
				 (cons util (cons (- depth rd) '())))
                (list util)))))
	  (let ((saved-suc (get-saved-successors board turn (- depth rd))))
	    (car (each-successor (cdr saved-suc)
				 +inf.0 alfa beta
				 (car saved-suc)))))))

  ;; 
  ;; returns ai's next move 
  ;;

  (define (get-move board turn prev-move allowed-moves)
    ;; (gui 'display-ai-action "prev-move: " prev-move)

    ;; Alfa-beta search won't be started at all if only one move is
    ;; available
    (if (= (hash-size allowed-moves) 1)
	(let ((moves (get-successors board turn)))
	  (car moves))
        (let ((move (alfa-beta-search board turn prev-move)))
          move)))


  ;; Dispatch prosedure for message passing
  (define (dispatch m . args)
    (cond ((eq? m 'get-move) (apply get-move args))
	  ((eq? m 'get-player-type) 'ai)
	  ((eq? m 'get-successors) (apply get-successors args))
	  (else (error "Invalid operation for ai-player"))))


  ;; Set optional arguments if given
  (if (not (null? args)) (set! max-depth (car args)))
  (if (> (length args) 1) (set! max-rdepth (cadr args)))	   
  dispatch)
