;;;
;;; Hierarkia - Scheme Implementation
;;; ai.scm: General AI procedures
;;;
;;; Copyright 2003 Sami Virpioja, Ilari Lhteenmki
;;;
;;; ---
;;;
;;; This file is part of Hierarkia - Scheme Implementation.
;;;
;;; Hierarkia - Scheme Implementation is free software; you can redistribute 
;;; it and/or modify it under the terms of the GNU General Public License as 
;;; published by the Free Software Foundation; either version 2 of the 
;;; License, or (at your option) any later version.
;;;
;;; Hierarkia - Scheme Implementation  is distributed in the hope that it will 
;;; be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
;;; Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License along 
;;; with Hierarkia - Scheme Implementation; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;
;;; ---
;;;
;;; See README for more details.
;;;
;;; This is artificial intelligence part of the game. File includes objects 
;;; and routines needed by computer player.
;;;

;; Rounds floating point number f to d decimals

(define (round-decimal f d)
  (let ((b (* f (expt 10 d))))
    (/ (round b) (expt 10 d))))


;; Simple priority list. No deleting or searching, O(N) additions.

(define (make-priority-list cmp)
  (let ((items '()))
    
    (define (put! priority value)
      (define (loop clist)
        (cond ((null? clist) (list (cons priority value)))
              ((cmp (car (car clist)) priority)
               (cons (car clist) (loop (cdr clist))))
              (else (cons (cons priority value) clist))))
      (set! items (loop items)))

    (define (put-many! priority values)
      (define (loop clist)
        (cond ((null? clist) (map (lambda (v) (cons priority v)) values))
              ((cmp (car (car clist)) priority)
               (cons (car clist) (loop (cdr clist))))
              (else (append (map (lambda (v) (cons priority v)) values) 
			    clist))))
      (set! items (loop items)))

    (define (append! priority values)
      (set! items (append items (map (lambda (v) (cons priority v)) values))))
    
    (define (dispatch m . args)
      (cond ((eq? m 'get-items) items)
            ((eq? m 'get-values) (map (lambda (i) (cdr i)) items))
            ((eq? m 'put!) (apply put! args))
            ((eq? m 'put-many!) (apply put-many! args))
            ((eq? m 'append!) (apply append! args))
            (else (error "MAKE-PRIORITY-LIST: unknown message" m))))

    dispatch))


;; Tree structure for saving moves and their utility evaluations.
;; Nodes are added to tree whenever moves are evaluated in min-val and max-val.
;; Nodes are ordered using cmp and tree is used to choose the next move to
;; evaluate at next turn.  

(define (make-node move cmp)
  (let ((age 0) (children '()) (value 'undef))
    
    (define (add-child! node v)
      (define (loop clist)
        (cond ((null? clist) (list node))
              ((cmp ((car clist) 'get-value) (node 'get-value))
               (cons (car clist) (loop (cdr clist))))
              (else (cons node clist))))
      (node 'set-value! v)
      (set! children (loop children)))

    (define (add-children! clist)
      (for-each (lambda (c) (add-child! (car c) (cdr c)))
		clist))

    (define (get-child move)
      (let loop ((clist children))
        (cond ((null? clist) '())  ; (error "NODE: no child found"))
              ((equal? move ((car clist) 'get-move)) (car clist))
              (else (loop (cdr clist))))))

    (define (dispatch m . args)
      (cond ((eq? m 'get-move) move)
            ((eq? m 'get-value) value)
            ((eq? m 'set-value!) (set! value (car args)))
            ((eq? m 'get-age) age)
            ((eq? m 'get-children) children)
            ((eq? m 'get-child) (apply get-child args))
            ((eq? m 'add-age!) (set! age (+ age 1)))
            ((eq? m 'add-child!) (apply add-child! args))
            ((eq? m 'add-children!) (apply add-children! args))
            (else (error "MAKE-NODE: unknown message" m))))

    dispatch))


;; Prints the contents of the tree (for debugging)

(define (plot-tree tree level)
  (if (null? tree) 
      (display-line "End-of-tree reached")   
      (begin
	(display-line level ": " (tree 'get-move) "/ val: " (tree 'get-value))
	(let plot-loop ((childlist (tree 'get-children)))
	  (if (null? childlist)
	      (display-line "No childs")  
	      (begin
		(plot-tree (car childlist) (+ level 1))
		(plot-loop (cdr childlist))))))))


;;
;; AI routines
;;

;; get-successors :: Collection of possible next moves
;;
;; uses allowed-moves, which returns ((unit) (move move move ...)) lists.
;; returns a list like (((unit) (move)) ((unit) (move))) , which has
;; attacks before ordinary moves.

(define (get-successors board turn) 
  (let ((hash-list
         (hash-table-map (get-allowed-moves board turn)
                         (lambda (key value) (list key value))))
        (attacks '()) (moves '()))
    (if (null? hash-list) (list 'pass)
        (let each-unit ((hl hash-list))
          (cond ((null? hl) (append attacks moves))
                ((null? (cadr (car hl))) (each-unit (cdr hl)))
                (else (let ((unit (car (car hl))))
                        (let each-move ((ml (cadr (car hl))))
                          (if (null? ml) (each-unit (cdr hl))
                              (let ((thismove (car ml)))
                                (if (eq? (car thismove) 'move)
                                    (set! moves (cons (list unit thismove) 
                                                      moves))
                                    (set! attacks (cons (list unit thismove) 
                                                        attacks)))
                                (each-move (cdr ml))))))))))))


(define (quiescent-state? board turn prev-move new-move)
  (and (eq? (move-type new-move) 'move)
       (or (not (eq? (move-type prev-move) 'attack))
           (is-block-move? board turn new-move))))

(define (terminal-test board turn) (get-winner board))

(define (utility board turn depth)
  
  ;; Returns unit's firepower
  (define (get-firepower board turn ux uy) 
    (let* ((unit (board-ref board ux uy))
           (side (unit-side unit))
           (moves (get-unit-moves board ux uy turn)))
      (* (if (null? (filter (lambda (m) (eq? 'attack (car m)))
                            moves))
             0.4
             1)            
         (apply + (map (lambda (p) 
                         (get-firepower-modifier 
                          board (car p) (cadr p) side turn))
                       (map (lambda (dir) (map + (list ux uy) dir))
                            dirs))))))
  
  ;; utility function body
  (let ((winner (get-winner board)))
    (if winner (if (eq? winner turn) 
		   (- 1000 depth)
		   (- depth 1000))
        (let loop ((x 0) (y 0) (value 0))
          (cond ((> y (- board-size 1)) value)
                ((> x (- board-size 1)) (loop 0 (+ y 1) value))
                (else 
                 (if (null? (board-ref board x y))
                     (loop (+ x 1) y value)
                     (let ((u (board-ref board x y)))
                       (loop (+ x 1) y
                             (+ value
                                (* (if (eq? (unit-side u) turn) 1 -1)
                                   (+ (* 2 (unit-rank u))
				      (if (and (= (unit-rank u) max-rank)
					       (unit-defends? u))
					  -100 0)
                                      (if (or (in-home? u x y) 
                                              (unit-defends? u)) 
                                          0
                                          (* 0.5 
                                             (get-firepower 
                                              board turn x y)))))))))))))))

;; Loads different AIs according to given type

(define (make-ai-player type . args)
  (cond ((eq? type 'mtd) (apply make-mtd-ai args))
	((eq? type 'pvs) (apply make-pvs-ai args))
	((eq? type 'iter) (apply make-iter-ai args))
	((eq? type 'imtd) (apply make-imtd-ai args))
	((eq? type 'pvs2) (apply make-pvs2-ai args))
	(else (apply make-plain-ai args))))
