/*****************************************************************************
 *
 *                           SerialNumberManager.java
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.protection;

import java.io.*;
import java.awt.*;
import java.util.zip.*;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Enumeration;
import java.net.URL;

/**
 * Class for creating, managing and validating serial numbers and other
 * stuff related to product registration and copyright issues.
 *
 * @author Kary FR&Auml;MLING
 */
public class SerialNumberManager {
    //--------------------------------------------------------------------------------------
    // Public constants.
    //--------------------------------------------------------------------------------------
    public static final String 	STR_REG_VERIFICATION_FAILED = "Unregistered version or evaluation period expired!";
    
    //--------------------------------------------------------------------------------------
    // Protected constants.
    //--------------------------------------------------------------------------------------
    protected static final String 	STR_REG_INFO_FILE_NAME = "RegInfo";
    protected static final String 	STR_REG_INFO_ENTRY_NAME = "fi/faidon/protection/" + STR_REG_INFO_FILE_NAME;
    protected static final String 	STR_TMP_PACKAGE_FILE_NAME = "regpack.tmp";
    protected static final String 	STR_VERSION_ENTRY_NAME = "Version.txt";
    protected final String 	STR_CONFIG_FILE_NAME = "config";
    protected final String 	STR_VERSION_TAG = "Version";
    protected final String 	STR_SERIAL_NO_TAG = "Serial#";
    protected final String 	STR_TAG_VALUE_SEPARATOR = ":";
    protected final String 	STR_COMPANY_CODE = "FADN";
    protected final String 	STR_PRODUCT_CODE = "JVG";
    protected final String 	STR_SER_NO_FIELD_SEPARATOR = "-";
    protected final int		CHECKSUM_STR_LEN = 4;
    protected final int		EVALUATION_VERSION_TYPE = 5;
    protected final int		REGISTERED_VERSION_TYPE = 8;
    protected final int		EVALUATION_DAYS = 30;
    
    //--------------------------------------------------------------------------------------
    // Protected variables.
    //--------------------------------------------------------------------------------------
    protected String	companyCode;
    protected String	productCode;
    protected String	version;
    protected int	year;
    protected int	month;
    protected int	day;
    protected int	hour;
    protected int	minutes;
    protected int	seconds;
    protected int	type;
    protected int	checksum;
    
    //--------------------------------------------------------------------------------------
    // Private variables.
    //--------------------------------------------------------------------------------------
    private static boolean	currentPackageIsVerified;	// This static variable makes it
    // possible to do the registration
    // check only once.
    
    //--------------------------------------------------------------------------------------
    // Public methods.
    //--------------------------------------------------------------------------------------
/*
	public static void main(String[] argv)
	{
		System.out.println((new SerialNumberManager()).verifySerialNumber("FADNJVG1.0-519824-05182351238"));
	}
 */
    //=============================================================================
    // verifyCurrentPackage
    //=============================================================================
    /**
     * We should be able to access a RegInfo entry with a valid serial number
     * in it if there is one in the current package. In fact, it might also
     * be in a separate file.
     */
    //=============================================================================
    public boolean verifyCurrentPackage() {
	// See if we have already verified the registration.
	if ( currentPackageIsVerified ) return true;
	
	URL		file_URL;
	InputStream		is;
	byte[]	byte_buf = new byte[1024];
	int		nbr_bytes;
	String	serial_no;
	
	// Read in background picture. We can fetch it as a system resource since
	// it is in the same place as the application class files.
	file_URL = getClass().getResource(STR_REG_INFO_FILE_NAME);
	if ( file_URL == null ) return false;
	
	// Get an input stream on it.
	try {
	    is = file_URL.openStream();
	    nbr_bytes = is.read(byte_buf);
	    serial_no = new String(byte_buf, 0, nbr_bytes);
	    if ( !verifySerialNumber(serial_no) ) return false;
	}
	catch ( Exception e ) { return false; }
	
	// Everything OK, return true.
	currentPackageIsVerified = true;
	return true;
    }
    
    //=============================================================================
    // getSerialNumber
    //=============================================================================
    /**
     * Create a serial number using the version string. "isEvalVersion" indicates
     * if it should be an evaluation version serial number or a registered
     * one.
     */
    //=============================================================================
    public String getSerialNumber(String versionNo, boolean isEvalVersion) {
	Calendar		current_date = Calendar.getInstance();
	
	// Set version.
	version = versionNo;
	
	// Set all field values to current date & time.
	seconds = current_date.get(Calendar.SECOND);
	year = current_date.get(Calendar.YEAR);
	minutes = current_date.get(Calendar.MINUTE);
	month = current_date.get(Calendar.MONTH);
	hour = current_date.get(Calendar.HOUR_OF_DAY);
	day = current_date.get(Calendar.DAY_OF_MONTH);
	
	// If it is an evaluation version, then we put the value to 5, otherwise
	// to 8. This will make it possible to use an even/uneven coding in the
	// future if needed.
	if ( isEvalVersion ) {
	    type = EVALUATION_VERSION_TYPE;
	}
	else {
	    type = REGISTERED_VERSION_TYPE;
	}
	
	return makeSerialNoFromFields();
    }
    
    //=============================================================================
    // verifySerialNumber
    //=============================================================================
    /**
     * Verify that the serial number is correct. If we see that it is an
     * evaluation version, then we check that the date is still valid.
     */
    //=============================================================================
    public boolean verifySerialNumber(String serialNumber) {
	int			i;
	int			calc_checksum;
	double		tmp_checksum;
	String		tmp;
	
	// Split up the serial number into components.
	if ( !getFieldsFromSerialNo(serialNumber) ) return false;
	
	// See if checksum is correct.
	calc_checksum = calcChecksum();
	tmp = String.valueOf(calc_checksum);
	tmp = tmp.substring(Math.max(0, tmp.length() - CHECKSUM_STR_LEN));
	calc_checksum = new Integer(tmp).intValue();
	if ( calc_checksum != checksum ) return false;
	
	// If it is an evaluation version, then check that the evaluation time
	// hasn't expired.
	if ( type != REGISTERED_VERSION_TYPE ) {
	    Calendar current_date = Calendar.getInstance();
	    GregorianCalendar serial_no_date;
	    serial_no_date = new GregorianCalendar(1900 + year, month, day, hour, minutes, seconds);
	    
	    // Verify that we are not BEFORE the activation time.
	    if ( current_date.before(serial_no_date) ) return false;
	    
	    // Add 30 days to the registration date and verify that we are not AFTER it.
	    serial_no_date.add(Calendar.DATE, EVALUATION_DAYS);
	    if ( current_date.after(serial_no_date) ) return false;
	}
	
	// Serial number OK.
	return true;
    }
    
    //--------------------------------------------------------------------------------------
    // Protected methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // makeSerialNoFromFields
    //=============================================================================
    /**
     * Set up a serial number from the current field values.
     */
    //=============================================================================
    protected String makeSerialNoFromFields() {
	int		i, serial_no_length;
	String	tmp, final_serial_number;
	StringBuffer	serial_no;
	
	// Make a string buffer for storing the serial number.
	serial_no_length = STR_COMPANY_CODE.length() + STR_PRODUCT_CODE.length() +
	version.length() +
	STR_SER_NO_FIELD_SEPARATOR.length() +
	6 +
	STR_SER_NO_FIELD_SEPARATOR.length() +
	6 +
	1 +
	4;
	serial_no = new StringBuffer(serial_no_length);
	
	// Set up the string.
	serial_no.append(STR_COMPANY_CODE);
	serial_no.append(STR_PRODUCT_CODE);
	serial_no.append(version);
	serial_no.append(STR_SER_NO_FIELD_SEPARATOR);
	
	// Set seconds.
	tmp = String.valueOf(seconds); if ( tmp.length() < 2 ) serial_no.append('0');
	serial_no.append(tmp);
	
	// Set year. We limit it to the 2 last.
	tmp = String.valueOf(year); if ( tmp.length() > 3 ) tmp = tmp.substring(2);
	if ( tmp.length() < 2 ) serial_no.append('0');
	year = new Integer(tmp).intValue();
	serial_no.append(tmp);
	
	// Set minutes.
	tmp = String.valueOf(minutes); if ( tmp.length() < 2 ) serial_no.append('0');
	serial_no.append(tmp);
	
	// Separator.
	serial_no.append(STR_SER_NO_FIELD_SEPARATOR);
	
	// Set month.
	tmp = String.valueOf(month); if ( tmp.length() < 2 ) serial_no.append('0');
	serial_no.append(tmp);
	
	// Set hour.
	tmp = String.valueOf(hour); if ( tmp.length() < 2 ) serial_no.append('0');
	serial_no.append(tmp);
	
	// Set day.
	tmp = String.valueOf(day); if ( tmp.length() < 2 ) serial_no.append('0');
	serial_no.append(tmp);
	
	// If it is an evaluation version, then we put the value to 5, otherwise
	// to 8. This will make it possible to use an even/uneven coding in the
	// future if needed.
	tmp = String.valueOf(type);
	serial_no.append(tmp);
	
	// Calculate serial number checksum.
	checksum = calcChecksum();
	tmp = String.valueOf(checksum);
	for ( i = 0 ; i < CHECKSUM_STR_LEN - tmp.length() ; i++ ) serial_no.append('0');
	serial_no.append(tmp.substring(Math.max(0, tmp.length() - CHECKSUM_STR_LEN)));
	
	// Convert StringBuffer to String.
	final_serial_number = new String(serial_no);
	return final_serial_number;
    }
    
    //=============================================================================
    // getFieldsFromSerialNo
    //=============================================================================
    /**
     * Set serial number values from the serial number string. Return false if
     * there was any format problems.
     */
    //=============================================================================
    protected boolean getFieldsFromSerialNo(String serialNumber) {
	int		i, start_index;
	String	stripped_num;
	
	// Go for the first "-" and strip off the base information.
	try {
	    i = serialNumber.indexOf(STR_SER_NO_FIELD_SEPARATOR);
	    stripped_num = serialNumber.substring(i + 1);
	    start_index = 0;
	    companyCode = serialNumber.substring(start_index, STR_COMPANY_CODE.length());
	    start_index += STR_COMPANY_CODE.length();
	    productCode = serialNumber.substring(start_index, start_index + STR_PRODUCT_CODE.length());
	    start_index += STR_PRODUCT_CODE.length();
	    version = serialNumber.substring(start_index, i);
	} catch ( StringIndexOutOfBoundsException e ) { return false; }
	
	// Get all the information fields.
	try {
	    seconds = new Integer(stripped_num.substring(0, 2)).intValue();
	    year = new Integer(stripped_num.substring(2, 4)).intValue();
	    minutes = new Integer(stripped_num.substring(4, 6)).intValue();
	    month = new Integer(stripped_num.substring(7, 9)).intValue();
	    hour = new Integer(stripped_num.substring(9, 11)).intValue();
	    day = new Integer(stripped_num.substring(11, 13)).intValue();
	    type = new Integer(stripped_num.substring(13, 14)).intValue();
	    checksum = new Integer(stripped_num.substring(14, 18)).intValue();
	} catch ( NumberFormatException e ) { return false; }
	return true;
    }
    
    //--------------------------------------------------------------------------------------
    // Private methods.
    //--------------------------------------------------------------------------------------
    
    //=============================================================================
    // calcChecksum
    //=============================================================================
    /**
     * Return the checksum calculated for the current serial number fields.
     */
    //=============================================================================
    private int calcChecksum() {
	return (seconds*4 + minutes*2 + hour*4 + year + month*4 + day + type*3)*679/29;
    }
    
}

