/*****************************************************************************
 *
 *                          VectorImageProducer.java
 *
 * Interface for all classes that can draw vectorized data coming from a
 * picture file, a recorded drawing sequence or something else.
 *
 * Created by Kary FRAMLING 4/4/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jvg;

import java.awt.Image;
import java.awt.Graphics;
import java.awt.Component;
import java.awt.image.ImageObserver;

/**
 * Interface for all classes that can draw vectorized data coming from a
 * picture file, a recorded drawing sequence or something else.
 *
 * @author Kary FR&Auml;MLING
 */
public abstract class VectorImageProducer {
    
    //=============================================================================
    // getWidth
    //=============================================================================
    /**
     * Get the width of the original image. If we are reading image information
     * from a file and the frame size isn't known,  then we read in the
     * information immediately and return it. If we don't have any size information,
     * then we return -1. This is the case for recorded drawing operations,
     * for instance, where we don't know (and don't want to know) anything about
     * the size of the drawing surface used.
     *
     * @author Kary FR&Auml;MLING 4/4/1998
     */
    //=============================================================================
    public abstract int getWidth();
    
    //=============================================================================
    // getHeight
    //=============================================================================
    /**
     * Get the height of the original image. If we are reading image information
     * from a file and the frame size isn't known,  then we read in the
     * information immediately and return it. If we don't have any size information,
     * then we return -1. This is the case for recorded drawing operations,
     * for instance, where we don't know (and don't want to know) anything about
     * the size of the drawing surface used.
     *
     * @author Kary FR&Auml;MLING 4/4/1998
     */
    //=============================================================================
    public abstract int getHeight();
    
    //=============================================================================
    // playIt
    //=============================================================================
    /**
     * Immediately draw the image using the given graphics. This method is
     * especially useful for direct on-screen drawing and for drawing small
     * images (containing few objects).
     *
     * @author Kary FR&Auml;MLING 4/4/1998
     */
    //=============================================================================
    public void playIt(Graphics g) {
	playIt(g, null);
    }
    
    //=============================================================================
    // playIt
    //=============================================================================
    /**
     * Immediately draw the image using the given graphics and the given component.
     * We need a component if AWT images are included in the image for having
     * an ImageObserver.
     *
     * @author Kary FR&Auml;MLING 8/4/1998
     */
    //=============================================================================
    public abstract void playIt(Graphics g, Component c);
    
    //=============================================================================
    // playIt
    //=============================================================================
    /**
     * Do the drawing operations into the graphics of the given image. This is done
     * in a separate thread so that several images can be drawn simultaneously.
     * That is why we need an ImageObserver object that we can call for imageUpdate
     * with ALLBITS once we are ready. The caller should normally take care of not
     * launching several draws into the same image at the same time.
     *
     * @author Kary FR&Auml;MLING 4/4/1998
     */
    //=============================================================================
    public void playIt(Image img, Component caller) {
	// Create a new thread.
	
	// Draw the stuff into the offscreen graphics. This should be done in the
	// new Thread, in fact.
	playIt(img.getGraphics(), caller);
    }
    
    //=============================================================================
    // checkImage
    //=============================================================================
    /**
     * Return the status of the asynchronous draw. This can only be 0 and ALLBITS.
     * For some cases, it may also be WIDTH and HEIGHT. If there are pixmap images
     * then it may be SOMEBITS too.
     *
     * @author Kary FR&Auml;MLING 4/4/1998
     */
    //=============================================================================
    public abstract int checkImage();
    
}



