/*****************************************************************************
 *
 *                            RecorderGraphics.java
 *
 * Class that extends the awt.Graphics class with a recording functionality.
 * It also makes it possible to save the drawing operations into a file,
 * from which they can be re-read in a standardized way.
 * Image passing at least between Java applications through the clipboard
 * also becomes feasible since the comminucating programs just need to
 * implement this class for understanding each other.
 *
 * @author Kary FRAMLING 6/4/1998
 *
 * Copyright 1998-2003 Kary Frmling
 * Source code distributed under GNU LESSER GENERAL PUBLIC LICENSE,
 * included in the LICENSE.txt file in the topmost directory
 *
 *****************************************************************************/

package fi.faidon.jvg;

import java.awt.*;
import java.awt.image.*;
import java.util.*;

/**
 * Class that extends the awt.Graphics class with a recording functionality.
 * It also makes it possible to save the drawing operations into a file,
 * from which they can be re-read in a standardized way.
 * Image passing at least between Java applications through the clipboard
 * also becomes feasible since the comminucating programs just need to
 * implement this class for understanding each other.
 *
 * @author Kary FRAMLING 6/4/1998
 */
public class RecorderGraphics extends Graphics {
    
    //-----------------------------------------------------------------
    // Private constants.
    //-----------------------------------------------------------------
    private final boolean	OBLIGATORY_REGISTRATION = false;
    private final String	DEFAULT_FONT_NAME = "Times";
    private final int	INIT_VECTOR_CAPACITY = 40;
    private final int	DEFAULT_VECTOR_INCREMENT = 20;
    
    //-----------------------------------------------------------------
    // Private fields.
    //-----------------------------------------------------------------
    private Graphics	drawGraphics;
    private boolean		doDrawing;
    private Rectangle	curClipRect;
    private Shape		curClipShape;
    private Font		curFont;
    private Vector		opVector;
    private GraphicsRecording	recorder;
    
    //-----------------------------------------------------------------
    // main
    //-----------------------------------------------------------------
    /**
     * Main function for debugging and test purposes.
     */
    //-----------------------------------------------------------------
    public static void main(String[] argv) {
	System.out.println("Hello world!\n");
    }
    
    //-----------------------------------------------------------------
    // Constructor
    //-----------------------------------------------------------------
    /**
     * Create a recorder graphics with no draw graphics. This simply
     * means that there will be no simultaneous graphics output.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public RecorderGraphics() {
	this(null);
    }
    
    //-----------------------------------------------------------------
    // Constructor
    //-----------------------------------------------------------------
    /**
     * Create a recorder graphics with a draw graphics. This
     * means that there will be simultaneous graphics output and that
     * all "get..."-methods return correct information.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public RecorderGraphics(Graphics g) {
	// Registration check.
	if ( OBLIGATORY_REGISTRATION ) {
	    fi.faidon.protection.SerialNumberManager sm = new fi.faidon.protection.SerialNumberManager();
	    if ( !sm.verifyCurrentPackage() ) {
		System.err.println(fi.faidon.protection.SerialNumberManager.STR_REG_VERIFICATION_FAILED);
		System.err.println("Exiting...");
		System.exit(1);
	    }
	}
	
	// Store the draw graphics. No drawing by default.
	drawGraphics = g;
	doDrawing = false;
	
	// Set up the recording vector.
	recorder = new GraphicsRecording();
	
	// Set up temporary draw object vector.
	opVector = new Vector();
	
	// Set up default values especially for treating the case of no
	// draw graphics.
	curFont = Font.decode(DEFAULT_FONT_NAME);
    }
    
    //-----------------------------------------------------------------
    // getRecording
    //-----------------------------------------------------------------
    /**
     * Returns the graphics recording object.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //-----------------------------------------------------------------
    public GraphicsRecording getRecording() {
	return recorder;
    }
    
    //-----------------------------------------------------------------
    // create {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * If we have a draw graphics, then let it do the job.
     * NULL otherwise.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public Graphics create()	/* ABSTRACT */ {
	return ( drawGraphics == null ) ? null : drawGraphics.create();
    }
    
    //-----------------------------------------------------------------
    // create {OVERRIDE}
    //-----------------------------------------------------------------
    /**
     * Let parent class do the job.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public Graphics create(int x, int y, int width, int height) {
	return super.create(x, y, width, height);
    }
    
    //-----------------------------------------------------------------
    // translate {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Record and pass on if we have a display graphics.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public void translate(int x, int y) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.translate(x, y);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.TRANSLATE));
	opVector.addElement(new Point(x, y));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // getColor {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Return null if no draw graphics, return display graphics color
     * otherwise.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public Color getColor() {
	return ( drawGraphics == null ) ? null : drawGraphics.getColor();
    }
    
    //-----------------------------------------------------------------
    // setColor {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Set foreground color.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //-----------------------------------------------------------------
    public void setColor(Color c) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing ) drawGraphics.setColor(c);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.SET_COLOR));
	opVector.addElement(c);
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // setPaintMode {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Record and pass on if we have a display graphics.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public void setPaintMode() {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing ) drawGraphics.setPaintMode();
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.SET_PAINT_MODE));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // setXORMode {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Record and pass on if we have a display graphics.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public void setXORMode(Color c1) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing ) drawGraphics.setXORMode(c1);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.SET_XOR_MODE));
	opVector.addElement(c1);
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // getFont {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Return null if no draw graphics, return display graphics font
     * otherwise.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public Font getFont()	/* ABSTRACT */ {
	return ( drawGraphics == null ) ? curFont : drawGraphics.getFont();
    }
    
    //-----------------------------------------------------------------
    // setFont {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Set the font used.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //-----------------------------------------------------------------
    public void setFont(Font font) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing ) drawGraphics.setFont(font);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.SET_FONT));
	opVector.addElement(font);
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // getFontMetrics {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Return null if no draw graphics, return display graphics fotn metrics
     * otherwise.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public FontMetrics getFontMetrics() {
	return ( drawGraphics == null ) ? null : drawGraphics.getFontMetrics();
    }
    
    //-----------------------------------------------------------------
    // getFontMetrics {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Return null if no draw graphics, return display graphics fotn metrics
     * otherwise.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public FontMetrics getFontMetrics(Font f)	/* ABSTRACT */ {
	return ( drawGraphics == null ) ? null : drawGraphics.getFontMetrics(f);
    }
    
    //-----------------------------------------------------------------
    // getClipBounds {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Return null if no draw graphics, return display graphics clip bounds
     * otherwise.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public Rectangle getClipBounds()	/* ABSTRACT */ {
	return ( drawGraphics == null ) ? null : drawGraphics.getClipBounds();
    }
    
    //-----------------------------------------------------------------
    // clipRect {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Record and pass on if we have a display graphics.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public void clipRect(int x, int y, int width, int height) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.clipRect(x, y, width, height);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.SET_CLIP));
	opVector.addElement(new Rectangle(x, y, width, height));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // setClip {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Set clip rect.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //-----------------------------------------------------------------
    public void setClip(int x, int y, int width, int height) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.setClip(x, y, width, height);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.SET_CLIP));
	opVector.addElement(new Rectangle(x, y, width, height));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // getClip {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Return the clip shape set previously if no draw graphics,
     * otherwise return display graphics clip shape.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public Shape getClip() {
	return ( drawGraphics == null ) ? curClipShape : drawGraphics.getClip();
    }
    
    //-----------------------------------------------------------------
    // setClip {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Set the clip shape. Also store a local copy.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public void setClip(Shape clip) {
	curClipShape = clip;
	if ( drawGraphics != null ) drawGraphics.setClip(clip);
	
	// Record operation.
	
    }
    
    //-----------------------------------------------------------------
    // copyArea {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Do operation if we have a graphics object, then record.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void copyArea(int x, int y, int width, int height, int dx, int dy) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.copyArea(x, y, width, height, dx, dy);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.COPY_AREA));
	opVector.addElement(new Rectangle(x, y, width, height));
	opVector.addElement(new Point(dx, dy));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawLine {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw a line.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //-----------------------------------------------------------------
    public void drawLine(int x1, int y1, int x2, int y2) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.drawLine(x1, y1, x2, y2);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_LINE));
	opVector.addElement(new Point(x1, y1));
	opVector.addElement(new Point(x2, y2));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // fillRect {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Do operation if we have a graphics object, then record.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void fillRect(int x, int y, int width, int height) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.fillRect(x, y, width, height);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.FILL_RECT));
	opVector.addElement(new Rectangle(x, y, width, height));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawRect {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw a rectangle.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //-----------------------------------------------------------------
    public void drawRect(int x, int y, int width, int height) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.drawRect(x, y, width, height);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_RECT));
	opVector.addElement(new Rectangle(x, y, width, height));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // clearRect {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Do operation if we have a graphics object, then record.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void clearRect(int x, int y, int width, int height) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.clearRect(x, y, width, height);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.CLEAR_RECT));
	opVector.addElement(new Rectangle(x, y, width, height));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawRoundRect {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Do operation if we have a graphics object, then record.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void drawRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.drawRoundRect(x, y, width, height, arcWidth, arcHeight);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_ROUND_RECT));
	opVector.addElement(new Rectangle(x, y, width, height));
	opVector.addElement(new Dimension(arcWidth, arcHeight));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // fillRoundRect {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Do operation if we have a graphics object, then record.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void fillRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.fillRoundRect(x, y, width, height, arcWidth, arcHeight);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.FILL_ROUND_RECT));
	opVector.addElement(new Rectangle(x, y, width, height));
	opVector.addElement(new Dimension(arcWidth, arcHeight));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // draw3DRect {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Do operation if we have a graphics object, then record.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void draw3DRect(int x, int y, int width, int height, boolean raised) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.draw3DRect(x, y, width, height, raised);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_3D_RECT));
	opVector.addElement(new Rectangle(x, y, width, height));
	opVector.addElement(new Boolean(raised));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // fill3DRect {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Do operation if we have a graphics object, then record.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void fill3DRect(int x, int y, int width, int height, boolean raised) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.fill3DRect(x, y, width, height, raised);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.FILL_3D_RECT));
	opVector.addElement(new Rectangle(x, y, width, height));
	opVector.addElement(new Boolean(raised));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawOval {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw an oval.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //-----------------------------------------------------------------
    public void drawOval(int x, int y, int width, int height) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.drawOval(x, y, width, height);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_OVAL));
	opVector.addElement(new Rectangle(x, y, width, height));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // fillOval {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Fill an oval if graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void fillOval(int x, int y, int width, int height) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.fillOval(x, y, width, height);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.FILL_OVAL));
	opVector.addElement(new Rectangle(x, y, width, height));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawArc {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw an arc if graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void drawArc(int x, int y, int width, int height, int startAngle, int arcAngle) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.drawArc(x, y, width, height, startAngle, arcAngle);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_ARC));
	opVector.addElement(new Rectangle(x, y, width, height));
	opVector.addElement(new Point(startAngle, arcAngle));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // fillArc {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Fill an arc if graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void fillArc(int x, int y, int width, int height, int startAngle, int arcAngle) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.fillArc(x, y, width, height, startAngle, arcAngle);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.FILL_ARC));
	opVector.addElement(new Rectangle(x, y, width, height));
	opVector.addElement(new Point(startAngle, arcAngle));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawPolyline {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw a polyline if graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void drawPolyline(int xPoints[], int yPoints[], int nPoints) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.drawPolyline(xPoints, yPoints, nPoints);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_POLYLINE));
	opVector.addElement(new Polygon(xPoints, yPoints, nPoints));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawPolygon {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw a polygon.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void drawPolygon(int xPoints[], int yPoints[], int nPoints) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.drawPolygon(xPoints, yPoints, nPoints);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_POLYGON));
	opVector.addElement(new Polygon(xPoints, yPoints, nPoints));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawPolygon {OVERRIDE}
    //-----------------------------------------------------------------
    /**
     * Draw a polygon.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void drawPolygon(Polygon p) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.drawPolygon(p);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_POLYGON));
	opVector.addElement(p);
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // fillPolygon {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Fill a polygon.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void fillPolygon(int xPoints[], int yPoints[], int nPoints) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.fillPolygon(xPoints, yPoints, nPoints);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.FILL_POLYGON));
	opVector.addElement(new Polygon(xPoints, yPoints, nPoints));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // fillPolygon {OVERRIDE}
    //-----------------------------------------------------------------
    /**
     * Fill a polygon.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void fillPolygon(Polygon p) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null ) drawGraphics.fillPolygon(p);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.FILL_POLYGON));
	opVector.addElement(p);
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawString {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw a string.
     *
     * @author Kary FR&Auml;MLING 15/4/1998
     */
    //-----------------------------------------------------------------
    public void drawString(String str, int x, int y) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing ) drawGraphics.drawString(str, x, y);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_STRING));
	opVector.addElement(str);
	opVector.addElement(new Point(x, y));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawString {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw a string. This is not finished, just to make it
     * compile with JDK 1.3.
     *
     * @author Kary FR&Auml;MLING 13/8/2001
     */
    //-----------------------------------------------------------------
    public void drawString(java.text.AttributedCharacterIterator iterator, int x, int y) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing ) drawGraphics.drawString(iterator, x, y);
    }
    
    //-----------------------------------------------------------------
    // drawChars {OVERRIDE}
    //-----------------------------------------------------------------
    /**
     * Draw chars if graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void drawChars(char data[], int offset, int length, int x, int y) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing ) drawGraphics.drawChars(data, offset, length, x, y);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_STRING));
	opVector.addElement(new String(data, offset, length));
	opVector.addElement(new Point(x, y));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawBytes {OVERRIDE}
    //-----------------------------------------------------------------
    /**
     * Draw bytes if graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void drawBytes(byte data[], int offset, int length, int x, int y) {
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing ) drawGraphics.drawBytes(data, offset, length, x, y);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_STRING));
	opVector.addElement(new String(data, offset, length));
	opVector.addElement(new Point(x, y));
	recorder.addOperation(opVector);
    }
    
    //-----------------------------------------------------------------
    // drawImage {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw image if we have a draw graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 27/4/1998
     */
    //-----------------------------------------------------------------
    public boolean drawImage(Image img, int x, int y, ImageObserver observer) {
	boolean		ret_val = true;
	
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing )
	    ret_val = drawGraphics.drawImage(img, x, y, observer);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_IMAGE_AT_POINT));
	opVector.addElement(img);
	opVector.addElement(new Point(x, y));
	recorder.addOperation(opVector);
	
	return ret_val;
    }
    
    //-----------------------------------------------------------------
    // drawImage {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw image if we have a draw graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 27/4/1998
     */
    //-----------------------------------------------------------------
    public boolean drawImage(Image img, int x, int y, int width, int height, ImageObserver observer)	/* ABSTRACT */ {
	boolean		ret_val = true;
	
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing )
	    ret_val = drawGraphics.drawImage(img, x, y, width, height, observer);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_IMAGE_IN_RECT));
	opVector.addElement(img);
	opVector.addElement(new Rectangle(x, y, width, height));
	recorder.addOperation(opVector);
	
	return ret_val;
    }
    
    //-----------------------------------------------------------------
    // drawImage {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw image if we have a draw graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 27/4/1998
     */
    //-----------------------------------------------------------------
    public boolean drawImage(Image img, int x, int y, Color bgcolor, ImageObserver observer) {
	boolean		ret_val = true;
	
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing )
	    ret_val = drawGraphics.drawImage(img, x, y, bgcolor, observer);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_IMAGE_AT_POINT_WITH_BGCOLOR));
	opVector.addElement(img);
	opVector.addElement(new Point(x, y));
	opVector.addElement(bgcolor);
	recorder.addOperation(opVector);
	
	return ret_val;
    }
    
    //-----------------------------------------------------------------
    // drawImage {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw image if we have a draw graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 27/4/1998
     */
    //-----------------------------------------------------------------
    public boolean drawImage(Image img, int x, int y, int width, int height, Color bgcolor, ImageObserver observer) {
	boolean		ret_val = true;
	
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing )
	    ret_val = drawGraphics.drawImage(img, x, y, width, height, bgcolor, observer);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_IMAGE_IN_RECT_WITH_BGCOLOR));
	opVector.addElement(img);
	opVector.addElement(new Rectangle(x, y, width, height));
	opVector.addElement(bgcolor);
	recorder.addOperation(opVector);
	
	return ret_val;
    }
    
    //-----------------------------------------------------------------
    // drawImage {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw image if we have a draw graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 27/4/1998
     */
    //-----------------------------------------------------------------
    public boolean drawImage(Image img, int dx1, int dy1, int dx2, int dy2,
    int sx1, int sy1, int sx2, int sy2, ImageObserver observer) {
	boolean		ret_val = true;
	
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing )
	    ret_val = drawGraphics.drawImage(img, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, observer);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_SUBIMAGE_IN_RECT));
	opVector.addElement(img);
	opVector.addElement(new Rectangle(sx1, sy1, sx2 - sx1, sy2 - sy1));
	opVector.addElement(new Rectangle(dx1, dy1, dx2 - dx1, dy2 - dy1));
	recorder.addOperation(opVector);
	
	return ret_val;
    }
    
    //-----------------------------------------------------------------
    // drawImage {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Draw image if we have a draw graphics, record operation.
     *
     * @author Kary FR&Auml;MLING 27/4/1998
     */
    //-----------------------------------------------------------------
    public boolean drawImage(Image img, int dx1, int dy1, int dx2, int dy2,
    int sx1, int sy1, int sx2, int sy2, Color bgcolor, ImageObserver observer) {
	boolean		ret_val = true;
	
	// Draw if we have a draw graphics.
	if ( drawGraphics != null && doDrawing )
	    ret_val = drawGraphics.drawImage(img, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, bgcolor, observer);
	
	// Record operation.
	opVector.removeAllElements();
	opVector.addElement(new GraphicsOperation(GraphicsOperation.DRAW_SUBIMAGE_IN_RECT_WITH_BGCOLOR));
	opVector.addElement(img);
	opVector.addElement(new Rectangle(sx1, sy1, sx2 - sx1, sy2 - sy1));
	opVector.addElement(new Rectangle(dx1, dy1, dx2 - dx1, dy2 - dy1));
	opVector.addElement(bgcolor);
	recorder.addOperation(opVector);
	
	return ret_val;
    }
    
    //-----------------------------------------------------------------
    // dispose {IMPLEMENT}
    //-----------------------------------------------------------------
    /**
     * Release all memory resources.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void dispose() {
	if ( drawGraphics != null ) drawGraphics.dispose();
    }
    
    //-----------------------------------------------------------------
    // finalize {OVERRIDE}
    //-----------------------------------------------------------------
    /**
     * Finalize everything.
     *
     * @author Kary FR&Auml;MLING 20/4/1998
     */
    //-----------------------------------------------------------------
    public void finalize() {
	if ( drawGraphics != null ) drawGraphics.finalize();
	super.finalize();
    }
    
    //-----------------------------------------------------------------
    // toString {OVERRIDE}
    //-----------------------------------------------------------------
    /**
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public String toString() {
	if ( drawGraphics == null )
	    return "RecorderGraphics object";
	else
	    return "RecorderGraphics object using display graphics: " + drawGraphics;
    }
    
    //-----------------------------------------------------------------
    // getClipRect {OVERRIDE}
    //-----------------------------------------------------------------
    /**
     * Return the clip rect that has been set by a previous call
     * if we have no draw graphics, otherwise return that of the
     * draw graphics.
     *
     * @author Kary FR&Auml;MLING 6/4/1998
     */
    //-----------------------------------------------------------------
    public Rectangle getClipRect() {
	return ( drawGraphics == null ) ? curClipRect : drawGraphics.getClipRect();
    }
    
}

